#include "bigb_client.h"

#include <travel/hotels/lib/cpp/util/profiletimer.h>

#include <library/cpp/neh/neh.h>
#include <library/cpp/neh/http_common.h>
#include <library/cpp/blockcodecs/codecs.h>
#include <library/cpp/logger/global/global.h>

#include <util/string/builder.h>

#include <contrib/libs/re2/re2/re2.h>

namespace {
    NTravel::TUrl prepareBaseUrl(const NTravelProto::NAppConfig::TBigbClientConfig& bigbClientConfig) {
        NTravel::TUrl baseUrl{};
        baseUrl.SetScheme(NUri::TScheme::SchemeHTTP);
        baseUrl.SetHost(bigbClientConfig.GetHost());
        baseUrl.SetPath(bigbClientConfig.GetPath());
        baseUrl.SetCgiParam("client", bigbClientConfig.GetClient());
        baseUrl.SetCgiParam("format", "protobuf");
        baseUrl.SetCgiParam("compression", "lz4fast");
        return baseUrl;
    }

    const re2::RE2 YandexUidPattern = re2::RE2("^\\d{10}\\d*$");
    const re2::RE2 PassportUidPattern = re2::RE2("^\\d+$");
}

namespace NTravel::NGeoCounter {
    TBigbClient::TBigbClient(const NTravel::NTvm::TTvmService& tvmService, const NTravelProto::NAppConfig::TBigbClientConfig& bigbClientConfig)
        : TvmService_(tvmService)
        , BaseUrl_(prepareBaseUrl(bigbClientConfig))
        , TvmDstAlias_(bigbClientConfig.GetTvmDstAlias())
        , Timeout_(TDuration::MilliSeconds(bigbClientConfig.GetTimeoutMs()))
    {
        NBlockCodecs::SetMaxPossibleDecompressedLength(10 * 1024 * 1024); // 10 mb
    }

    void TBigbClient::RegisterCounters(NMonitor::TCounterSource& source) {
        source.RegisterSource(&Counters_, "BigbClient");
    }

    std::pair<bool, TString> TBigbClient::IsValidRequest(const TRequest& request) const {
        bool hasYandexUid = !request.YandexUid.Empty();
        bool hasPassportUid = request.PassportUid.Defined() && !request.PassportUid.GetRef().Empty();
        if (!hasYandexUid && !hasPassportUid) {
            return {false, "Either yandexUid or passportUid is required"};
        }
        if (hasYandexUid && !re2::RE2::FullMatch(request.YandexUid, YandexUidPattern)) {
            return {false, "YandexUid is invalid " + request.YandexUid};
        }
        if (hasPassportUid && !re2::RE2::FullMatch(request.PassportUid.GetRef(), PassportUidPattern)) {
            return {false, "PassportUid is invalid"};
        }
        return {true, ""};
    }

    yabs::proto::Profile TBigbClient::GetProfile(const TRequest& request) const {
        TString requestUrl;
        auto [requestValid, requestError] = IsValidRequest(request);
        Y_ENSURE(requestValid, requestError);
        try {
            TProfileTimer timer;
            Counters_.NRequests.Inc();

            TUrl bigbUrl = BaseUrl_;
            if (!request.YandexUid.Empty()) {
                bigbUrl.SetCgiParam("bigb-uid", request.YandexUid);
            }
            if (request.PassportUid.Defined() && !request.PassportUid.GetRef().Empty()) {
                bigbUrl.SetCgiParam("puid", request.PassportUid.GetRef());
            }

            requestUrl = bigbUrl.ToString();

            auto msg = NNeh::TMessage::FromString(requestUrl);
            auto headers = TStringBuilder{} << "X-Ya-Service-Ticket: " << TvmService_.GetServiceTicketFor(TvmDstAlias_);
            Y_ENSURE(NNeh::NHttp::MakeFullRequest(msg, headers, "" /*content is empty, request is GET*/), "Failed to make bigb request");

            auto resp = NNeh::Request(msg)->Wait(Timeout_);
            Y_ENSURE(resp, "Failed to get response from bigb");

            if (resp->IsError()) {
                throw yexception() << "Error in bigb response: code=" << resp->GetErrorCode() << " " << resp->GetErrorText();
            }

            auto protoData = NBlockCodecs::Codec("lz4fast")->Decode(resp->Data);

            yabs::proto::Profile message{};
            Y_ENSURE(message.ParseFromString(protoData), "Failed to parse bigb response");

            Counters_.RequestTimeMs.Update(timer.Get().MilliSeconds());

            return message;
        } catch (...) {
            Counters_.NRequestErrors.Inc();
            ERROR_LOG << "Got error while getting profile. Request: " << requestUrl << ", Error: " << CurrentExceptionMessage() << Endl;
            throw;
        }
    }

    TBigbClient::TCounters::TCounters()
        : RequestTimeMs({0, 1, 5, 10, 20, 30, 50, 75, 100, 200})
    {
    }

    void TBigbClient::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
        ct->insert(MAKE_COUNTER_PAIR(NRequestErrors));
        ct->insert(MAKE_COUNTER_PAIR(NRequests));

        RequestTimeMs.QueryCounters("RequestTimeMs", "", ct);
    }
}
