#include "data.h"

#include <travel/hotels/lib/cpp/util/sizes.h>

#include <util/string/builder.h>
#include <util/digest/multi.h>
#include <util/generic/ymath.h>

#include <utility>

namespace {
    const double eps = 1e-6;

    double NormalizeLat(double lat) {
        return Min(Max(lat, -90 + eps), 90 - eps);
    }

    double NormalizeLon(double lon) {
        if (lon > 180 + eps) {
            lon -= (static_cast<int>(lon / 360) + 1) * 360;
        }
        if (lon < -180 - eps) {
            lon += (static_cast<int>(lon / -360) + 1) * 360;
        }
        return Min(Max(lon, -180 + eps), 180 - eps);
    }
}


namespace NTravel::NGeoCounter {
    TPosition::TPosition(double lat, double lon)
        : Lat(NormalizeLat(lat))
        , Lon(NormalizeLon(lon))
    {
    }

    TBoundingBox::TBoundingBox(TPosition lowerLeft, TPosition upperRight)
        : LowerLeft(lowerLeft)
        , UpperRight(upperRight)
    {
    }

    TBoundingBox TBoundingBox::GetExtended(TPosition point) const {
        double leftLon, rightLon;
        if (LowerLeft.Lon <= UpperRight.Lon) {
            leftLon = Min(LowerLeft.Lon, point.Lon);
            rightLon = Max(UpperRight.Lon, point.Lon);
        } else { // inverted bbox, LowerLeft.Lon > 0, UpperRight.Lon < 0
            if (point.Lon < 0) {
                leftLon = LowerLeft.Lon;
                rightLon = Max(UpperRight.Lon, point.Lon);
            } else {
                leftLon = Min(LowerLeft.Lon, point.Lon);
                rightLon = UpperRight.Lon;
            }
        }

        auto downLat = Min(LowerLeft.Lat, point.Lat);
        auto upLat = Max(UpperRight.Lat, point.Lat);

        return TBoundingBox(TPosition(downLat, leftLon), TPosition(upLat, rightLon));
    }

    TBoundingBox TBoundingBox::GetExtendedByAbsoluteValue(double absoluteDiffLon, double absoluteDiffLat) const {
        auto newLeftDown = TPosition(LowerLeft.Lat - absoluteDiffLat, LowerLeft.Lon - absoluteDiffLon);
        auto newUpRight = TPosition(UpperRight.Lat + absoluteDiffLat, UpperRight.Lon + absoluteDiffLon);
        return TBoundingBox(newLeftDown, newUpRight);
    }

    TBoundingBox TBoundingBox::GetExtendedByRelativeValue(double relativeDiff, double minAbsoluteDiff) const {
        double sizeLon = Abs(UpperRight.Lon - LowerLeft.Lon);
        double sizeLat = Abs(UpperRight.Lat - LowerLeft.Lat);

        double deltaLon = Max(minAbsoluteDiff, sizeLon * relativeDiff);
        double deltaLat = Max(minAbsoluteDiff, sizeLat * relativeDiff);

        return GetExtendedByAbsoluteValue(deltaLon, deltaLat);
    }

    TBoundingBox TBoundingBox::GetExtendedByRelativeValue(double relativeDiff) const {
        return GetExtendedByRelativeValue(relativeDiff, 0);
    }

    bool TBoundingBox::Contains(TPosition point) const {
        if (LowerLeft.Lat <= point.Lat && point.Lat <= UpperRight.Lat) {
            if (LowerLeft.Lon <= UpperRight.Lon && LowerLeft.Lon <= point.Lon && point.Lon <= UpperRight.Lon) {
                return true;
            }
            if (LowerLeft.Lon >= UpperRight.Lon && (LowerLeft.Lon <= point.Lon || point.Lon <= UpperRight.Lon)) {
                return true;
            }
        }
        return false;
    }

    TStaticPermalinkInfo::TStaticPermalinkInfo(
        TPermalink permalink,
        TPosition position,
        THashMap<int, TFeature> features,
        TBitMask predefinedFiltersResults)
        : Permalink(permalink)
        , Position(position)
        , Features(std::move(features))
        , PredefinedFiltersResults(predefinedFiltersResults)
    {
    }

    size_t TStaticPermalinkInfo::CalcTotalByteSize() const {
        size_t result = 0;
        result += sizeof(TStaticPermalinkInfo);
        result += GetHashMapByteSizeWithoutElementAllocations(Features) - sizeof(Features);
        for (const auto& [_, values] : Features) {
            result += GetByteSizeWithoutSizeof(values);
        }
        return result;
    }

    TString TStaticPermalinkInfo::ToDebugString(const TStringEncoder& stringEncoder) const {
        TStringBuilder result;
        result << "TStaticPermalinkInfo(";
        result << "Permalink: " << Permalink << ", ";
        result << "Position: " << Position << ", ";
        result << "Features: ";
        for (const auto& [key, value] : Features) {
            result << "(" << stringEncoder.Decode(key) << " [" << key << "]"
                   << ": " << value.ToDebugString(stringEncoder) << ") ";
        }
        result << ")";
        return result;
    }

    bool TPriceRange::IsTrivial() const {
        return MinPrice == MAX_RANGE.MinPrice && MaxPrice == MAX_RANGE.MaxPrice;
    }

    TFeature::TFeature(double value)
        : Value(value) {
    }

    TFeature::TFeature(const TVector<int>& values) {
        if (values.empty()) {
            Value = Empty();
        } else if (values.size() == 1) {
            Value = values[0];
        } else if (values.size() == 2) {
            Value = std::make_pair(values[0], values[1]);
        } else { // zero or many
            Value = MakeIntrusive<Many>(values);
        }
    }

    size_t TFeature::CalcTotalByteSize() const {
        auto res = sizeof(TFeature);
        if (std::holds_alternative<TIntrusivePtr<Many>>(Value)) {
            const auto& many = std::get<TIntrusivePtr<Many>>(Value);
            Y_ENSURE(many);
            res += GetVectorByteSizeWithoutElementAllocations(many->Values);
        }
        return res;
    }

    TString TFeature::ToDebugString(const TStringEncoder& stringEncoder) const {
        struct TToDebugStringVisitor {
            TStringBuilder& Result;
            const TStringEncoder& StringEncoder;

            TToDebugStringVisitor(TStringBuilder& result, const TStringEncoder& stringEncoder)
                : Result(result)
                , StringEncoder(stringEncoder) {
            }

            void operator()(const std::monostate&) const {
                Result << "<undefined>";
            }
            void operator()(const double& data) const {
                Result << data;
            }
            void operator()(const Empty&) const {
                Result << "";
            }
            void operator()(const int& data) const {
                Result << StringEncoder.Decode(data);
            }
            void operator()(const std::pair<int, int>& data) const {
                Result << StringEncoder.Decode(data.first) << ", " << StringEncoder.Decode(data.second);
            }
            void operator()(const TIntrusivePtr<Many>& data) const {
                Y_ENSURE(data);
                const auto& vect = data->Values;
                for (size_t i = 0; i < vect.size(); i++) {
                    Result << StringEncoder.Decode(vect[i]);
                    if (i + 1 < vect.size()) {
                        Result << ", ";
                    }
                }
            }
        };

        TStringBuilder result;
        result << "TFeature(";
        std::visit(TToDebugStringVisitor(result, stringEncoder), Value);
        result << ")";
        return result;
    }

    bool TFeature::HasOneOfValues(const TVector<int>& values) const {
        struct TVisitor {
            const TVector<int>& ExpectedValues;

            TVisitor(const TVector<int>& expectedValues)
                : ExpectedValues(expectedValues) {
            }

            bool operator()(const std::monostate&) const {
                ythrow yexception() << "TFeature::HasOneOfValues can't be applied to empty feature";
            }
            bool operator()(const double&) const {
                ythrow yexception() << "TFeature::HasOneOfValues can't be applied to feature with double value";
            }
            bool operator()(const Empty&) const {
                return false;
            }
            bool operator()(const int& featureValue) const {
                return HasValue(featureValue);
            }
            bool operator()(const std::pair<int, int>& featureValue) const {
                return HasValue(featureValue.first) || HasValue(featureValue.second);
            }
            bool operator()(const TIntrusivePtr<Many>& featureValuePtr) const {
                Y_ENSURE(featureValuePtr);
                const auto& featureValue = featureValuePtr->Values;
                for (const auto& value: featureValue) {
                    if (HasValue(value)) {
                        return true;
                    }
                }
                return false;
            }

            bool HasValue(const int& featureValue) const {
                return std::find(ExpectedValues.begin(), ExpectedValues.end(), featureValue) != ExpectedValues.end();
            }
        };

        return std::visit(TVisitor(values), Value);
    }

    TPriceRange::TPriceRange(ui32 minPrice, ui32 maxPrice)
        : MinPrice(minPrice)
        , MaxPrice(maxPrice)
    {
    }

    const TPriceRange TPriceRange::MAX_RANGE = TPriceRange(std::numeric_limits<ui32>::min(), std::numeric_limits<ui32>::max());

    TBookingRange::TBookingRange(NOrdinalDate::TOrdinalDate checkInDate, NOrdinalDate::TOrdinalDate checkOutDate)
        : CheckInDate(checkInDate)
        , CheckOutDate(checkOutDate)
    {
    }

    TOfferSearchParams::TOfferSearchParams(TBookingRange bookingRange, TAges ages)
        : BookingRange(bookingRange)
        , Ages(ages)
    {
    }

    void TPriceInfo::UpdateDefaultPrice(TPriceRange range) {
        if (DefaultPrice.Empty()) {
            DefaultPrice = range;
        } else {
            auto price = DefaultPrice.Get();
            price->MinPrice = Min(price->MinPrice, range.MinPrice);
            price->MaxPrice = Max(price->MaxPrice, range.MaxPrice);
        }
    }

    void TPriceInfo::UpdateDayPrice(NOrdinalDate::TOrdinalDate date, TPriceRange range) {
        UpdateDefaultPrice(range);
        if (!PerDayPrices.contains(date)) {
            PerDayPrices.emplace(date, range);
        } else {
            auto& price = PerDayPrices.at(date);
            price.MinPrice = Min(price.MinPrice, range.MinPrice);
            price.MaxPrice = Max(price.MaxPrice, range.MaxPrice);
        }
    }

    bool TPriceInfo::HasData() const {
        return !DefaultPrice.Empty();
    }

    TPriceRange TPriceInfo::GetPrice(NOrdinalDate::TOrdinalDate date) const {
        if (PerDayPrices.contains(date)) {
            return PerDayPrices.at(date);
        }
        if (DefaultPrice.Empty()) {
            ythrow yexception() << "Invalid price info: empty default price";
        }
        return *DefaultPrice.Get();
    }

    TPriceRange TPriceInfo::GetPrice(TBookingRange bookingRange) const {
        ui32 totalMinPrice = 0;
        ui32 totalMaxPrice = 0;
        for (auto date = bookingRange.CheckInDate; date < bookingRange.CheckOutDate; date++) {
            auto range = GetPrice(date);
            totalMinPrice += range.MinPrice;
            totalMaxPrice += range.MaxPrice;
        }
        return TPriceRange(totalMinPrice, totalMaxPrice);
    }

    size_t TPriceInfo::CalcTotalByteSize() const {
        size_t result = sizeof(TPriceInfo);
        result += GetHashMapByteSizeWithoutElementAllocations(PerDayPrices);
        return result;
    }

    size_t THotelTraits::CalcTotalByteSize() const {
        size_t result = sizeof(THotelTraits);
        if (PansionAliases.Defined()) {
            result += GetVectorByteSizeWithoutElementAllocations(PansionAliases.GetRef()) - sizeof(TVector<int>);
        }
        return result;
    }

    bool TOfferBusDataKey::operator==(const TOfferBusDataKey& rhs) const {
        return BookingRange.CheckInDate == rhs.BookingRange.CheckInDate &&
               BookingRange.CheckOutDate == rhs.BookingRange.CheckOutDate &&
               Occupancy == rhs.Occupancy;
    }

    TOfferBusDataKey::TOfferBusDataKey(TBookingRange bookingRange, const TString& occupancy)
        : BookingRange(bookingRange)
        , Occupancy(occupancy)
    {
    }

    size_t TOfferBusDataKey::CalcTotalByteSize() const {
        return sizeof(this) + GetByteSizeWithoutSizeof(Occupancy);
    }

    size_t TOfferBusDataKey::Hash() const {
        return MultiHash(BookingRange.CheckInDate, BookingRange.CheckOutDate, Occupancy);
    }

    size_t TOfferBusData::TCacheItem::CalcTotalByteSize() const {
        return sizeof(TOfferBusData::TCacheItem) + GetVectorByteSizeWithoutElementAllocations(Offers) - sizeof(Offers);
    }

    size_t TOfferBusData::SlowCalcTotalByteSize() const {
        size_t result = sizeof(TOfferBusData) + GetHashMapByteSizeWithoutElementAllocations(Offers) - sizeof(decltype(Offers));
        for (const auto& [k, v] : Offers) {
            result += GetByteSizeWithoutSizeof(k);
            result += GetHashMapByteSizeWithoutElementAllocations(v) - sizeof(v);
            for (const auto& [_, innerValue] : v) {
                result += GetByteSizeWithoutSizeof(innerValue);
            }
        }
        return result;
    }

    TRealTimeRankingInfo::TRealTimeRankingInfo()
        : CatBoostUsed(false)
        , CatBoostPrediction(0)
    {
    }

    size_t TFilteringPermalinkInfo::CalcTotalByteSize() const {
        return sizeof(TFilteringPermalinkInfo)
            + GetByteSizeWithoutSizeof(StaticPermalinkInfo)
            + GetByteSizeWithoutSizeof(PriceInfo)
            + (OfferBusData.Defined() ? OfferBusData->SlowCalcTotalByteSize() - sizeof(TOfferBusData) : 0);
    }

    THotelsResults::THotelFeature::THotelFeature(const TString& id, const TString& name, bool value)
        : Id(id)
        , Name(name)
        , BooleanValue(value) {
    }

    THotelsResults::THotelFeature::THotelFeature(const TString& id, const TString& name, double value)
        : Id(id)
        , Name(name)
        , DoubleValue(value) {
    }

    THotelsResults::THotelFeature::THotelFeature(const TString& id, const TString& name, const TVector<TStringFeatureValue>& values)
        : Id(id)
        , Name(name)
        , StringValues(values) {
    }


    THotelsResults::THotelResult::THotelResult()
        : Coordinates(0, 0) {
    }

    bool TCryptaSegments::IsEmpty() const {
        return SegmentValues.empty() && WeightedSegmentValues.empty();
    }

    TUserSegmentsRegistry::TKeyword::TKeyword() {
    }

    TUserSegmentsRegistry::TKeyword::TKeyword(int keywordId, const TString& keywordName, const TVector<TSegment>& possibleSegments, EKeywordType keywordType)
        : KeywordId(keywordId)
        , KeywordName(keywordName)
        , PossibleSegments(possibleSegments)
        , KeywordType(keywordType) {
    }

    TMaybe<TUserSegmentsRegistry::TKeyword> TUserSegmentsRegistry::GetKeywordById(int keywordId) const {
        auto keywordIt = Keywords.find(keywordId);
        if (keywordIt == Keywords.end()) {
            return {};
        }
        return keywordIt->second;
    }

    TUserSegmentsRegistry::TKeyword TUserSegmentsRegistry::GetKeywordByIdOrFail(int keywordId) const {
        auto result = GetKeywordById(keywordId);
        Y_ENSURE(result.Defined(), "Keyword not found by id " + ToString(keywordId));
        return result.GetRef();
    }

    TMaybe<TUserSegmentsRegistry::TKeywordAndSegment> TUserSegmentsRegistry::GetSegmentById(int keywordId, int segmentId) const {
        auto keywordIt = Keywords.find(keywordId);
        if (keywordIt == Keywords.end()) {
            return {};
        }
        for (const auto& segment: keywordIt->second.PossibleSegments) {
            if (segment.SegmentId == segmentId) {
                return TKeywordAndSegment{keywordIt->second, segment};
            }
        }
        return {};
    }

    TUserSegmentsRegistry::TKeywordAndSegment TUserSegmentsRegistry::GetSegmentByIdOrFail(int keywordId, int segmentId) const {
        auto result = GetSegmentById(keywordId, segmentId);
        Y_ENSURE(result.Defined(), "Keyword and segment not found by id " + ToString(keywordId) + " "  + ToString(segmentId));
        return result.GetRef();
    }

    TMaybe<TUserSegmentsRegistry::TKeywordAndSegment> TUserSegmentsRegistry::GetSegmentByName(const TString& keywordName, const TString& segmentName) const {
        auto keywordIt = KeywordsByName.find(keywordName);
        if (keywordIt == KeywordsByName.end()) {
            return {};
        }
        for (const auto& segment: keywordIt->second.PossibleSegments) {
            if (segment.SegmentName == segmentName) {
                return TKeywordAndSegment{keywordIt->second, segment};
            }
        }
        return {};
    }

    TUserSegmentsRegistry::TKeywordAndSegment TUserSegmentsRegistry::GetSegmentByNameOrFail(const TString& keywordName, const TString& segmentName) const {
        auto result = GetSegmentByName(keywordName, segmentName);
        Y_ENSURE(result.Defined(), "Keyword and segment not found by names " + ToString(keywordName) + " "  + ToString(segmentName));
        return result.GetRef();
    }

    TUserSegmentsRegistry TUserSegmentsRegistry::BuildRegistry() {
        auto result = TUserSegmentsRegistry();
        result.RegisterKeyword(TKeyword(174, "gender", {
            {0, "m"},
            {1, "f"}
        }, TUserSegmentsRegistry::EKeywordType::KT_WEIGHTED_UINT_WITH_SELECTED_VALUES));
        result.RegisterKeyword(TKeyword(176, "", {}, TUserSegmentsRegistry::EKeywordType::KT_IGNORED));
        result.RegisterKeyword(TKeyword(543, "user_age_6s", {
            {0, "0_17"},
            {1, "18_24"},
            {2, "25_34"},
            {3, "35_44"},
            {4, "45_54"},
            {5, "55_99"}
        }, TUserSegmentsRegistry::EKeywordType::KT_WEIGHTED_UINT_WITH_SELECTED_VALUES));
        result.RegisterKeyword(TKeyword(614, "income_5_segments", {
            {0, "A"},
            {1, "B1"},
            {2, "B2"},
            {3, "C1"},
            {4, "C2"}
        }, TUserSegmentsRegistry::EKeywordType::KT_WEIGHTED_UINT_WITH_SELECTED_VALUES));
        result.RegisterKeyword(TKeyword(546, "probabilistic_segments", {
            {2465, "segment-family_status_married"},                 // Люди, которые, судя по поведению в интернете, состоят в браке.
            {2466, "segment-family_status_not_married"},             // Люди, которые, судя по поведению в интернете, не состоят в браке.
        }, TUserSegmentsRegistry::EKeywordType::KT_WEIGHTED_UINT_VALUES));
        result.RegisterKeyword(TKeyword(547, "heuristic_segments", {
            {1023, "segment-family_status_married"},                 // Люди, которые, судя по поведению в интернете, состоят в браке.
            {1024, "segment-family_status_not_married"},             // Люди, которые, судя по поведению в интернете, не состоят в браке.
            {1192, "segment-20f8a577"},                              // Люди, которые совершают частые поездки заграницу. Определяется по смене IP, гео из приложений и др. Учитываются поездки за последний год.
            {1193, "segment-0ca1964d"},                              // Люди, которые совершают частые поездки по стране. Определяется по смене IP, гео из приложений и др. Учитываются поездки за последний год.
            {1043, "segment-bd2f8563"},                              // Люди, у которых определился сотовый оператор и которые в более 70% случаев выходили в интернет через мобильную сеть. Сотовый оператор определяется по номеру привязанного мобильного телефона или по ip через IPGSM. Тип подключения определяется по connection_hist из dev_info_yt.
            {1045, "segment-647009ee"},                              // Люди, которые в более чем 70% случаях пользовались WiFi за последние 30 дней. Тип подключения определяется по connection_hist из dev_info_yt.
            {1046, "segment-a2ca4eec"},                              // Люди, которые редко заходят на сайты со смартфона и проводят там мало времени. К ним относится треть наименее активных пользователей.
            {1047, "segment-e9402061"},                              // Люди, которые часто заходят на сайты со смартфона и проводят там много времени. К ним относится треть наиболее активных пользователей.
            {1048, "segment-07804b5b"},                              // Треть самых неактивных пользователей интернета по среднему количеству хитов в день. Попадают только люди, которые были активны хотя бы три дня за последний месяц.
            {1049, "segment-77272017"},                              // Треть самых активных пользователей интернета по среднему количеству хитов в день. Попадают только люди, которые были активны хотя бы три дня за последний месяц.
            {1050, "segment-dff46ae2"},                              // Люди, которые совершают покупки в интернет-магазинах
            {2861, "segment-960a3cca"},                              // Пользователи, которые тратят деньги на внутренние покупки в приложениях
            {1055, "segment-e45e2096"},                              // Люди, у которых есть планшет на базе Android.
            {1056, "segment-bfe1ae3a"},                              // Люди, у которых есть планшет на базе iOS.
            {1057, "segment-86d40600"},                              // Люди, у которых есть планшет на базе Windows.
            {1058, "segment-c2c9fa67"},                              // Люди, у которых есть смартфон на базе Android.
            {1059, "segment-c65e8ebf"},                              // Люди, у которых есть смартфон на базе iOS.
            {1060, "segment-de5ae29b"},                              // Люди, у которых есть смартфон на базе Windows.
            {1061, "segment-5a0e35a3"},                              // Люди, у которых в домохозяйстве есть Smart TV.
            {1062, "segment-5c3ec7d2"},                              // Люди, у которых есть смартфон стоимостью до 15 тыс. руб.
            {1063, "segment-b1ca29ad"},                              // Люди, у которых есть смартфон за 15–25 тыс. руб.
            {1064, "segment-a9b0483c"},                              // Люди, у которых есть смартфон за 25–35 тыс. руб.
            {1065, "segment-b65d65a0"},                              // Люди, у которых есть смартфон за 35–50 тыс. руб.
            {1066, "segment-2d89bcc9"},                              // Люди, у которых есть смартфон стоимостью от 50 до 100 тыс. руб.
            {2044, "segment-bc2c68cd"},                              // Пользователи смартфонов от 100 тыс. руб.
            {2677, "segment-e10ec8d5"},                              // Пользователи (cryptaid), у которых в есть Smart TV.
            {1026, "segment-family_has_infants"},                    // Люди, у которых, судя по поведению в интернете, есть дети до одного года.
            {1027, "segment-family_has_children_1_3"},               // Люди, у которых, судя по поведению в интернете, есть дети от 1 до 3 лет.
            {1028, "segment-family_has_children_3_6"},               // Люди, у которых, судя по поведению в интернете, есть дети от 3 до 6 лет.
            {1029, "segment-family_has_children_6_11"},              // Люди, у которых, судя по поведению в интернете, есть дети от 6 до 11 лет.
            {1030, "segment-family_has_children_11_16"},             // Люди, у которых, судя по поведению в интернете, есть дети от 11 до 16 лет.
            {1918, "segment-d021966a"},                              // Люди, у которых, судя по поведению в интернете, есть дети, возраст которых не определен
            {1919, "segment-132dc2d0"},                              // Люди, у которых есть дети, т.е. те люди, у которых есть сегмент Крипты «Есть дети до 1 года»,  «Есть дети 1–3 лет», «Есть дети 3–6 лет», «Есть дети 6–11 лет» или «Есть дети 11–16 лет».
            {1105, "segment-visited-hotels"},                        // Люди, которые посещают отели и гостиницы
            {1106, "segment-visited-resorts"},                       // Люди, которые посещают санатории и дома отдыха
            {1107, "segment-visited-hostels"},                       // Люди, которые посещают хостелы
        }, TUserSegmentsRegistry::EKeywordType::KT_UINT_VALUES));
        TVector<TSegment> interests = {
            {39, "segment-holydays_in_russia"},                      // Люди, которые интересуются турами, экскурсиями, достопримечательностями в России и странах СНГ.
            {73, "segment-hiking_holidays"},                         // Люди, которые интересуются турклубами, спортивным туризмом, пешими и горными походами.
            {74, "segment-car_rental"},                              // Люди, которые интересуются арендой автомобилей за рубежом.
            {103, "segment-holidays_in_europe"},                     // Люди, которые интересуются турами, экскурсиями, достопримечательностями в странах Европы.
            {113, "segment-childrens_holidays"},                     // Люди, которые интересуются детскими лагерями и экскурсиями.
            {123, "segment-holidays_in_australia_and_oceania"},      // Люди, которые интересуются турами, экскурсиями, достопримечательностями в Австралии и странах Океании.
            {129, "segment-ski_holidays"},                           // Люди, которые интересуются горнолыжными курортами и турами.
            {135, "segment-travel_insurance"},                       // Люди, которые интересуются оформлением, информацией и ценами туристических страховых полисов.
            {142, "segment-flight_tickets"},                         // Люди, которые интересуются сравнением цен на авиабилеты, бронированием и покупкой авиабилетов.
            {143, "segment-wedding_holidays"},                       // Люди, которые интересуются свадебными путешествиями и турами.
            {144, "segment-holidays_in_africa"},                     // Люди, которые интересуются турами, экскурсиями, достопримечательностями в странах Африки.
            {151, "segment-holidays_in_asia"},                       // Люди, которые интересуются турами, экскурсиями, достопримечательностями в странах Азии.
            {174, "segment-holidays_in_america"},                    // Люди, которые интересуются турами, экскурсиями, достопримечательностями в странах Северной и Южной Америки.
            {178, "segment-cruises"},                                // Люди, которые интересуются речными и морскими круизами.
            {201, "segment-intercity_buses"},                        // Люди, которые интересуются расписанием и маршрутами движения транспорта.
            {217, "segment-train_tickets"},                          // Люди, которые интересуются железнодорожными билетами, их стоимостью, бронированием и покупкой.
            {219, "segment-hotels"},                                 // Люди, которые интересуются бронированием отелей и гостиниц, читают отзывы и информацию о них.
            {342, "segment-d9bb10ab"},                               // Интересуются арендой загородной недвижимости (по каталогии)
            {344, "segment-b8e63679"},                               // Интересуются долгосрочной арендой квартир и комнат
            {346, "segment-ab29c54e"},                               // Интересуются арендой помещений для мероприятий (по каталогии)
            {348, "segment-43d045a9"},                               // Интересуются коммерческой арендой (по Каталогии)
            {350, "segment-87ba5032"},                               // Интересуются посуточной арендой загородной недвижимости (по Каталогии)
            {352, "segment-c1a52a1d"},                               // Интересуются посуточной или почасовой арендой квартир (по Каталогии)
        };
        result.RegisterKeyword(TKeyword(601, "longterm_interests", interests, TUserSegmentsRegistry::EKeywordType::KT_UINT_VALUES));
        result.RegisterKeyword(TKeyword(602, "shortterm_interests", interests, TUserSegmentsRegistry::EKeywordType::KT_UINT_VALUES));
        result.RegisterKeyword(TKeyword(557, "audience_segments", {
            {2022695931, "segment-a9c0d70e"},                       // Плюсовики на Яндекс Путешествиях - без брони отелей
            {2022695928, "segment-a9c0d70e"},                       // Плюсовики на Яндекс Путешествиях c бронью отеля
            {2022516900, "segment-9a0e3b30"},                       // Пользователи, которые недавно делали поиск отеля на Я.Путешествиях
            {2022516848, "segment-9a0e3b30"},                       // Посещение сайтов про путешествия - на СЕРПе у пользователя были отельные сайты
            {2022516537, "segment-9a0e3b30"},                       // Посещение сайтов про путешествия - был запрос на СЕРПе с ключами про отели
            {2022516495, "segment-9a0e3b30"},                       // Посещение сайтов про путешествия - букинг и агода
            {2022516470, "segment-9a0e3b30"},                       // Посещение сайтов про путешествия - российские сайты бронирования отелей
            {2022474687, "segment-9a0e3b30"},                       // Посещение сайтов про путешествия - островок
            {2021722561, "segment-c1e7907a"},                       // Посещали через СЕРП сайты турагенств
            {2021722554, "segment-c1e7907a"},                       // Визит на сайт турагенств
            {2021722546, "segment-c1e7907a"},                       // Ключевики про туры на СЕРПе
            {2021722538, "segment-c1e7907a"},                       // Искали туры через СЕРП в последние 30 дней
            {2021722527, "segment-c1e7907a"},                       // Искали через СЕРП туры 14 дней
        }, TUserSegmentsRegistry::EKeywordType::KT_UINT_VALUES));
        return result;
    }

    void TUserSegmentsRegistry::RegisterKeyword(const TKeyword& keyword) {
        Y_ENSURE(!Keywords.contains(keyword.KeywordId), "Duplicate keyword_id: " + ToString(keyword.KeywordId)) ;
        Keywords[keyword.KeywordId] = keyword;
        KeywordsByName[keyword.KeywordName] = keyword;
    }

    int TUserSegmentsRegistry::GenderKeywordId = 174;
    int TUserSegmentsRegistry::AgeKeywordId = 543;
    int TUserSegmentsRegistry::IncomeKeywordId = 614;
}

template <>
void Out<NTravel::NGeoCounter::TPosition>(IOutputStream& out, const NTravel::NGeoCounter::TPosition& position) {
    out << "(lat: " << position.Lat << "; lon: " << position.Lon << ")";
}

template <>
void Out<NTravel::NGeoCounter::TBoundingBox>(IOutputStream& out, const NTravel::NGeoCounter::TBoundingBox& boundingBox) {
    out << "Bbox(" << boundingBox.LowerLeft << "; " << boundingBox.UpperRight << ")";
}

template <>
void Out<NTravel::NGeoCounter::TBookingRange>(IOutputStream& out, const NTravel::NGeoCounter::TBookingRange& bookingRange) {
    out << "BookingRange(" << NTravel::NOrdinalDate::ToString(bookingRange.CheckInDate) << "; " << NTravel::NOrdinalDate::ToString(bookingRange.CheckOutDate) << ")";
}

template <>
void Out<NTravel::NGeoCounter::TOfferBusDataKey>(IOutputStream& out, const NTravel::NGeoCounter::TOfferBusDataKey& offerBusDataKey) {
    out << "OfferBusDataKey(" << offerBusDataKey.BookingRange << "; " << offerBusDataKey.Occupancy << ")";
}
