#pragma once

#include "string_encoder.h"

#include <travel/hotels/geocounter/proto/geocounter_data.pb.h>
#include <travel/hotels/proto/geocounter_service/geocounter_service.pb.h>
#include <travel/hotels/lib/cpp/data/data.h>
#include <travel/hotels/lib/cpp/util/string_compressor.h>
#include <travel/hotels/proto/offercache_api/commons.pb.h>

#include <bitset>
#include <utility>

namespace NTravel::NGeoCounter {
    constexpr static size_t g_SortTypeCount = 6 + 16 + 5 + 2; // 5 usual + 1 exp + 16 crypta segments + 5 exploration + 2 catboost
    typedef std::array<int, g_SortTypeCount> TSortIndices;

    typedef std::bitset<64> TBitMask;

    typedef i64 TTravelGeoId;

    struct TPosition {
        TPosition(double lat, double lon);

        double Lat;
        double Lon;
    };

    struct TBoundingBox {
        TBoundingBox(TPosition lowerLeft, TPosition upperRight);

        TBoundingBox GetExtended(TPosition point) const;
        TBoundingBox GetExtendedByAbsoluteValue(double absoluteDiffLon, double absoluteDiffLat) const;
        TBoundingBox GetExtendedByRelativeValue(double relativeDiff, double minAbsoluteDiff) const;
        TBoundingBox GetExtendedByRelativeValue(double relativeDiff) const;
        bool Contains(TPosition point) const;

        TPosition LowerLeft;
        TPosition UpperRight;
    };

    struct TFeature {
        struct Empty {
        };

        struct Many : public TThrRefBase {
            explicit Many(const TVector<int>& values)
                : Values(values) {
            }

            TVector<int> Values;
        };

        TFeature() = default;
        explicit TFeature(double value);
        explicit TFeature(const TVector<int>& values);

        size_t CalcTotalByteSize() const;
        TString ToDebugString(const TStringEncoder& stringEncoder) const;
        bool HasOneOfValues(const TVector<int>& values) const;

        std::variant<std::monostate, double, Empty, int, std::pair<int, int>, TIntrusivePtr<Many>> Value; // Double numeric value or 0|1|2|many encoded string values
    };

    static_assert(sizeof(TFeature) == 16);

    struct TExtraPermalinkInfo {
        TCompressedString Name;
        TCompressedString Address;
        int ImageCount;
        int GeoId;
        int DisplayedLocationGeoId;
        THashMap<int, float> RealtimeRankingFloatFeatures;
        THashMap<int, ui32> RealtimeRankingCatFeaturesCatBoostEncoded;
    };

    struct TStaticPermalinkInfo {
        TStaticPermalinkInfo(
            TPermalink permalink,
            TPosition position,
            THashMap<int, TFeature> features,
            TBitMask predefinedFiltersResults);

        size_t CalcTotalByteSize() const;

        TString ToDebugString(const TStringEncoder& stringEncoder) const;

        TPermalink Permalink;
        TPosition Position;
        THashMap<int, TFeature> Features;
        TBitMask PredefinedFiltersResults;
    };

    struct TPriceRange {
        TPriceRange(ui32 minPrice, ui32 maxPrice);

        ui32 MinPrice;
        ui32 MaxPrice;

        bool IsTrivial() const;

        const static TPriceRange MAX_RANGE;
    };

    struct TBookingRange {
        TBookingRange(NOrdinalDate::TOrdinalDate checkInDate, NOrdinalDate::TOrdinalDate checkOutDate);

        NOrdinalDate::TOrdinalDate CheckInDate;
        NOrdinalDate::TOrdinalDate CheckOutDate;
    };

    struct TOfferSearchParams {
        TOfferSearchParams(TBookingRange bookingRange, TAges ages);

        TBookingRange BookingRange;
        TAges Ages;
    };

    struct THotelAltayData {
        TCompressedString Name;
        TCompressedString Address;
        int PhotoCount;
        int GeoId;
        int DisplayedLocationGeoId;
        TPosition Position;
        THashMap<int, TFeature> Features;
        THashMap<int, double> RankingFactors;
        THashMap<int, float> RealtimeRankingFloatFeatures;
        THashMap<int, ui32> RealtimeRankingCatFeaturesCatBoostEncoded;
    };

    struct TPriceInfo {
        void UpdateDefaultPrice(TPriceRange range);
        void UpdateDayPrice(NOrdinalDate::TOrdinalDate date, TPriceRange range);
        bool HasData() const;
        TPriceRange GetPrice(NOrdinalDate::TOrdinalDate date) const;
        TPriceRange GetPrice(TBookingRange bookingRange) const;

        size_t CalcTotalByteSize() const;

        TMaybe<TPriceRange> DefaultPrice;
        THashMap<NOrdinalDate::TOrdinalDate, TPriceRange> PerDayPrices;
    };

    struct THotelSortData {
        TPermalink Permalink;
        const THotelAltayData* HotelAltayData;
        const TPriceInfo* PriceInfo;
    };

    struct THotelTraits {
        TMaybe<bool> BreakfastIncluded;
        TMaybe<bool> FreeCancellation;
        TMaybe<TVector<int>> PansionAliases;

        size_t CalcTotalByteSize() const;
    };

    struct TOfferBusDataKey {
        explicit TOfferBusDataKey(TBookingRange bookingRange, const TString& occupancy);

        TBookingRange BookingRange;
        TString Occupancy;

        size_t CalcTotalByteSize() const;

        bool operator==(const TOfferBusDataKey& rhs) const;
        size_t Hash() const;
    };

    struct TRealTimeRankingInfo {
        TRealTimeRankingInfo();

        bool CatBoostUsed;
        float CatBoostPrediction;
    };

    struct THotelsResults {
        struct TStringFeatureValue {
            TStringFeatureValue(const TString& id, const TString& name)
                : Id(id)
                , Name(name) {
            }
            TString Id;
            TString Name;
        };

        struct THotelFeature {
            THotelFeature(const TString& id, const TString& name, bool value);
            THotelFeature(const TString& id, const TString& name, double value);
            THotelFeature(const TString& id, const TString& name, const TVector<TStringFeatureValue>& values);

            TString Id;
            TString Name;
            TMaybe<bool> BooleanValue;
            TMaybe<double> DoubleValue;
            TMaybe<TVector<TStringFeatureValue>> StringValues;
        };

        struct TCancellationInfo {
            bool HasFreeCancellation;
            NTravelProto::ERefundType RefundType;
            TVector<NTravelProto::NOfferCache::NApi::TRefundRule> RefundRules;
        };

        struct TPansionInfo {
            NTravelProto::EPansionType PansionType;
            TString PansionName;
        };

        struct TOfferInfo {
            TPriceVal Price;
            TPansionInfo PansionInfo;
            TCancellationInfo CancellationInfo;
            TMaybe<NTravelProto::NOfferCache::NApi::TStrikethroughPrice> StrikethroughPrice;
            TMaybe<NTravelProto::NOfferCache::NApi::TYandexPlusInfo> YandexPlusInfo;
        };

        struct THotelBadge {
            TString Id;
            TString Text;
            TString Style;
            TMaybe<NTravelProto::NOfferCache::NApi::TBadge::TAdditionalInfo> AdditionalInfo;
            NTravelProto::NOfferCache::NApi::EBadgeTheme Theme;
        };

        struct TRubric {
            i64 Id;
            TString Name;
        };

        struct THotelResult {
            THotelResult();

            TPermalink Permalink;
            TString Name;
            TRubric Rubric;
            TPosition Coordinates;
            TString Address;
            TMaybe<int> Stars;
            TMaybe<double> Rating;
            int ImageCount;
            TVector<THotelFeature> Features;
            TMaybe<TOfferInfo> OfferInfo;
            TVector<THotelBadge> Badges;
            bool IsPollingFinished;
            bool IsPlusAvailable;
            bool IsTopHotel;
            int DisplayedLocationGeoId;
            bool HasBoyOffers;

            TMaybe<TExtraPermalinkInfo> ExtraInfo;
            TRealTimeRankingInfo RealTimeRankingInfo;
        };

        TVector<THotelResult> Hotels;
        bool IsPollingFinished;
        bool HotelLocationUseful;
    };

    struct TCryptaSegments {
        THashSet<std::pair<int, int>> SegmentValues; // (keyword_id, segment_id)
        THashMap<std::pair<int, int>, int> WeightedSegmentValues; // (keyword_id, segment_id) -> weight

        bool IsEmpty() const;
    };
}

template <>
struct THash<NTravel::NGeoCounter::TOfferBusDataKey> {
    inline size_t operator()(const NTravel::NGeoCounter::TOfferBusDataKey& v) const {
        return v.Hash();
    }
};

namespace NTravel::NGeoCounter {
    struct TOfferBusData {
        struct TOfferData {
            TOfferData(ui32 price, TMaybe<bool> freeCancellation, NTravelProto::EPansionType pansion)
                : Price(price)
                , FreeCancellation(freeCancellation)
                , Pansion(pansion)
            {
            }

            ui32 Price;
            TMaybe<bool> FreeCancellation;
            NTravelProto::EPansionType Pansion;
        };

        struct TCacheItem {
            TInstant ExpirationTimestamp;
            bool IsBoY;
            TVector<TOfferData> Offers;

            size_t CalcTotalByteSize() const;
        };

        size_t SlowCalcTotalByteSize() const;

        THashMap<TOfferBusDataKey, THashMap<NTravelProto::EPartnerId, TCacheItem>> Offers;
    };

    struct TFilteringPermalinkInfo {
        TStaticPermalinkInfo StaticPermalinkInfo;
        TPriceInfo PriceInfo;
        TMaybe<TOfferBusData> OfferBusData;
        TSortIndices SortIndices;

        size_t CalcTotalByteSize() const;
    };

    class TUserSegmentsRegistry {
    public:
        struct TSegment {
            int SegmentId;
            TString SegmentName;
        };

        enum class EKeywordType {
            KT_UNKNOWN,
            KT_PAIR_VALUES,
            KT_WEIGHTED_UINT_VALUES,
            KT_UINT_VALUES,
            KT_WEIGHTED_UINT_WITH_SELECTED_VALUES, // Особый тип для соцдема, который передаётся дважды: и с весами, и с выбранным наиболее вероятным сегментом в кейворде 569
            KT_IGNORED,
        };

        struct TKeyword {
            TKeyword();
            TKeyword(int keywordId, const TString& keywordName, const TVector<TSegment>& possibleSegments, EKeywordType keywordType);

            int KeywordId;
            TString KeywordName;
            TVector<TSegment> PossibleSegments;
            EKeywordType KeywordType;
        };

        struct TKeywordAndSegment {
            TKeyword Keyword;
            TSegment Segment;
        };

        static int GenderKeywordId;
        static int AgeKeywordId;
        static int IncomeKeywordId;

        TMaybe<TKeyword> GetKeywordById(int keywordId) const;
        TKeyword GetKeywordByIdOrFail(int keywordId) const;
        TMaybe<TKeywordAndSegment> GetSegmentById(int keywordId, int segmentId) const;
        TKeywordAndSegment GetSegmentByIdOrFail(int keywordId, int segmentId) const;
        TMaybe<TKeywordAndSegment> GetSegmentByName(const TString& keywordName, const TString& segmentName) const;
        TKeywordAndSegment GetSegmentByNameOrFail(const TString& keywordName, const TString& segmentName) const;

        static TUserSegmentsRegistry BuildRegistry();
    private:
        void RegisterKeyword(const TKeyword& keyword);

        THashMap<int, TKeyword> Keywords;
        THashMap<TString, TKeyword> KeywordsByName;
    };

    const static TString HotelPansionFeatureBusinessId = "hotel_pansion";
    const static TString HotelPartnerFeatureBusinessId = "hotel_provider";
}
