#include "features_filtering.h"

namespace {
    TMaybe<std::bitset<32>> BuildAllowedPansionsMask(TMaybe<TVector<TString>> pansionAliases) {
        if (pansionAliases.Empty()) {
            return {};
        }
        std::bitset<32> res;
        for (const auto& alias: pansionAliases.GetRef()) {
            // Should be same lists as here https://a.yandex-team.ru/arc/trunk/arcadia/travel/hotels/tools/build_hotel_traits/queries/hotel_traits.yql
            TVector<NTravelProto::EPansionType> currPansions;
            if (alias == "hotel_pansion_breakfast_included") {
                currPansions = {
                    NTravelProto::EPansionType::PT_AI,
                    NTravelProto::EPansionType::PT_BB,
                    NTravelProto::EPansionType::PT_HB,
                    NTravelProto::EPansionType::PT_FB,
                    NTravelProto::EPansionType::PT_UAI,
                    NTravelProto::EPansionType::PT_LAI,
                };
            } else if (alias == "hotel_pansion_breakfast_dinner_included") {
                currPansions = {
                    NTravelProto::EPansionType::PT_AI,
                    NTravelProto::EPansionType::PT_HB,
                    NTravelProto::EPansionType::PT_FB,
                    NTravelProto::EPansionType::PT_UAI,
                    NTravelProto::EPansionType::PT_LAI,
                };
            } else if (alias == "hotel_pansion_breakfast_lunch_dinner_included") {
                currPansions = {
                    NTravelProto::EPansionType::PT_AI,
                    NTravelProto::EPansionType::PT_FB,
                    NTravelProto::EPansionType::PT_UAI,
                    NTravelProto::EPansionType::PT_LAI,
                };
            } else if (alias == "hotel_pansion_all_inclusive") {
                currPansions = {
                    NTravelProto::EPansionType::PT_AI,
                    NTravelProto::EPansionType::PT_UAI,
                    NTravelProto::EPansionType::PT_LAI,
                };
            } else if (alias == "hotel_pansion_no_pansion_included") {
                currPansions = {
                    NTravelProto::EPansionType::PT_UNKNOWN,
                    NTravelProto::EPansionType::PT_RO,
                };
            } else {
                ythrow yexception() << "Unknown pansion value: '" << alias << Endl;
            }
            for (auto curr: currPansions) {
                res.set(curr);
            }
        }
        return res;
    }
    TMaybe<TVector<int>> BuildPansionAliases(TMaybe<TVector<TString>> pansionAliases, NTravel::NGeoCounter::TStringEncoder& stringEncoder) {
        if (pansionAliases.Empty()) {
            return {};
        }
        TVector<int> res;
        res.reserve(pansionAliases->size());
        for (const auto& alias: pansionAliases.GetRef()) {
            res.push_back(stringEncoder.Encode(alias));
        }
        return res;
    }
}

namespace NTravel::NGeoCounter {
    TFilterBase::TFilterBase(int uniqueId)
        : UniqueId_(uniqueId)
    {
    }

    int TFilterBase::GetUniqueId() const {
        return UniqueId_;
    }

    TBasicFilterBase::TBasicFilterBase(int uniqueId, int businessId)
        : TFilterBase(uniqueId)
        , BusinessId_(businessId)
    {
    }

    int TBasicFilterBase::GetBusinessId() const {
        return BusinessId_;
    }

    TBasicFilterCompare::TBasicFilterCompare(int uniqueId, int businessId, bool isPassingOnNoData, double value, NTravelProto::NFilters::THotelFilter_TComparableValue_EMode mode)
        : TBasicFilterBase(uniqueId, businessId)
        , Value(value)
        , Mode(mode)
        , IsPassingOnNoData(isPassingOnNoData)
    {
    }

    bool TBasicFilterCompare::IsPassingFilter(const TFilteringPermalinkInfo& permalinkInfo) const {
        auto feature = GetFeature(permalinkInfo.StaticPermalinkInfo);
        if (feature.Empty()) {
            return IsPassingOnNoData;
        }
        if (!std::holds_alternative<double>(feature->Value)) {
            ythrow yexception() << "Invalid type of feature (BusinessId_=" << BusinessId_ << ")";
        }
        auto featureValue = std::get<double>(feature->Value);
        auto diff = featureValue - Value;
        switch (Mode) {
            case NTravelProto::NFilters::THotelFilter_TComparableValue_EMode::THotelFilter_TComparableValue_EMode_Greater:
                return diff > Eps_;
            case NTravelProto::NFilters::THotelFilter_TComparableValue_EMode::THotelFilter_TComparableValue_EMode_LessOrEqual:
                return diff < Eps_ || std::fabs(diff) < Eps_;
            case NTravelProto::NFilters::THotelFilter_TComparableValue_EMode::THotelFilter_TComparableValue_EMode_GreaterOrEqual:
                return diff > Eps_ || std::fabs(diff) < Eps_;
            case NTravelProto::NFilters::THotelFilter_TComparableValue_EMode::THotelFilter_TComparableValue_EMode_Less:
            default:
                return diff < Eps_;
        }
    }

    void TBasicFilterCompare::UpdateOfferSearchParams(const TOfferSearchParams&) {
    }

    TBasicFilterIntersect::TBasicFilterIntersect(int uniqueId, int businessId, bool isPassingOnNoData, TVector<int> values)
        : TBasicFilterBase(uniqueId, businessId)
        , Values(std::move(values))
        , IsPassingOnNoData(isPassingOnNoData)
    {
    }

    bool TBasicFilterIntersect::IsPassingFilter(const TFilteringPermalinkInfo& permalinkInfo) const {
        auto feature = GetFeature(permalinkInfo.StaticPermalinkInfo);
        if (feature.Empty()) {
            return IsPassingOnNoData;
        }

        return feature->HasOneOfValues(Values);
    }

    void TBasicFilterIntersect::UpdateOfferSearchParams(const TOfferSearchParams&) {
    }

    TBasicFilterIgnore::TBasicFilterIgnore(int uniqueId, int businessId)
        : TBasicFilterBase(uniqueId, businessId)
    {
    }

    bool TBasicFilterIgnore::IsPassingFilter(const TFilteringPermalinkInfo&) const {
        return true;
    }

    void TBasicFilterIgnore::UpdateOfferSearchParams(const TOfferSearchParams&) {
    }

    TFilterByPrice::TFilterByPrice(int uniqueId, TPriceRange totalPrice, TBookingRange bookingRange)
        : TFilterBase(uniqueId)
        , TotalPrice(totalPrice)
        , BookingRange(bookingRange)
    {
    }

    bool TFilterByPrice::IsPassingFilter(const TFilteringPermalinkInfo& permalinkInfo) const {
        if (!permalinkInfo.PriceInfo.HasData()) {
            return true;
        }
        auto permalinkTotalPrice = permalinkInfo.PriceInfo.GetPrice(BookingRange);
        return Max(TotalPrice.MinPrice, permalinkTotalPrice.MinPrice) <= Min(TotalPrice.MaxPrice, permalinkTotalPrice.MaxPrice);
    }

    void TFilterByPrice::UpdateOfferSearchParams(const TOfferSearchParams& offerSearchParams) {
        BookingRange = offerSearchParams.BookingRange;
    }

    TFilterByOfferBusData::TFilterByOfferBusData(int uniqueId,
                                                 TBookingRange bookingRange,
                                                 const TString& occupancy,
                                                 TPriceRange totalPrice,
                                                 TMaybe<bool> freeCancellation,
                                                 TMaybe<bool> withBreakfast,
                                                 bool yandexOffers,
                                                 TMaybe<bool> hasMirOffers,
                                                 TMaybe<TVector<TString>> pansionAliases,
                                                 TStringEncoder& stringEncoder)
        : TFilterBase(uniqueId)
        , BookingRange(bookingRange)
        , Occupancy(occupancy)
        , TotalPrice(totalPrice)
        , FreeCancellation(freeCancellation)
        , WithBreakfast(withBreakfast)
        , YandexOffers(yandexOffers)
        , HasMirOffers(hasMirOffers)
        , AllowedPansionsMask(BuildAllowedPansionsMask(pansionAliases))
        , PansionAliases(BuildPansionAliases(pansionAliases, stringEncoder))
        , HotelsPansionBusinessId(stringEncoder.Encode(HotelPansionFeatureBusinessId))
    {
    }

    bool TFilterByOfferBusData::IsPassingFilter(const TFilteringPermalinkInfo& permalinkInfo) const {
        if (PansionAliases.Defined()) {
            // todo (mpivko): adhoc solution. It's better to apply static part of filter to all hotels, not only when dynamic filter is checked
            auto it = permalinkInfo.StaticPermalinkInfo.Features.find(HotelsPansionBusinessId);
            if (it != permalinkInfo.StaticPermalinkInfo.Features.end()) {
                auto feature = it->second;
                if (!feature.HasOneOfValues(PansionAliases.GetRef())) {
                    return false;
                }
            }
        }
        if (permalinkInfo.OfferBusData.Empty()) {
            return true;
        }
        const auto& offerBusData = *permalinkInfo.OfferBusData.Get();
        auto key = TOfferBusDataKey(BookingRange, Occupancy);
        auto offersIt = offerBusData.Offers.find(key);
        if (offersIt == offerBusData.Offers.end()) {
            return true;
        }

        for (const auto& [partnerId, cacheItem] : offersIt->second) {
            bool isBoYOffer = cacheItem.IsBoY;
            if (!YandexOffers || isBoYOffer) {
                for (const auto& offer : cacheItem.Offers) {
                    bool hasBreakfast = HasBreakfast(offer.Pansion);
                    if (TotalPrice.MinPrice <= offer.Price && offer.Price <= TotalPrice.MaxPrice &&
                        (FreeCancellation.Empty() || offer.FreeCancellation == FreeCancellation) &&
                        (WithBreakfast.Empty() || hasBreakfast == WithBreakfast) &&
                        (HasMirOffers.Empty() || isBoYOffer == HasMirOffers /* rude approx */) &&
                        (AllowedPansionsMask.Empty() || AllowedPansionsMask.GetRef().test(offer.Pansion)))
                    {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    void TFilterByOfferBusData::UpdateOfferSearchParams(const TOfferSearchParams& offerSearchParams) {
        BookingRange = offerSearchParams.BookingRange;
        Occupancy = offerSearchParams.Ages.ToOccupancyString();
    }
}

template <>
void Out<NTravel::NGeoCounter::TBasicFilterCompare>(IOutputStream& out, const NTravel::NGeoCounter::TBasicFilterCompare& filter) {
    std::string stringMode = "<";
    switch (filter.Mode) {
        default:
        case NTravelProto::NFilters::THotelFilter_TComparableValue_EMode_Less:
            stringMode = "<";
            break;
        case NTravelProto::NFilters::THotelFilter_TComparableValue_EMode_Greater:
            stringMode = ">";
            break;
        case NTravelProto::NFilters::THotelFilter_TComparableValue_EMode_GreaterOrEqual:
            stringMode = ">=";
            break;
        case NTravelProto::NFilters::THotelFilter_TComparableValue_EMode_LessOrEqual:
            stringMode = "<=";
            break;
    }
    out << stringMode << filter.Value;
}

template <>
void Out<NTravel::NGeoCounter::TBasicFilterIntersect>(IOutputStream& out, const NTravel::NGeoCounter::TBasicFilterIntersect& filter) {
    out << "list(" << JoinVectorIntoString(filter.Values, "|") << ")";
}

template <>
void Out<NTravel::NGeoCounter::TBasicFilterIgnore>(IOutputStream& out, const NTravel::NGeoCounter::TBasicFilterIgnore&) {
    out << "ignore()";
}

template <>
void Out<NTravel::NGeoCounter::TFilterByPrice>(IOutputStream& out, const NTravel::NGeoCounter::TFilterByPrice& filter) {
    out << "price([" << filter.TotalPrice.MinPrice << "; " << filter.TotalPrice.MaxPrice << "] on [" << filter.BookingRange.CheckInDate << "; " << filter.BookingRange.CheckOutDate
        << "])";
}

template <>
void Out<NTravel::NGeoCounter::TFilterByOfferBusData>(IOutputStream& out, const NTravel::NGeoCounter::TFilterByOfferBusData& filter) {
    out << "offerBusData([" << filter.BookingRange.CheckInDate << "; " << filter.BookingRange.CheckOutDate << "; " << filter.Occupancy << "] "
        << "[" << filter.TotalPrice.MinPrice << "; " << filter.TotalPrice.MaxPrice << "] "
        << "freeCancellation: " << filter.FreeCancellation << " withBreakfast: " << filter.WithBreakfast << ")";
}

template <>
void Out<std::unique_ptr<NTravel::NGeoCounter::TBasicFilterBase>>(IOutputStream& out, const std::unique_ptr<NTravel::NGeoCounter::TBasicFilterBase>& filter) {
    auto intersectFilter = dynamic_cast<const NTravel::NGeoCounter::TBasicFilterIntersect*>(filter.get());
    if (intersectFilter) {
        out << *intersectFilter;
        return;
    }
    auto compareFilter = dynamic_cast<const NTravel::NGeoCounter::TBasicFilterCompare*>(filter.get());
    if (compareFilter) {
        out << *compareFilter;
        return;
    }
    ythrow yexception() << "Unknown TBasicFilterBase child";
}

template <>
void Out<std::unique_ptr<NTravel::NGeoCounter::TFilterBase>>(IOutputStream& out, const std::unique_ptr<NTravel::NGeoCounter::TFilterBase>& filter) {
    auto intersectFilter = dynamic_cast<const NTravel::NGeoCounter::TBasicFilterIntersect*>(filter.get());
    if (intersectFilter) {
        out << *intersectFilter;
        return;
    }
    auto compareFilter = dynamic_cast<const NTravel::NGeoCounter::TBasicFilterCompare*>(filter.get());
    if (compareFilter) {
        out << *compareFilter;
        return;
    }
    auto ignoreFilter = dynamic_cast<const NTravel::NGeoCounter::TBasicFilterIgnore*>(filter.get());
    if (ignoreFilter) {
        out << *ignoreFilter;
        return;
    }
    auto filterByPrice = dynamic_cast<const NTravel::NGeoCounter::TFilterByPrice*>(filter.get());
    if (filterByPrice) {
        out << *filterByPrice;
        return;
    }
    auto filterByOfferBusData = dynamic_cast<const NTravel::NGeoCounter::TFilterByOfferBusData*>(filter.get());
    if (filterByOfferBusData) {
        out << *filterByOfferBusData;
        return;
    }
    ythrow yexception() << "Unknown TFilterBase child";
}

template <>
void Out<NTravel::NGeoCounter::TBasicFilterGroup>(IOutputStream& out, const NTravel::NGeoCounter::TBasicFilterGroup& group) {
    out << "Group[" << group.Id << "](" << JoinVectorIntoString(group.Filters, ", ") << ")";
}
