#pragma once

#include "data.h"

namespace NTravel::NGeoCounter {
    class TFilterBase {
    public:
        virtual bool IsPassingFilter(const TFilteringPermalinkInfo& permalinkInfo) const = 0;
        int GetUniqueId() const;
        virtual void UpdateOfferSearchParams(const TOfferSearchParams& offerSearchParams) = 0;
        virtual ~TFilterBase() = default;

    protected:
        explicit TFilterBase(int uniqueId);
        TFilterBase() = default;

    private:
        int UniqueId_;
    };

    class TBasicFilterBase: public TFilterBase {
    public:
        explicit TBasicFilterBase(int uniqueId, int businessId);
        int GetBusinessId() const;

    protected:
        TMaybe<TFeature> GetFeature(const TStaticPermalinkInfo& permalinkInfo) const {
            if (!permalinkInfo.Features.contains(BusinessId_)) {
                return {};
            }
            return permalinkInfo.Features.at(BusinessId_);
        }

        int BusinessId_;
    };

    class TBasicFilterCompare: public TBasicFilterBase {
    public:
        explicit TBasicFilterCompare(
                int uniqueId,
                int businessId,
                bool isPassingOnNoData,
                double value,
                NTravelProto::NFilters::THotelFilter_TComparableValue_EMode mode
                    = NTravelProto::NFilters::THotelFilter_TComparableValue_EMode::THotelFilter_TComparableValue_EMode_Less);
        bool IsPassingFilter(const TFilteringPermalinkInfo& permalinkInfo) const override;
        void UpdateOfferSearchParams(const TOfferSearchParams& offerSearchParams) override;

        double Value;
        NTravelProto::NFilters::THotelFilter_TComparableValue_EMode Mode;
        bool IsPassingOnNoData;

    private:
        const double Eps_ = 1e-5;
    };

    class TBasicFilterIntersect: public TBasicFilterBase {
    public:
        explicit TBasicFilterIntersect(int uniqueId, int businessId, bool isPassingOnNoData, TVector<int> values);
        bool IsPassingFilter(const TFilteringPermalinkInfo& permalinkInfo) const override;
        void UpdateOfferSearchParams(const TOfferSearchParams& offerSearchParams) override;

        TVector<int> Values;
        bool IsPassingOnNoData;
    };

    class TBasicFilterIgnore: public TBasicFilterBase {
     public:
        explicit TBasicFilterIgnore(int uniqueId, int businessId);
        bool IsPassingFilter(const TFilteringPermalinkInfo& permalinkInfo) const override;
        void UpdateOfferSearchParams(const TOfferSearchParams& offerSearchParams) override;
    };

    class TFilterByPrice: public TFilterBase {
    public:
        explicit TFilterByPrice(int uniqueId, TPriceRange totalPrice, TBookingRange bookingRange);
        bool IsPassingFilter(const TFilteringPermalinkInfo& permalinkInfo) const override;
        void UpdateOfferSearchParams(const TOfferSearchParams& offerSearchParams) override;

        TPriceRange TotalPrice;
        TBookingRange BookingRange;
    };

    class TFilterByOfferBusData: public TFilterBase {
    public:
        explicit TFilterByOfferBusData(int uniqueId,
                                       TBookingRange bookingRange,
                                       const TString& occupancy,
                                       TPriceRange totalPrice,
                                       TMaybe<bool> freeCancellation,
                                       TMaybe<bool> withBreakfast,
                                       bool yandexOffers,
                                       TMaybe<bool> hasMirOffers,
                                       TMaybe<TVector<TString>> pansionAliases,
                                       TStringEncoder& stringEncoder);
        bool IsPassingFilter(const TFilteringPermalinkInfo& permalinkInfo) const override;
        void UpdateOfferSearchParams(const TOfferSearchParams& offerSearchParams) override;

        TBookingRange BookingRange;
        TString Occupancy;
        TPriceRange TotalPrice;
        TMaybe<bool> FreeCancellation;
        TMaybe<bool> WithBreakfast;
        bool YandexOffers;
        TMaybe<bool> HasMirOffers;
        TMaybe<std::bitset<32>> AllowedPansionsMask;
        TMaybe<TVector<int>> PansionAliases;
        int HotelsPansionBusinessId;
    };

    struct TBasicFilterGroup {
        int Id;
        TVector<std::unique_ptr<TFilterBase>> Filters;
    };
}
