#include "filter_registry.h"

namespace {
    static const TString hotelPansionFeatureId = "hotel_pansion";
}

namespace NTravel::NGeoCounter {
    TFilterRegistry::TFilterRegistry(TStringEncoder& stringEncoder)
        : StringEncoder_(stringEncoder)
    {
    }

    TFeature TFilterRegistry::BuildFeature(const TString& id, const TVector<TString>& values) const {
        if (id == "rating") {
            if (values.size() != 1) {
                Y_VERIFY(!values.empty());
                ythrow yexception() << "Expected single value for numeric feature " << id;
            }
            const TString& value = values[0];
            return TFeature(FromString<double>(value));
        } else {
            TVector<int> newValues(values.size());
            for (size_t i = 0; i < values.size(); i++) {
                newValues[i] = StringEncoder_.Encode(values[i]);
            }
            return TFeature(newValues);
        }
    }

    std::unique_ptr<TFilterBase> TFilterRegistry::BuildFilterForPredefined(const NTravelProto::NFilters::THotelFilter& pbFilter) const {
        return BuildFilter(pbFilter, TBookingRange(NOrdinalDate::TOrdinalDate(), NOrdinalDate::TOrdinalDate()), {}); // todo (mpivko): avoid using this dummy-values here
    }

    std::unique_ptr<TFilterByOfferBusData> TFilterRegistry::BuildFilterByOfferData(const NTravelProto::NFilters::THotelFilter& pbFilter,
                                                                                   const TBookingRange& bookingRange,
                                                                                   const TAges& ages) const {
        const auto& uniqueIdRaw = pbFilter.GetUniqueId();
        auto priceRange = NTravel::NGeoCounter::TPriceRange::MAX_RANGE;
        if (pbFilter.HasPriceValue()) {
            const auto& priceValue = pbFilter.GetPriceValue();
            if (priceValue.HasTotalPriceFrom()) {
                priceRange.MinPrice = priceValue.GetTotalPriceFrom().value();
            }
            if (priceValue.HasTotalPriceTo()) {
                priceRange.MaxPrice = priceValue.GetTotalPriceTo().value();
            }
        }
        auto pansionAliases = TMaybe<TVector<TString>>();
        if (pbFilter.GetFeatureId() == hotelPansionFeatureId) {
            if (pbFilter.HasListValue()) {
                pansionAliases = TVector<TString>(pbFilter.GetListValue().GetValue().begin(), pbFilter.GetListValue().GetValue().end());
            } else {
                pansionAliases = TVector<TString>();
            }
        }
        return std::make_unique<TFilterByOfferBusData>(StringEncoder_.Encode("~" + uniqueIdRaw + "~"), // todo (mpivko): tmp hack to avoid collisions
                                                       bookingRange,
                                                       ages.ToOccupancyString(),
                                                       uniqueIdRaw == "~price~" ? priceRange : TPriceRange::MAX_RANGE,
                                                       uniqueIdRaw == "free_cancellation" ? true : TMaybe<bool>(),
                                                       uniqueIdRaw == "breakfast_included" ? true : TMaybe<bool>(), // todo (mpivko): remove it, it's legacy
                                                       uniqueIdRaw == "yandex-offers",
                                                       uniqueIdRaw == "mir-offers" ? true : TMaybe<bool>(),
                                                       pansionAliases,
                                                       StringEncoder_);
    }

    std::unique_ptr<TFilterBase> TFilterRegistry::BuildFilterSimple(const NTravelProto::NFilters::THotelFilter& pbFilter,
                                                                    const TBookingRange& bookingRange,
                                                                    bool isPassingOnNoData) const {
        auto featureId = StringEncoder_.Encode(pbFilter.GetFeatureId());
        auto uniqueId = StringEncoder_.Encode(pbFilter.GetUniqueId());
        if (pbFilter.HasListValue()) {
            TVector<TString> rawValues(pbFilter.GetListValue().GetValue().begin(), pbFilter.GetListValue().GetValue().end());
            TVector<int> encodedValues(rawValues.size());
            for (size_t i = 0; i < rawValues.size(); i++) {
                encodedValues[i] = StringEncoder_.Encode(rawValues[i]);
            }
            return {std::make_unique<TBasicFilterIntersect>(uniqueId, featureId, isPassingOnNoData, encodedValues)};
        } else if (pbFilter.HasComparableValue()) {
            auto value = pbFilter.GetComparableValue().GetValue();
            auto mode = pbFilter.GetComparableValue().GetMode();
            return {std::make_unique<TBasicFilterCompare>(uniqueId, featureId, isPassingOnNoData, value, mode)};
        } else if (pbFilter.HasPriceValue()) {
            const auto& priceValue = pbFilter.GetPriceValue();
            auto priceRange = TPriceRange::MAX_RANGE;
            if (priceValue.HasTotalPriceFrom()) {
                priceRange.MinPrice = priceValue.GetTotalPriceFrom().value();
            }
            if (priceValue.HasTotalPriceTo()) {
                priceRange.MaxPrice = priceValue.GetTotalPriceTo().value();
            }
            return {std::make_unique<TFilterByPrice>(uniqueId, priceRange, bookingRange)};
        } else if (pbFilter.HasIgnoredValue()) {
            return {std::make_unique<TBasicFilterIgnore>(uniqueId, featureId)};
        } else {
            ythrow yexception() << "None filter value is set";
        }
    }

    TVector<std::unique_ptr<TFilterBase>> TFilterRegistry::BuildMultiFilter(const NTravelProto::NFilters::THotelFilter& pbFilter,
                                                                            const TBookingRange& bookingRange,
                                                                            const TAges& ages,
                                                                            bool addOfferDataFilters,
                                                                            bool addSimplePreFilters) const {
        TVector<std::unique_ptr<TFilterBase>> result{};
        if (IsIn({"FAKE-ID-yandex-offers", "FAKE-ID-mir-offers", "FAKE-ID-only-boy-offers"}, pbFilter.GetFeatureId())) {
            if (addOfferDataFilters) {
                result.push_back(BuildFilterByOfferData(pbFilter, bookingRange, ages));
            }
            if (addSimplePreFilters) {
                result.push_back(std::make_unique<TBasicFilterIntersect>(StringEncoder_.Encode(pbFilter.GetUniqueId()),
                                                                         StringEncoder_.Encode("hotel_has_boy_partner"),
                                                                         false,
                                                                         TVector<int>{StringEncoder_.Encode("1")}));
            }
        } else if (IsIn<TString>({"free_cancellation", "~price~", hotelPansionFeatureId}, pbFilter.GetFeatureId())) {
            if (addOfferDataFilters) {
                result.push_back(BuildFilterByOfferData(pbFilter, bookingRange, ages));
            }
            if (addSimplePreFilters) {
                result.push_back(BuildFilterSimple(pbFilter, bookingRange, true));
            }
        } else {
            result.push_back(BuildFilterSimple(pbFilter, bookingRange, false));
        }
        return result;
    }

    std::unique_ptr<TFilterBase> TFilterRegistry::BuildFilter(const NTravelProto::NFilters::THotelFilter& pbFilter,
                                                              const TBookingRange& bookingRange,
                                                              const TAges& ages) const {
        auto filters = BuildMultiFilter(pbFilter, bookingRange, ages, true, false);
        Y_ENSURE(filters.size() == 1, "Expected exactly one filter");
        return std::move(filters.at(0));
    }

    TAdditionalFilter TFilterRegistry::BuildAdditionalFilter(const NTravelProto::NGeoCounter::TGetCountsRequest::TAdditionalFilter& pbAdditionalFilter,
                                                             const TBookingRange& bookingRange,
                                                             const TAges& ages) const {
        return TAdditionalFilter(StringEncoder_.Encode(pbAdditionalFilter.GetGroupId()),
                                 ConvertAdditionalFilterTypeEnum(pbAdditionalFilter.GetType()),
                                 BuildFilter(pbAdditionalFilter.GetFilter(), bookingRange, ages));
    }

    TAdditionalFilter::EType TFilterRegistry::ConvertAdditionalFilterTypeEnum(const NTravelProto::NGeoCounter::TGetCountsRequest::TAdditionalFilter::EType& type) const {
        switch (type) {
#define TYPE(_PB_NAME_, _INNER_NAME_)                                                                                                   \
    case NTravelProto::NGeoCounter::TGetCountsRequest::TAdditionalFilter::EType::TGetCountsRequest_TAdditionalFilter_EType_##_PB_NAME_: \
        return TAdditionalFilter::EType::_INNER_NAME_
            TYPE(Single, Single);
            TYPE(Or, Or);
            TYPE(And, And);
#undef TYPE
            default:
                ythrow yexception() << "Unknown enum value " << type;
        }
    }
}
