#include "coordinate_utils.h"
#include "hotel_search_service.h"

#include <travel/hotels/lib/cpp/util/base64.h>

#include <util/digest/murmur.h>

namespace {
    const static int DefaultGeoId = 213; // Moscow

    const THashMap<TString, NTravelProto::NOfferCache::NApi::EPansionAlias> PansionAliasesMapping = {
        {"hotel_pansion_breakfast_included", NTravelProto::NOfferCache::NApi::EPansionAlias::BREAKFAST},
        {"hotel_pansion_breakfast_dinner_included", NTravelProto::NOfferCache::NApi::EPansionAlias::BREAKFAST_DINNER},
        {"hotel_pansion_breakfast_lunch_dinner_included", NTravelProto::NOfferCache::NApi::EPansionAlias::BREAKFAST_LUNCH_DINNER},
        {"hotel_pansion_all_inclusive", NTravelProto::NOfferCache::NApi::EPansionAlias::ALL_INCLUSIVE},
        {"hotel_pansion_no_pansion_included", NTravelProto::NOfferCache::NApi::EPansionAlias::NO_PANSION},
    }; // todo (mpivko): get rid of it

    const static TString mirOfferFilterBusinessId = "FAKE-ID-mir-offers";
    const static TString onlyBoYOffers = "FAKE-ID-only-boy-offers";

    const NTravel::NGeoCounter::TBasicFilterGroup* FindFilterGroupById(const TVector<NTravel::NGeoCounter::TBasicFilterGroup>& filterGroups, int id) {
        for (const auto& filterGroup: filterGroups) {
            if (filterGroup.Id == id) {
                return &filterGroup;
            }
        }
        return nullptr;
    }

    const NTravel::NGeoCounter::TFilterByOfferBusData* FindFirstMatchingOfferBusFilter(const TVector<std::unique_ptr<NTravel::NGeoCounter::TFilterBase>>& offerBusDataFilters,
                                                                                       const std::function<bool(NTravel::NGeoCounter::TFilterByOfferBusData*)>& isMatching) {
        for (const auto& filter: offerBusDataFilters) {
            auto offerBusData = dynamic_cast<NTravel::NGeoCounter::TFilterByOfferBusData*>(filter.get());
            Y_ENSURE(offerBusData, "Expected TFilterByOfferBusData in offerBusDataFilters");
            if (isMatching(offerBusData)) {
                return offerBusData;
            }
        }
        return nullptr;
    }

    const TString CreateUid() {
        return JoinStrings(StringSplitter(CreateGuidAsString()).Split('-').ToList<TString>(), "");
    }

    TString EncodePollingId(const NTravelProto::NGeoCounter::TIncrementalPollingIterationToken& token) {
        TStringBuilder builder;
        builder << token.GetSessionId() << "-";
        builder << token.GetSearchIteration() << "-";
        builder << "newsearch";
        return builder;
    }

    TString EncodePollingToken(const NTravelProto::NGeoCounter::TIncrementalPollingIterationToken& token) {
        TStringBuilder builder;
        builder << EncodePollingId(token) << "-";
        builder << token.GetPollingIteration() << "~";
        builder << NTravel::Base64EncodeUrlShort(token.SerializeAsString());
        return builder;
    }

    NTravelProto::NGeoCounter::TIncrementalPollingIterationToken DecodePollingToken(const TString& encodedToken) {
        auto encodedData = StringSplitter(encodedToken).Split('~').ToList<TString>().back();
        TString tokenData;
        try {
            tokenData = NTravel::Base64DecodeAny(encodedData);
        } catch (...) {
            throw NTravel::NGeoCounter::THotelSearchService::invalid_polling_token_exception() << "Failed to decode base64 from token";
        }
        if (tokenData.empty()) {
            throw NTravel::NGeoCounter::THotelSearchService::invalid_polling_token_exception() << "No data was decoded";
        }
        NTravelProto::NGeoCounter::TIncrementalPollingIterationToken token;
        if (!token.ParseFromString(tokenData)) {
            throw NTravel::NGeoCounter::THotelSearchService::invalid_polling_token_exception() << "Failed to parse protobuf in token";
        }
        return token;
    }

    ui64 CalcSearchParamsHash(const NTravel::NGeoCounter::TBoundingBox& bbox,
                              int geoId,
                              const NTravel::NGeoCounter::TOfferSearchParams& offerSearchParams,
                              const NTravelProto::NGeoCounter::ESortType& sortType,
                              size_t skip,
                              size_t limit) {
        TStringBuilder builder;
        Y_UNUSED(bbox); // Can't use it in hash because of precision
        builder << geoId << "|";
        builder << offerSearchParams.BookingRange.CheckInDate << "|";
        builder << offerSearchParams.BookingRange.CheckOutDate << "|";
        builder << offerSearchParams.Ages.ToAgesString() << "|";
        builder << sortType << "|";
        builder << skip << "|";
        builder << limit << "|";

        TString input = builder;
        return MurmurHash<ui64>(input.Data(), input.Size());
    }

    NTravel::NGeoCounter::TPosition ParseCoordinatesFromProto(const NTravelProto::NGeoCounter::TCoordinates& pbCoordinates) {
        return NTravel::NGeoCounter::TPosition(pbCoordinates.GetLat(), pbCoordinates.GetLon());
    }
}

namespace NTravel::NGeoCounter {
    THotelSearchService::THotelSearchService(TOfferCacheClient& offerCacheClient,
                                             TPromoServiceClient& promoServiceClient,
                                             TIndex& index,
                                             TStringEncoder& stringEncoder,
                                             TRegionsService& regionsService,
                                             const TSortTypeRegistry& sortTypeRegistry,
                                             const TFilterRegistry& filterRegistry,
                                             const TBigbClient& bigbClient,
                                             const TUserSegmentsRegistry& userSegmentsRegistry)
        : OfferCacheClient_(offerCacheClient)
        , PromoServiceClient_(promoServiceClient)
        , Index_(index)
        , StringEncoder_(stringEncoder)
        , RegionsService_(regionsService)
        , SortTypeRegistry_(sortTypeRegistry)
        , FilterRegistry_(filterRegistry)
        , BigbClient_(bigbClient)
        , UserSegmentsRegistry_(userSegmentsRegistry)
    {
    }

    void THotelSearchService::RegisterCounters(NMonitor::TCounterSource& source) {
        source.RegisterSource(&Counters_, "HotelSearchService");
    }

    THotelSearchService::TSearchHotelsParsedRequestData THotelSearchService::ParseSearchHotels(const NTravelProto::NGeoCounter::TGetHotelsRequest& req,
                                                                                               const TString& grcpCallId) const {
        TString reqId = req.GetDebugData().GetRequestId().empty() ? CreateUid() + "-g" : req.GetDebugData().GetRequestId();
        TString logId = "HotelsSearch(" + reqId + "-" + CreateUid() + ")";

        INFO_LOG << logId << ": Got request, grpc call id: " << grcpCallId << Endl;

        TMaybe<TBoundingBox> bbox{};
        TMaybe<TOfferSearchParams> offerSearchParams{};
        TVector<TBasicFilterGroup> filterGroups{};
        THashMap<TString, TVector<std::unique_ptr<TFilterBase>>> filterGroupsMap{};
        TVector<std::unique_ptr<TFilterBase>> offerBusDataFilters{};
        TMaybe<TTravelGeoId> travelGeoId;
        if (req.HasTravelGeoId() && req.GetTravelGeoId() != 0) { // Backward compatibility
            travelGeoId = req.GetTravelGeoId();
        }
        Y_ENSURE(travelGeoId.Defined() || req.HasTopHotelPermalink(), "Either GeoId or TopHotelPermalink is required");
        if (req.HasBbox()) {
            bbox = TBoundingBox(TPosition(req.GetBbox().GetLowerLeft().GetLat(), req.GetBbox().GetLowerLeft().GetLon()),
                                TPosition(req.GetBbox().GetUpperRight().GetLat(), req.GetBbox().GetUpperRight().GetLon()));
        }
        if (!req.GetCheckInDate().empty() && !req.GetCheckOutDate().empty() && !req.GetAges().empty()) {
            auto bookingRange = TBookingRange(NOrdinalDate::FromString(req.GetCheckInDate()), NOrdinalDate::FromString(req.GetCheckOutDate()));
            auto ages = TAges::FromAgesString(req.GetAges());
            offerSearchParams = TOfferSearchParams(bookingRange, ages);
        }
        auto bookingRangeForFilter = offerSearchParams.Defined() ? offerSearchParams->BookingRange : TBookingRange(0, 0);
        auto agesForFilter = offerSearchParams.Defined() ? offerSearchParams->Ages : TAges();
        for (const auto& filter : req.GetFilters()) {
            for (auto& builtFilter: FilterRegistry_.BuildMultiFilter(filter, bookingRangeForFilter, agesForFilter, true, true)) {
                if (dynamic_cast<TFilterByOfferBusData*>(builtFilter.get())) {
                    offerBusDataFilters.push_back(std::move(builtFilter));
                } else {
                    filterGroupsMap[filter.GetFeatureId()].push_back(std::move(builtFilter));
                }
            }
        }
        for (auto& [featureId, filters] : filterGroupsMap) {
            filterGroups.push_back(TBasicFilterGroup{StringEncoder_.Encode(featureId), std::move(filters)});
        }

        Y_ENSURE(req.GetSortType() != NTravelProto::NGeoCounter::ESortType::ST_Unknown, "Sort type is required");
        Y_ENSURE(NTravelProto::NGeoCounter::ESortType_IsValid(req.GetSortType()), "Sort type is invalid");
        if (req.GetSortType() == NTravelProto::NGeoCounter::ESortType::ST_ByDistance) {
            Y_ENSURE(req.HasSortOrigin(), "SortOrigin is required for sort by distance");
            Y_ENSURE(!req.HasTopHotelPermalink(), "Can't use TopHotelPermalink with sort by distance");
        }

        Y_ENSURE(req.GetSkip() >= 0, "Skip should be non-negative");
        Y_ENSURE(req.GetLimit() > 0, "Limit should be positive");
        Y_ENSURE(req.GetLimit() <= 1000, "Limit can't be more than 1000");

        bool incrementalPollingEnabled = false;
        TMaybe<NTravelProto::NGeoCounter::TIncrementalPollingIterationToken> incrementalPollingPreviousToken{};
        if (req.HasIncrementalPolling() && req.GetIncrementalPolling().GetEnabled()) {
            incrementalPollingEnabled = true;
            if (!req.GetIncrementalPolling().GetPreviousIterationToken().empty()) {
                incrementalPollingPreviousToken = DecodePollingToken(req.GetIncrementalPolling().GetPreviousIterationToken());

                if (req.GetIncrementalPolling().GetPollingIteration() > 0) {
                    Y_ENSURE(travelGeoId.Defined(), "GeoId is required for polling continuation");
                    if (bbox.Empty() || offerSearchParams.Empty()) {
                        WARNING_LOG << logId << ": Got polling token, but no bbox or OfferSearchParams" << Endl;
                        incrementalPollingPreviousToken = {};
                    } else if (incrementalPollingPreviousToken->GetSearchParamsHash() !=
                        CalcSearchParamsHash(bbox.GetRef(), travelGeoId.GetRef(), offerSearchParams.GetRef(),
                                             req.GetSortType(), static_cast<size_t>(req.GetSkip()),
                                             static_cast<size_t>(req.GetLimit()))) {
                        WARNING_LOG << logId << ": Got polling token, but hashes don't match" << Endl;
                        incrementalPollingPreviousToken = {};
                    }
                }
            }
        }

        return TSearchHotelsParsedRequestData{
            .RawReq = &req,
            .BoundingBox = bbox,
            .TravelGeoId = travelGeoId,
            .SelectedFiltersGroups = std::move(filterGroups),
            .OfferBusDataFilters = std::move(offerBusDataFilters),
            .OfferSearchParams = offerSearchParams,
            .ProtoSortType = req.GetSortType(),
            .SortOrigin = req.HasSortOrigin() ? ParseCoordinatesFromProto(req.GetSortOrigin()) : TMaybe<TPosition>(),
            .Skip = static_cast<size_t>(req.GetSkip()),
            .Limit = static_cast<size_t>(req.GetLimit()),
            .IncrementalPollingEnabled = incrementalPollingEnabled,
            .IncrementalPollingPreviousToken = incrementalPollingPreviousToken,
            .PollingIteration = req.GetIncrementalPolling().GetPollingIteration(),
            .ReqId = reqId,
            .LogId = logId,
            .UseSearcher = req.GetUseSearcher(),
            .TopHotelPermalink = req.HasTopHotelPermalink() ? req.GetTopHotelPermalink().value() : TMaybe<TPermalink>(),
        };
    }

    NTravelProto::NGeoCounter::TGetHotelsResponse THotelSearchService::SearchHotels(const THotelSearchService::TSearchHotelsParsedRequestData& requestData) const {
        auto logId = requestData.LogId;
        auto now = TInstant::Now();
        auto hasMirFilter = FindFilterGroupById(requestData.SelectedFiltersGroups, StringEncoder_.Encode(mirOfferFilterBusinessId)) != nullptr; // todo (mpivko): dirty hack
        auto mirParams = PromoServiceClient_.GetActivePromos(now, logId).GetMir();
        auto rawFilledOfferSearchParams = requestData.OfferSearchParams.Defined()
                                       ? requestData.OfferSearchParams.GetRef()
                                       : DetermineOfferSearchParams(now, hasMirFilter, mirParams);
        auto pollingContinuation = requestData.IncrementalPollingPreviousToken.Defined() && requestData.PollingIteration > 0;
        auto filledOfferSearchParams = pollingContinuation ? rawFilledOfferSearchParams : AdjustDates(now, rawFilledOfferSearchParams);
        if (requestData.OfferSearchParams.Empty()) {
            for (auto& group: requestData.SelectedFiltersGroups) {
                for (auto& filter: group.Filters) {
                    filter->UpdateOfferSearchParams(filledOfferSearchParams);
                }
            }
            for (auto& filter: requestData.OfferBusDataFilters) {
                filter->UpdateOfferSearchParams(filledOfferSearchParams);
            }
        }
        auto filledTravelGeoId = DetermineTravelGeoId(logId, requestData.TravelGeoId, requestData.TopHotelPermalink);
        auto filledBoundingBox = requestData.BoundingBox.Defined() ? requestData.BoundingBox.GetRef() :
                                 DetermineBbox(logId, filledTravelGeoId, requestData.TopHotelPermalink, filledOfferSearchParams, requestData.SelectedFiltersGroups);

        TCryptaSegments userCryptaSegments{};
        if (pollingContinuation && requestData.IncrementalPollingPreviousToken.Defined() && requestData.IncrementalPollingPreviousToken->HasCryptaInfo()) {
            userCryptaSegments = ConvertCryptaInfoFromProto(requestData.IncrementalPollingPreviousToken->GetCryptaInfo());
        } else {
            userCryptaSegments = DetermineUserCryptaSegments(requestData);
        }
        TSortType sortType = SortTypeRegistry_.GetSortType(requestData.ProtoSortType, userCryptaSegments);
        auto sortTypeWithContext = TSortTypeWithContext(sortType, TSortType::TSortContext(userCryptaSegments, requestData.SortOrigin));

        THotelsResults result;
        bool allFinished = false;
        size_t finishedPrefixLength = 0;
        TVector<THotelsResults::THotelResult> newHotels;

        // We shouldn't pass skip to index, because some hotels among skip prefix could have been dropped because of absence of offers.
        // Also, it's required for sorts with CmpSortBeforeResp.
        // So, we use fakeLimit to trim head later
        auto fakeLimit = requestData.Skip + requestData.Limit;
        auto minFinishedPrefix = fakeLimit + 1; // +1 for haveMoreHotels // todo (mpivko): actually we don't need to wait for that last hotel, one offer is enough
        TVector<size_t> rawLimits = {fakeLimit * 2, static_cast<size_t>(400), static_cast<size_t>(1000)};
        if (sortType.GetCmpSortBeforeResp().Defined()) {
            minFinishedPrefix = Max(minFinishedPrefix, static_cast<size_t>(400));
            rawLimits = {Max(minFinishedPrefix, fakeLimit * 2), static_cast<size_t>(1000)};
        }
        size_t minIndexRecords = sortType.GetReorderTopRecordsParams().Defined() ? sortType.GetReorderTopRecordsParams().GetRef().TopSize : 0;
        TVector<std::pair<size_t, size_t>> limits;
        for (const auto& rawLimit: rawLimits) {
            limits.emplace_back(Max(minIndexRecords, rawLimit), rawLimit);
        }

        auto haveMoreHotels = false;
        for (auto [indexLimit, currentLimit]: limits) {
            result = Index_.GetHotels(filledBoundingBox, requestData.SelectedFiltersGroups, sortTypeWithContext,
                                      filledOfferSearchParams.BookingRange, filledOfferSearchParams.Ages.ToOccupancyString(),
                                      requestData.TopHotelPermalink, 0, indexLimit);
            if (sortType.GetReorderTopRecordsParams().Defined()) {
                TProfileTimer timer;
                sortType.GetReorderTopRecordsParams().GetRef().ReorderFunc(&result.Hotels, sortTypeWithContext.Context);
                auto reorderTime = timer.Get();
                if (result.Hotels.size() < 25) {
                    Counters_.ReorderTopRecordsTimeSmall.Update(reorderTime.MilliSeconds());
                } else if (result.Hotels.size() >= 25 && result.Hotels.size() < 400) {
                    Counters_.ReorderTopRecordsTimeMedium.Update(reorderTime.MilliSeconds());
                } else {
                    Counters_.ReorderTopRecordsTimeLarge.Update(reorderTime.MilliSeconds());
                }
                Counters_.ReorderTopRecordsTimeTotal.Update(reorderTime.MilliSeconds());
            }
            if (result.Hotels.size() > currentLimit) {
                result.Hotels.resize(currentLimit);
            }

            auto ocResp = PrepareAndSendOfferCacheRequest(requestData,
                                                          result,
                                                          requestData.SelectedFiltersGroups,
                                                          requestData.OfferBusDataFilters,
                                                          filledOfferSearchParams.BookingRange,
                                                          filledOfferSearchParams.Ages);
            std::tie(allFinished, finishedPrefixLength, newHotels) =
                BuildHotelsWithOcData(logId, result.Hotels, ocResp, requestData.RawReq->GetAllowHotelsWithoutPrices(), minFinishedPrefix);
            auto cmpSort = sortType.GetCmpSortBeforeResp();
            if (cmpSort.Defined()) {
                StableSort(newHotels, cmpSort.GetRef());
            }
            if (newHotels.size() > fakeLimit) {
                haveMoreHotels = true;
                newHotels.resize(fakeLimit);
            }

            if (!allFinished || newHotels.size() >= fakeLimit || result.Hotels.size() < currentLimit) {
                break;
            }
        }

        bool hasBoyOffers = false;
        result.HotelLocationUseful = false;
        for (const auto& hotel: newHotels) {
            if (hotel.DisplayedLocationGeoId != newHotels.at(0).DisplayedLocationGeoId) {
                result.HotelLocationUseful = true;
            }
            if (hotel.HasBoyOffers) {
                hasBoyOffers = true;
            }
        }

        auto hotelsAfterSkip = TVector<THotelsResults::THotelResult>();
        if (newHotels.size() > requestData.Skip) {
            hotelsAfterSkip = TVector<THotelsResults::THotelResult>(newHotels.begin() + requestData.Skip, newHotels.end());
        }

        int hotelsOnCurrentPageCount = hotelsAfterSkip.size();

        TMaybe<NTravelProto::NGeoCounter::TIncrementalPollingIterationToken> incrementalPollingNewToken{};

        finishedPrefixLength = finishedPrefixLength > requestData.Skip ? finishedPrefixLength - requestData.Skip : 0;
        if (requestData.IncrementalPollingEnabled) {
            if (hotelsAfterSkip.size() > finishedPrefixLength) {
                hotelsAfterSkip.resize(finishedPrefixLength);
            }
            size_t previousSent = 0;
            incrementalPollingNewToken = NTravelProto::NGeoCounter::TIncrementalPollingIterationToken();
            if (requestData.IncrementalPollingPreviousToken.Defined() && requestData.PollingIteration > 0) { // polling continuation
                previousSent = requestData.IncrementalPollingPreviousToken.GetRef().GetSentHotels();
                auto newHotelsAfterSkip = TVector<THotelsResults::THotelResult>();
                if (hotelsAfterSkip.size() > previousSent) {
                    newHotelsAfterSkip = TVector<THotelsResults::THotelResult>(hotelsAfterSkip.begin() + previousSent, hotelsAfterSkip.end());
                }
                std::swap(newHotelsAfterSkip, hotelsAfterSkip);

                incrementalPollingNewToken->SetSessionId(requestData.IncrementalPollingPreviousToken->GetSessionId());
                incrementalPollingNewToken->SetSearchIteration(requestData.IncrementalPollingPreviousToken->GetSearchIteration());
                incrementalPollingNewToken->SetPollingIteration(requestData.IncrementalPollingPreviousToken->GetPollingIteration() + 1);
                incrementalPollingNewToken->SetHasBoyOffers(requestData.IncrementalPollingPreviousToken->GetHasBoyOffers() || hasBoyOffers);
            } else if (requestData.IncrementalPollingPreviousToken.Defined()) { // new search (new polling)
                incrementalPollingNewToken->SetSessionId(requestData.IncrementalPollingPreviousToken->GetSessionId());
                incrementalPollingNewToken->SetSearchIteration(requestData.IncrementalPollingPreviousToken->GetSearchIteration() + 1);
                incrementalPollingNewToken->SetPollingIteration(0);
                incrementalPollingNewToken->SetHasBoyOffers(hasBoyOffers);
            } else { // new session
                incrementalPollingNewToken->SetSessionId(CreateUid());
                incrementalPollingNewToken->SetSearchIteration(0);
                incrementalPollingNewToken->SetPollingIteration(0);
                incrementalPollingNewToken->SetHasBoyOffers(hasBoyOffers);
            }
            incrementalPollingNewToken->SetSentHotels(previousSent + hotelsAfterSkip.size());
            incrementalPollingNewToken->SetSearchParamsHash(CalcSearchParamsHash(filledBoundingBox,
                                                                                 filledTravelGeoId,
                                                                                 filledOfferSearchParams,
                                                                                 sortType.SortType,
                                                                                 requestData.Skip,
                                                                                 requestData.Limit));
            ConvertCryptaInfoToProto(userCryptaSegments, incrementalPollingNewToken->MutableCryptaInfo());
        } else {
            if (!allFinished) {
                hotelsAfterSkip.clear();
            }
        }

        result.Hotels = hotelsAfterSkip;
        result.IsPollingFinished = allFinished;

        return BuildProtoResp(result, requestData, filledBoundingBox, filledOfferSearchParams, sortType,
                              hotelsOnCurrentPageCount, haveMoreHotels, filledTravelGeoId, incrementalPollingNewToken,
                              newHotels.size(), userCryptaSegments, hasBoyOffers, sortType.SortType == NTravelProto::NGeoCounter::ESortType::ST_ByDistance);
    }

    TTravelGeoId THotelSearchService::DetermineTravelGeoId(const TString& logId, TMaybe<TTravelGeoId> travelGeoId, TMaybe<TPermalink> topHotelPermalink) const {
        if (travelGeoId.Defined()) {
            return travelGeoId.GetRef();
        }
        Y_ENSURE(topHotelPermalink.Defined(), "topHotelPermalink is expected to be present when there is no geoId");
        auto permalinkInfo = Index_.GetExtraPermalinkInfo(topHotelPermalink.GetRef());
        if (permalinkInfo.Defined()) {
            return permalinkInfo.GetRef().GeoId;
        }
        ERROR_LOG << logId << ": Can't find geoId for TopPermalink: " << topHotelPermalink.GetRef() << Endl;
        return DefaultGeoId;
    }

    TBoundingBox THotelSearchService::DetermineBbox(const TString& logId,
                                                    TTravelGeoId travelGeoId,
                                                    TMaybe<TPermalink> topHotelPermalink,
                                                    const TOfferSearchParams& filledOfferSearchParams,
                                                    const TVector<TBasicFilterGroup>& selectedFiltersGroups) const {
        auto bbox = RegionsService_.GetRegionBboxByTravelGeoId(travelGeoId);
        if (!bbox.Defined()) {
            // fallback
            ERROR_LOG << logId << ": Can't find bbox for travelGeoId: " << travelGeoId << Endl;
            Counters_.BBoxNotFoundForGeoId.Inc();
            bbox = RegionsService_.GetRegionBboxByTravelGeoId(DefaultGeoId);
            if (bbox.Defined()) {
                return bbox.GetRef();
            }
            return TBoundingBox(TPosition(55.504611098091075, 37.10190082500168), TPosition(55.95562073628418, 38.19778705547044)); // Moscow
        }

        if (topHotelPermalink.Defined()) {
            auto permalinkInfo = Index_.GetPermalinkFilteringInfo(topHotelPermalink.GetRef());
            if (permalinkInfo.Defined()) {
                if (bbox->GetExtendedByRelativeValue(0.5).Contains(permalinkInfo->StaticPermalinkInfo.Position)) { // 50% to each side
                    // Extending bbox only if topHotelPermalink is near it
                    bbox = bbox.GetRef().GetExtended(permalinkInfo->StaticPermalinkInfo.Position).GetExtendedByRelativeValue(0.05);
                }
            }
        }

        auto initialCount = Index_.GetCounts(bbox.GetRef(), filledOfferSearchParams.BookingRange, selectedFiltersGroups,
                                             TVector<TAdditionalFilter>(), false, true).MatchedCount;
        if (initialCount < 10) {
            int prevCount = -1;
            TBoundingBox prevBbox = TBoundingBox(TPosition(0, 0), TPosition(0, 0));
            for (auto relativeDiff : {1.0, 0.75, 0.5, 0.25, 0.0}) {
                auto currBbox = bbox->GetExtendedByRelativeValue(relativeDiff);
                auto currCount = Index_.GetCounts(currBbox, filledOfferSearchParams.BookingRange,
                                                  selectedFiltersGroups, TVector<TAdditionalFilter>(), false, true).MatchedCount;
                if (prevCount == -1) {
                    prevCount = currCount;
                }
                if (prevCount - currCount > 0.3 * prevCount) { // We want to lose not more than 30% of hotels per step;
                    bbox = prevBbox;
                    break;
                }
                prevCount = currCount;
                prevBbox = currBbox;
            }
        }
        return bbox.GetRef();
    }

    TOfferSearchParams THotelSearchService::AdjustDates(const TInstant& now, const TOfferSearchParams& offerSearchParams) const {
        int nights = offerSearchParams.BookingRange.CheckOutDate - offerSearchParams.BookingRange.CheckInDate;
        NOrdinalDate::TOrdinalDate checkInDate = Max(NOrdinalDate::FromInstant(now), offerSearchParams.BookingRange.CheckInDate);
        return TOfferSearchParams{
            TBookingRange(checkInDate, checkInDate + nights),
            offerSearchParams.Ages
        };
    }

    TOfferSearchParams THotelSearchService::DetermineOfferSearchParams(const TInstant& now, bool hasMirFilter, const NTravelProto::NPromoService::TMirPromoParams& mirPromoParams) const {
        int defaultNights = 1;
        NOrdinalDate::TOrdinalDate minDate = NOrdinalDate::FromInstant(now) + 1;

        if (hasMirFilter && mirPromoParams.GetActive()) {
            defaultNights = mirPromoParams.GetMinNights();
            minDate = Max(minDate, NOrdinalDate::FromString(mirPromoParams.GetFirstCheckIn()));
        }

        return TOfferSearchParams{
            TBookingRange(minDate, minDate + defaultNights),
            TAges::FromOccupancyString("2")
        };
    }

    std::tuple<bool, size_t, TVector<THotelsResults::THotelResult>> THotelSearchService::BuildHotelsWithOcData(const TString& logId,
                                                                                                               const TVector<THotelsResults::THotelResult>& hotels,
                                                                                                               const NTravelProto::NOfferCache::NApi::TReadResp& ocResp,
                                                                                                               bool allowHotelsWithoutPrices,
                                                                                                               size_t limit) const {
        TVector<THotelsResults::THotelResult> newHotels;

        size_t finishedPrefixLength = 0;
        bool allFinished = true;
        for (const auto& hotel: hotels) {
            auto ocHotelRespIt = ocResp.GetHotels().find(hotel.Permalink);
            if (ocHotelRespIt == ocResp.GetHotels().end()) {
                ERROR_LOG << "THotelSearchService::SearchHotels(" << logId << "): No offercache result for " << hotel.Permalink << Endl;
                Counters_.NNoOfferCacheData.Inc();
                continue;
            }
            const auto& ocHotelResp = ocHotelRespIt->second;
            if (!ocHotelResp.GetIsFinished()) {
                allFinished = false;
                continue;
            }
            if (allFinished) {
                finishedPrefixLength++;
            }
            if (!allowHotelsWithoutPrices && ocHotelResp.PricesSize() == 0 && !hotel.IsTopHotel) {
                continue;
            }

            auto& newHotel = newHotels.emplace_back(hotel);

            if (ocHotelResp.PricesSize() > 0) {
                newHotel.OfferInfo = THotelsResults::TOfferInfo();
                auto& offerInfo = newHotel.OfferInfo.GetRef();
                const auto& firstPrice = ocHotelResp.GetPrices(0);
                offerInfo.Price = firstPrice.GetPrice();
                offerInfo.CancellationInfo = THotelsResults::TCancellationInfo{
                    firstPrice.GetFreeCancellation(),
                    firstPrice.GetRefundType(),
                    {firstPrice.GetRefundRule().begin(), firstPrice.GetRefundRule().end()}
                };
                offerInfo.PansionInfo = THotelsResults::TPansionInfo{
                    static_cast<NTravelProto::EPansionType>(firstPrice.GetPansion()),
                    ocResp.GetPansions().at(NTravelProto::NOfferCache::NApi::TOCPansion::EPansion_Name(firstPrice.GetPansion())).GetName()
                };
                if (firstPrice.HasStrikethroughPrice()) {
                    offerInfo.StrikethroughPrice = firstPrice.GetStrikethroughPrice();
                }
                if (firstPrice.HasYandexPlusInfo()) {
                    offerInfo.YandexPlusInfo = firstPrice.GetYandexPlusInfo();
                }
            }

            std::transform(ocHotelResp.GetBadges().begin(), ocHotelResp.GetBadges().end(), std::back_inserter(newHotel.Badges), [](const NTravelProto::NOfferCache::NApi::TBadge& badge) {
                auto hotelBadge = THotelsResults::THotelBadge{};
                hotelBadge.Id = badge.GetId();
                hotelBadge.Text = badge.GetText();
                hotelBadge.Style = badge.GetStyle();
                if (badge.HasAdditionalInfo()) {
                    hotelBadge.AdditionalInfo = badge.GetAdditionalInfo();
                }
                hotelBadge.Theme = badge.GetTheme();
                return hotelBadge;
            });

            newHotel.IsPollingFinished = ocHotelResp.GetIsFinished();
            newHotel.IsPlusAvailable = ocHotelResp.GetIsPlusAvailable();
            newHotel.HasBoyOffers = ocHotelResp.GetHasBoyOffers();

            if (newHotels.size() == limit) {
                break;
            }
        }
        return {allFinished, finishedPrefixLength, newHotels};
    }

    NTravelProto::NOfferCache::NApi::TReadResp THotelSearchService::PrepareAndSendOfferCacheRequest(const THotelSearchService::TSearchHotelsParsedRequestData& requestData,
                                                                                                    const THotelsResults& hotelsResults,
                                                                                                    const TVector<TBasicFilterGroup>& selectedFiltersGroups,
                                                                                                    const TVector<std::unique_ptr<TFilterBase>>& offerBusDataFilters,
                                                                                                    TBookingRange bookingRange,
                                                                                                    const TAges& ages) const {
        auto ocReq = NTravelProto::NOfferCache::NApi::TReadReq();

        for (const auto& hotel: hotelsResults.Hotels) {
            auto pbHotelId = ocReq.AddHotelId();
            pbHotelId->SetPermalink(hotel.Permalink);
        }

        ocReq.SetUseSearcher(requestData.UseSearcher);
        ocReq.SetRequestId(0);
        ocReq.SetSortOffersUsingPlus(requestData.RawReq->GetSortOffersUsingPlus());
        ocReq.SetAllowMobileRates(requestData.RawReq->GetAllowMobileRates());

        ocReq.MutableAttribution()->SetGeoOrigin(requestData.RawReq->GetGeoOrigin());
        ocReq.MutableAttribution()->SetGeoClientId(requestData.RawReq->GetGeoClientId());
        ocReq.MutableExp()->CopyFrom(requestData.RawReq->GetExperiments());
        ocReq.MutableStrExp()->CopyFrom(requestData.RawReq->GetStrExperiments());
        ocReq.MutableKVExperiments()->CopyFrom(requestData.RawReq->GetKVExperiments());

        ocReq.SetCheckInDate(NOrdinalDate::ToString(bookingRange.CheckInDate));
        ocReq.SetCheckOutDate(NOrdinalDate::ToString(bookingRange.CheckOutDate));
        ocReq.SetAges(ages.ToAgesString());

        ocReq.SetRespMode(NTravelProto::NOfferCache::NApi::ERespMode::RM_MultiHotel);
        ocReq.SetShowAllOperators(true);
        ocReq.SetShowAllPansions(true);
        ocReq.SetShowAllBadges(true);
        ocReq.SetCompactResponseForSearch(true);
        ocReq.SetFull(false);
        ocReq.SetAllowPastDates(true);
        ocReq.SetAllowRestrictedUserRates(requestData.RawReq->GetAllowRestrictedUserRates());
        ocReq.SetOnlyBoYWhenBoYAvailable(requestData.RawReq->GetOnlyBoYWhenBoYAvailable());

        for (int banOpId: requestData.RawReq->GetBanOpId()) {
            ocReq.AddBanOpId(banOpId);
        }

        auto partnerFilterGroup = FindFilterGroupById(selectedFiltersGroups, StringEncoder_.Encode(HotelPartnerFeatureBusinessId));
        if (partnerFilterGroup) {
            THashSet<int> partnerIds;
            for (const auto& filter: partnerFilterGroup->Filters) {
                auto filterIntersect = dynamic_cast<TBasicFilterIntersect*>(filter.get());
                Y_ENSURE(filterIntersect, "Expected TBasicFilterIntersect for " + HotelPartnerFeatureBusinessId);

                for (int value: filterIntersect->Values) {
                    NTravelProto::EPartnerId partnerId;
                    Y_ENSURE(NTravelProto::EPartnerId_Parse(StringEncoder_.Decode(value), &partnerId), "Failed to parse partner id: " + StringEncoder_.Decode(value));
                    partnerIds.insert(partnerId);
                }
            }
            for (auto partnerId: partnerIds) {
                ocReq.AddFilterPartnerIdAfterBoYSelection(partnerId);
            }
        }

        if (FindFilterGroupById(selectedFiltersGroups, StringEncoder_.Encode(onlyBoYOffers))) {
            ocReq.SetOnlyBoYOffers(true);
        }

        if (FindFirstMatchingOfferBusFilter(offerBusDataFilters, [](TFilterByOfferBusData* filter) { return filter->YandexOffers; })) {
            ocReq.SetBumpMirOffers(true);
            ocReq.SetBumpBoYOffers(true);
            ocReq.SetFilterRequireBoYOffer(true);
        }
        if (FindFirstMatchingOfferBusFilter(offerBusDataFilters, [](TFilterByOfferBusData* filter) { return filter->HasMirOffers.GetOrElse(false); })) {
            ocReq.SetBumpMirOffers(true);
            ocReq.SetFilterRequireMirOffer(true);
        }

        auto pansionFilter = FindFirstMatchingOfferBusFilter(offerBusDataFilters, [](TFilterByOfferBusData* filter) { return filter->PansionAliases.Defined(); });
        if (pansionFilter) {
            for (int pansionAlias: pansionFilter->PansionAliases.GetRef()) {
                ocReq.AddFilterPansionAlias(PansionAliasesMapping.at(StringEncoder_.Decode(pansionAlias)));
            }
        }

        auto priceFilter = FindFirstMatchingOfferBusFilter(offerBusDataFilters, [](TFilterByOfferBusData* filter) { return !filter->TotalPrice.IsTrivial(); });
        if (priceFilter) {
            if (priceFilter->TotalPrice.MinPrice != NTravel::NGeoCounter::TPriceRange::MAX_RANGE.MinPrice) {
                ocReq.SetFilterPriceFrom(priceFilter->TotalPrice.MinPrice);
            }
            if (priceFilter->TotalPrice.MaxPrice != NTravel::NGeoCounter::TPriceRange::MAX_RANGE.MaxPrice) {
                ocReq.SetFilterPriceTo(priceFilter->TotalPrice.MaxPrice);
            }
        }

        auto freeCancellationFilter = FindFirstMatchingOfferBusFilter(offerBusDataFilters, [](TFilterByOfferBusData* filter) { return filter->FreeCancellation.GetOrElse(false); });
        if (freeCancellationFilter) {
            if (freeCancellationFilter->FreeCancellation.GetRef()) {
                ocReq.SetFilterFreeCancellation(true);
            }
        }

        ocReq.SetRobotRequest(requestData.RawReq->GetRobotRequest());
        ocReq.MutableAttribution()->CopyFrom(requestData.RawReq->GetAttribution());
        ocReq.MutableAttribution()->SetGeoOrigin(requestData.RawReq->GetGeoOrigin());
        ocReq.MutableAttribution()->SetGeoClientId(requestData.RawReq->GetGeoClientId());

        ocReq.SetWhiteLabelPartnerId(requestData.RawReq->GetWhiteLabelPartnerId());

        return OfferCacheClient_.Read(ocReq, requestData.LogId);
    }

    NTravelProto::NGeoCounter::TGetHotelsResponse THotelSearchService::BuildProtoResp(const THotelsResults& hotelsResults,
                                                                                      const THotelSearchService::TSearchHotelsParsedRequestData& requestData,
                                                                                      const TBoundingBox& bbox,
                                                                                      const TOfferSearchParams& offerSearchParams,
                                                                                      const TSortType& sortType,
                                                                                      int hotelsOnCurrentPageCount,
                                                                                      bool haveMoreHotels,
                                                                                      int filledGeoId,
                                                                                      const TMaybe<NTravelProto::NGeoCounter::TIncrementalPollingIterationToken>& pollingToken,
                                                                                      int totalHotelsCount,
                                                                                      const TCryptaSegments& userCryptaSegments,
                                                                                      bool hasBoyOffers,
                                                                                      bool sortByDistance) const {
        NTravelProto::NGeoCounter::TGetHotelsResponse resp;
        auto pbHotels = resp.MutableHotels();
        for (const auto& hotel: hotelsResults.Hotels) {
            auto pbHotel = pbHotels->AddHotels();
            auto pbLogReadlTimeRanking = pbHotels->MutableLogData()->MutableRealTimeRankingInfo()->AddHotelInfo();
            pbHotel->SetPermalink(hotel.Permalink);
            pbHotel->SetName(hotel.Name);
            pbHotel->MutableRubric()->SetId(hotel.Rubric.Id);
            pbHotel->MutableRubric()->SetName(hotel.Rubric.Name);
            pbHotel->MutableCoordinates()->SetLat(hotel.Coordinates.Lat);
            pbHotel->MutableCoordinates()->SetLon(hotel.Coordinates.Lon);
            pbHotel->SetAddress(hotel.Address);
            pbHotel->SetStars(hotel.Stars.GetOrElse(0));
            pbHotel->SetRating(hotel.Rating.GetOrElse(0));
            pbHotel->SetTotalImageCount(hotel.ImageCount);
            if (hotel.DisplayedLocationGeoId > 0) {
                pbHotel->SetDisplayedLocationGeoId(hotel.DisplayedLocationGeoId);
            }
            for (const auto& feature: hotel.Features) {
                auto pbFeature = pbHotel->AddFeatures();
                pbFeature->SetId(feature.Id);
                pbFeature->SetName(feature.Name);
                if (feature.BooleanValue.Defined()) {
                    pbFeature->SetBooleanValue(feature.BooleanValue.GetRef());
                } else if (feature.DoubleValue.Defined()) {
                    pbFeature->SetNumericValue(feature.DoubleValue.GetRef());
                } else if (feature.StringValues.Defined()) {
                    for (const auto& value: feature.StringValues.GetRef()) {
                        auto mutableTextValue = pbFeature->MutableTextValues()->AddValues();
                        mutableTextValue->SetId(value.Id);
                        mutableTextValue->SetName(value.Name);
                    }
                } else {
                    ythrow yexception() << "Can't convert feature with id: " << feature.Id;
                }
            }
            if (hotel.OfferInfo.Defined()) {
                auto pbOfferInfo = pbHotel->MutableOfferInfo();
                pbOfferInfo->SetPrice(hotel.OfferInfo.GetRef().Price);

                pbOfferInfo->MutablePansionInfo()->SetPansionType(hotel.OfferInfo->PansionInfo.PansionType);
                pbOfferInfo->MutablePansionInfo()->SetPansionName(hotel.OfferInfo->PansionInfo.PansionName);

                pbOfferInfo->MutableCancellationInfo()->SetHasFreeCancellation(hotel.OfferInfo->CancellationInfo.HasFreeCancellation);
                pbOfferInfo->MutableCancellationInfo()->SetRefundType(hotel.OfferInfo->CancellationInfo.RefundType);
                for (const auto& refundRule: hotel.OfferInfo->CancellationInfo.RefundRules) {
                    pbOfferInfo->MutableCancellationInfo()->AddRefundRules()->CopyFrom(refundRule);
                }
                if (hotel.OfferInfo->StrikethroughPrice.Defined()) {
                    pbOfferInfo->MutableStrikethroughPrice()->CopyFrom(*hotel.OfferInfo->StrikethroughPrice);
                }
                if (hotel.OfferInfo->YandexPlusInfo.Defined()) {
                    pbOfferInfo->MutableYandexPlusInfo()->CopyFrom(*hotel.OfferInfo->YandexPlusInfo);
                }
            }
            for (const THotelsResults::THotelBadge& badge: hotel.Badges) {
                auto pbBadge = pbHotel->AddBadges();
                pbBadge->SetId(badge.Id);
                pbBadge->SetText(badge.Text);
                pbBadge->SetStyle(badge.Style);
                if (badge.AdditionalInfo.Defined()) {
                    pbBadge->MutableAdditionalInfo()->CopyFrom(badge.AdditionalInfo.GetRef());
                }
                pbBadge->SetTheme(badge.Theme);
            }
            pbHotel->SetPollingFinished(hotel.IsPollingFinished);
            pbHotel->SetIsPlusAvailable(hotel.IsPlusAvailable);
            pbHotel->SetIsTopHotel(hotel.IsTopHotel);

            if (sortByDistance) {
                Y_ENSURE(requestData.SortOrigin.Defined(), "SortOrigin is required for distance");
                pbHotel->SetDistanceMeters(GetDistanceMeters(hotel.Coordinates, requestData.SortOrigin.GetRef()));
            }

            pbLogReadlTimeRanking->SetPermalink(hotel.Permalink);
            pbLogReadlTimeRanking->SetCatBoostUsed(hotel.RealTimeRankingInfo.CatBoostUsed);
            pbLogReadlTimeRanking->SetCatBoostPrediction(hotel.RealTimeRankingInfo.CatBoostPrediction);
        }
        pbHotels->SetPollingFinished(hotelsResults.IsPollingFinished);

        pbHotels->MutableHotelResultCounts()->SetHotelsOnCurrentPageCount(hotelsOnCurrentPageCount);
        pbHotels->MutableHotelResultCounts()->SetHaveMoreHotels(haveMoreHotels);
        pbHotels->MutableHotelResultCounts()->SetTotalHotelsCount(totalHotelsCount);

        pbHotels->MutableSearchParams()->MutableBbox()->MutableLowerLeft()->SetLat(bbox.LowerLeft.Lat);
        pbHotels->MutableSearchParams()->MutableBbox()->MutableLowerLeft()->SetLon(bbox.LowerLeft.Lon);
        pbHotels->MutableSearchParams()->MutableBbox()->MutableUpperRight()->SetLat(bbox.UpperRight.Lat);
        pbHotels->MutableSearchParams()->MutableBbox()->MutableUpperRight()->SetLon(bbox.UpperRight.Lon);
        pbHotels->MutableSearchParams()->SetGeoId(filledGeoId);
        pbHotels->MutableSearchParams()->SetCheckInDate(NOrdinalDate::ToString(offerSearchParams.BookingRange.CheckInDate));
        pbHotels->MutableSearchParams()->SetCheckOutDate(NOrdinalDate::ToString(offerSearchParams.BookingRange.CheckOutDate));
        pbHotels->MutableSearchParams()->SetAges(offerSearchParams.Ages.ToAgesString());
        pbHotels->MutableSearchParams()->SetSortType(sortType.SortType);
        if (sortByDistance) {
            Y_ENSURE(requestData.SortOrigin.Defined(), "SortOrigin is required for sort by distance");
            pbHotels->MutableSearchParams()->MutableSortOrigin()->SetLat(requestData.SortOrigin.GetRef().Lat);
            pbHotels->MutableSearchParams()->MutableSortOrigin()->SetLon(requestData.SortOrigin.GetRef().Lon);
        }
        pbHotels->MutableAggregatedHotelData()->SetHotelLocationUseful(hotelsResults.HotelLocationUseful);
        pbHotels->MutableAggregatedHotelData()->SetHasBoyOffers(hasBoyOffers);

        if (pollingToken.Defined()) {
            pbHotels->MutableIncrementalPollingData()->SetPollingIterationToken(EncodePollingToken(pollingToken.GetRef()));
            pbHotels->MutableIncrementalPollingData()->SetPollingId(EncodePollingId(pollingToken.GetRef()));
        }

        pbHotels->MutableDebugData()->SetRequestId(requestData.ReqId);
        if (pollingToken.Defined()) {
            pbHotels->MutableDebugData()->SetSessionId(pollingToken->GetSessionId());
        }
        if (!userCryptaSegments.IsEmpty()) {
            ConvertCryptaInfoToProto(userCryptaSegments, pbHotels->MutableDebugData()->MutableDetectedUserCryptaInfo());
            pbHotels->MutableExperimentalData()->SetCryptaDataAvailable(true);
            ConvertSegmentsToLogProto(userCryptaSegments, pbHotels->MutableLogData()->MutableUserSegmentsInfo());
        } else {
            pbHotels->MutableExperimentalData()->SetCryptaDataAvailable(false);
        }
        auto sortInfo = pbHotels->MutableDebugData()->MutableSelectedSortInfo();
        sortInfo->SetSortType(sortType.SortType);
        sortInfo->SetSortRepr(sortType.GetStringRepr());

        return resp;
    }

    TCryptaSegments THotelSearchService::DetermineUserCryptaSegments(const THotelSearchService::TSearchHotelsParsedRequestData& requestData) const {
        if (requestData.RawReq->GetDebugData().HasForceCryptaInfo()) {
            return ConvertCryptaInfoFromProto(requestData.RawReq->GetDebugData().GetForceCryptaInfo());
        }

        auto request = TBigbClient::TRequest{
            requestData.RawReq->GetAttribution().GetYandexUid(),
            requestData.RawReq->GetAttribution().GetPassportUid()
        };

        auto [isValidRequest, validationError] = BigbClient_.IsValidRequest(request);
        if (!isValidRequest) {
            DEBUG_LOG << "Invalid crypta request data: " << validationError << Endl;
            return TCryptaSegments();
        }

        yabs::proto::Profile profile;
        try {
            profile = BigbClient_.GetProfile(request);
        } catch (...) {
            Counters_.FailedToGetCryptaProfile.Inc();
            ERROR_LOG << "Failed to get crypta user profile: " << CurrentExceptionMessage() << Endl;
            return TCryptaSegments();
        }
        auto foundSocdem = false;
        TCryptaSegments result{};
        for (const auto& item: profile.get_arr_items()) {
            if (item.keyword_id() == 569) {
                for (const auto& pair: item.get_arr_pair_values()) {
                    foundSocdem = true;
                    for (const auto& keywordId: {TUserSegmentsRegistry::GenderKeywordId,
                                                 TUserSegmentsRegistry::AgeKeywordId,
                                                 TUserSegmentsRegistry::IncomeKeywordId}) {
                        if (pair.first() == static_cast<ui64>(keywordId)) {
                            auto currKeywordAndSegment = UserSegmentsRegistry_.GetSegmentById(pair.first(), pair.second());
                            if (currKeywordAndSegment.Defined()) {
                                result.SegmentValues.emplace(currKeywordAndSegment->Keyword.KeywordId, currKeywordAndSegment->Segment.SegmentId);
                                break;
                            }
                            break;
                        }
                    }
                }
                continue;
            }

            auto keyword = UserSegmentsRegistry_.GetKeywordById(item.keyword_id());
            if (keyword.Empty()) {
                WARNING_LOG << "Unknown segment in bigb response for user yandex_uid=" << request.YandexUid << ", keyword_id=" << item.keyword_id() << Endl;
                continue;
            }

            if (keyword->KeywordType == TUserSegmentsRegistry::EKeywordType::KT_WEIGHTED_UINT_VALUES ||
                keyword->KeywordType == TUserSegmentsRegistry::EKeywordType::KT_WEIGHTED_UINT_WITH_SELECTED_VALUES) {
                for (const auto& pair: item.get_arr_weighted_uint_values()) {
                    auto segment = UserSegmentsRegistry_.GetSegmentById(item.keyword_id(), pair.first());
                    if (segment.Empty()) {
                        continue;
                    }
                    result.WeightedSegmentValues.emplace(std::make_pair(segment->Keyword.KeywordId, segment->Segment.SegmentId), pair.weight());
                }
            } else if (keyword->KeywordType == TUserSegmentsRegistry::EKeywordType::KT_UINT_VALUES) {
                for (const auto& value: item.get_arr_uint_values()) {
                    auto segment = UserSegmentsRegistry_.GetSegmentById(item.keyword_id(), value);
                    if (segment.Empty()) {
                        continue;
                    }
                    result.SegmentValues.emplace(segment->Keyword.KeywordId, segment->Segment.SegmentId);
                }
            } else if (keyword->KeywordType == TUserSegmentsRegistry::EKeywordType::KT_IGNORED) {
                DEBUG_LOG << "Keyword ignored yandex_uid=" << request.YandexUid << ", keyword_id=" << item.keyword_id() << Endl;
            } else {
                WARNING_LOG << "Unknown segment type in bigb response for user yandex_uid=" << request.YandexUid << ", keyword_id=" << item.keyword_id() << Endl;
            }
        }

        if (!foundSocdem) {
            Counters_.FailedToGetCryptaSocdem.Inc();
            INFO_LOG << "No exact socdem in crypta response for user yandex_uid=" << request.YandexUid << Endl;
        }

        if (result.IsEmpty()) {
            Counters_.EmptyBigbSegments.Inc();
            INFO_LOG << "No data in bigb response for user yandex_uid=" << request.YandexUid << Endl;
        }

        return result;
    }

    void THotelSearchService::ConvertCryptaInfoToProto(const TCryptaSegments& userCryptaSegments, NTravelProto::NGeoCounter::TCryptaInfo* cryptaInfo) const {
        for (const auto& [keywordId, segmentId]: userCryptaSegments.SegmentValues) {
            auto segment = cryptaInfo->AddOldSegments();
            auto currSegment = UserSegmentsRegistry_.GetSegmentByIdOrFail(keywordId, segmentId);
            segment->SetName(currSegment.Keyword.KeywordName);
            segment->SetValue(currSegment.Segment.SegmentName);
        }
        for (const auto& [keywordId, segmentId]: userCryptaSegments.SegmentValues) {
            auto segment = cryptaInfo->AddSegments();
            segment->SetKeywordId(keywordId);
            segment->SetSegmentId(segmentId);
        }
        for (const auto& [pair, weight]: userCryptaSegments.WeightedSegmentValues) {
            auto segment = cryptaInfo->AddWeightedSegments();
            segment->SetKeywordId(pair.first);
            segment->SetSegmentId(pair.second);
            segment->SetWeight(weight);
        }
    }

    TCryptaSegments THotelSearchService::ConvertCryptaInfoFromProto(const NTravelProto::NGeoCounter::TCryptaInfo& cryptaInfo) const {
        TCryptaSegments cryptaSegments{};
        if (cryptaInfo.SegmentsSize() > 0 || cryptaInfo.WeightedSegmentsSize() > 0) {
            for (const auto& segment: cryptaInfo.GetSegments()) {
                cryptaSegments.SegmentValues.emplace(segment.GetKeywordId(), segment.GetSegmentId());
            }
            for (const auto& segment: cryptaInfo.GetWeightedSegments()) {
                cryptaSegments.WeightedSegmentValues.emplace(std::make_pair(segment.GetKeywordId(), segment.GetSegmentId()), segment.GetWeight());
            }
        } else {
            for (const auto& segment: cryptaInfo.GetOldSegments()) {
                auto currSegment = UserSegmentsRegistry_.GetSegmentByName(segment.GetName(), segment.GetValue());
                if (currSegment.Defined()) {
                    cryptaSegments.SegmentValues.emplace(currSegment.GetRef().Keyword.KeywordId, currSegment.GetRef().Segment.SegmentId);
                }
            }
        }
        return cryptaSegments;
    }

    void THotelSearchService::ConvertSegmentsToLogProto(const TCryptaSegments& userCryptaSegments,
                                                        NTravelProto::NGeoCounter::TGetHotelsResponse::THotels::TLogData::TUserSegmentsInfo* userSegmentsInfo) const {
        for (const auto& [keywordId, segmentId]: userCryptaSegments.SegmentValues) {
            auto pbSegment = userSegmentsInfo->AddSegments();
            auto segment = UserSegmentsRegistry_.GetSegmentByIdOrFail(keywordId, segmentId);
            pbSegment->SetKeywordId(ToString(segment.Keyword.KeywordId));
            pbSegment->SetKeywordName(segment.Keyword.KeywordName);
            pbSegment->SetSegmentId(ToString(segment.Segment.SegmentId));
            pbSegment->SetSegmentName(segment.Segment.SegmentName);
            pbSegment->SetSegmentType(NTravelProto::NGeoCounter::EUserSegmentType::UST_NoValue);
        }
        for (const auto& [pair, weight]: userCryptaSegments.WeightedSegmentValues) {
            auto pbSegment = userSegmentsInfo->AddSegments();
            auto segment = UserSegmentsRegistry_.GetSegmentByIdOrFail(pair.first, pair.second);
            pbSegment->SetKeywordId(ToString(segment.Keyword.KeywordId));
            pbSegment->SetKeywordName(segment.Keyword.KeywordName);
            pbSegment->SetSegmentId(ToString(segment.Segment.SegmentId));
            pbSegment->SetSegmentName(segment.Segment.SegmentName);
            pbSegment->SetSegmentType(NTravelProto::NGeoCounter::EUserSegmentType::UST_Weighted);
            pbSegment->SetWeight(weight);
        }
    }

    static std::initializer_list<int> g_LatencyBuckets = {0, 1, 5, 10, 20, 50, 100, 250, 500, 1000};

    THotelSearchService::TCounters::TCounters()
        : ReorderTopRecordsTimeSmall(g_LatencyBuckets)
        , ReorderTopRecordsTimeMedium(g_LatencyBuckets)
        , ReorderTopRecordsTimeLarge(g_LatencyBuckets)
        , ReorderTopRecordsTimeTotal(g_LatencyBuckets)
    {
    }

    void THotelSearchService::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
        ct->insert(MAKE_COUNTER_PAIR(NNoOfferCacheData));
        ct->insert(MAKE_COUNTER_PAIR(BBoxNotFoundForGeoId));
        ct->insert(MAKE_COUNTER_PAIR(FailedToGetCryptaProfile));
        ct->insert(MAKE_COUNTER_PAIR(FailedToGetCryptaSocdem));
        ct->insert(MAKE_COUNTER_PAIR(EmptyBigbSegments));

        ReorderTopRecordsTimeSmall.QueryCounters("ReorderTopRecordsTimeSmall", "", ct);
        ReorderTopRecordsTimeMedium.QueryCounters("ReorderTopRecordsTimeMedium", "", ct);
        ReorderTopRecordsTimeLarge.QueryCounters("ReorderTopRecordsTimeLarge", "", ct);
        ReorderTopRecordsTimeTotal.QueryCounters("ReorderTopRecordsTimeTotal", "", ct);
    }
}
