#pragma once

#include "data.h"
#include "features_filtering.h"
#include "offercache_client.h"
#include "index.h"
#include "promo_service_client.h"
#include "regions_service.h"
#include "bigb_client.h"

#include <travel/hotels/geocounter/proto/incremental_polling_data.pb.h>

#include <travel/hotels/proto/geocounter_service/geocounter_service.pb.h>

namespace NTravel::NGeoCounter {
    class THotelSearchService {
    public:
        class invalid_polling_token_exception: public yexception {
        };

        struct TSearchHotelsParsedRequestData {
            const NTravelProto::NGeoCounter::TGetHotelsRequest* RawReq;
            TMaybe<TBoundingBox> BoundingBox;
            TMaybe<TTravelGeoId> TravelGeoId;
            TVector<TBasicFilterGroup> SelectedFiltersGroups;
            TVector<std::unique_ptr<TFilterBase>> OfferBusDataFilters;
            TMaybe<TOfferSearchParams> OfferSearchParams;
            NTravelProto::NGeoCounter::ESortType ProtoSortType;
            TMaybe<TPosition> SortOrigin;
            size_t Skip;
            size_t Limit;
            bool IncrementalPollingEnabled;
            TMaybe<NTravelProto::NGeoCounter::TIncrementalPollingIterationToken> IncrementalPollingPreviousToken;
            ui32 PollingIteration;
            TString ReqId;
            TString LogId;
            bool UseSearcher;
            TMaybe<TPermalink> TopHotelPermalink;
        };

        THotelSearchService(TOfferCacheClient& offerCacheClient,
                            TPromoServiceClient& promoServiceClient,
                            TIndex& index,
                            TStringEncoder& stringEncoder,
                            TRegionsService& regionsService,
                            const TSortTypeRegistry& sortTypeRegistry,
                            const TFilterRegistry& filterRegistry,
                            const TBigbClient& bigbClient,
                            const TUserSegmentsRegistry& userSegmentsRegistry);

        void RegisterCounters(NMonitor::TCounterSource& source);

        TSearchHotelsParsedRequestData ParseSearchHotels(const NTravelProto::NGeoCounter::TGetHotelsRequest& req,
                                                         const TString& grcpCallId) const;

        NTravelProto::NGeoCounter::TGetHotelsResponse SearchHotels(const TSearchHotelsParsedRequestData& requestData) const;

    private:
        struct TCounters: public NMonitor::TCounterSource {
            TCounters();

            NMonitor::TDerivCounter NNoOfferCacheData;
            NMonitor::TDerivCounter BBoxNotFoundForGeoId;
            NMonitor::TDerivCounter FailedToGetCryptaProfile;
            NMonitor::TDerivCounter FailedToGetCryptaSocdem;
            NMonitor::TDerivCounter EmptyBigbSegments;

            NMonitor::TDerivHistogramCounter ReorderTopRecordsTimeSmall;
            NMonitor::TDerivHistogramCounter ReorderTopRecordsTimeMedium;
            NMonitor::TDerivHistogramCounter ReorderTopRecordsTimeLarge;
            NMonitor::TDerivHistogramCounter ReorderTopRecordsTimeTotal;

            void QueryCounters(NMonitor::TCounterTable* ct) const override;
        };

        TOfferCacheClient& OfferCacheClient_;
        TPromoServiceClient& PromoServiceClient_;
        TIndex& Index_;
        TStringEncoder& StringEncoder_;
        TRegionsService& RegionsService_;
        const TSortTypeRegistry& SortTypeRegistry_;
        const TFilterRegistry& FilterRegistry_;
        const TBigbClient& BigbClient_;
        const TUserSegmentsRegistry& UserSegmentsRegistry_;

        mutable TCounters Counters_;

        TBoundingBox DetermineBbox(const TString& logId,
                                   TTravelGeoId travelGeoId,
                                   TMaybe<TPermalink> topHotelPermalink,
                                   const TOfferSearchParams& filledOfferSearchParams,
                                   const TVector<TBasicFilterGroup>& selectedFiltersGroups) const;
        TTravelGeoId DetermineTravelGeoId(const TString& logId, TMaybe<TTravelGeoId> travelGeoId, TMaybe<TPermalink> topHotelPermalink) const;
        TOfferSearchParams AdjustDates(const TInstant& now, const TOfferSearchParams& offerSearchParams) const;
        TOfferSearchParams DetermineOfferSearchParams(const TInstant& now, bool hasMirFilter, const NTravelProto::NPromoService::TMirPromoParams& mirPromoParams) const;
        NTravelProto::NOfferCache::NApi::TReadResp PrepareAndSendOfferCacheRequest(const THotelSearchService::TSearchHotelsParsedRequestData& requestData,
                                                                                   const THotelsResults& hotelsResults,
                                                                                   const TVector<TBasicFilterGroup>& selectedFiltersGroups,
                                                                                   const TVector<std::unique_ptr<TFilterBase>>& offerBusDataFilters,
                                                                                   TBookingRange bookingRange,
                                                                                   const TAges& ages) const;
        std::tuple<bool, size_t, TVector<THotelsResults::THotelResult>> BuildHotelsWithOcData(const TString& logId,
                                                                                              const TVector<THotelsResults::THotelResult>& hotels,
                                                                                              const NTravelProto::NOfferCache::NApi::TReadResp& ocResp,
                                                                                              bool allowHotelsWithoutPrices,
                                                                                              size_t limit) const;
        NTravelProto::NGeoCounter::TGetHotelsResponse BuildProtoResp(const THotelsResults& hotelsResults,
                                                                     const THotelSearchService::TSearchHotelsParsedRequestData& requestData,
                                                                     const TBoundingBox& bbox,
                                                                     const TOfferSearchParams& offerSearchParams,
                                                                     const TSortType& sortType,
                                                                     int hotelsOnCurrentPageCount,
                                                                     bool haveMoreHotels,
                                                                     int filledGeoId,
                                                                     const TMaybe<NTravelProto::NGeoCounter::TIncrementalPollingIterationToken>& pollingToken,
                                                                     int totalHotelsCount,
                                                                     const TCryptaSegments& userCryptaSegments,
                                                                     bool hasBoyOffers,
                                                                     bool sortByDistance) const;
        TCryptaSegments DetermineUserCryptaSegments(const THotelSearchService::TSearchHotelsParsedRequestData& requestData) const;
        void ConvertCryptaInfoToProto(const TCryptaSegments& userCryptaSegments, NTravelProto::NGeoCounter::TCryptaInfo* cryptaInfo) const;
        TCryptaSegments ConvertCryptaInfoFromProto(const NTravelProto::NGeoCounter::TCryptaInfo& cryptaInfo) const;
        void ConvertSegmentsToLogProto(const TCryptaSegments& userCryptaSegments,
                                       NTravelProto::NGeoCounter::TGetHotelsResponse::THotels::TLogData::TUserSegmentsInfo* userSegmentsInfo) const;
    };
}
