#include "index.h"
#include "filter_registry.h"
#include "utils.h"

#include <travel/hotels/lib/cpp/util/sizes.h>
#include <travel/hotels/proto2/bus_messages.pb.h>
#include <travel/hotels/proto/app_config/string_compressor.pb.h>
#include <travel/hotels/proto/offerbus_messages/bus_messages.pb.h>

#include <catboost/libs/cat_feature/cat_feature.h>
#include <library/cpp/protobuf/json/json2proto.h>
#include <util/generic/queue.h>

namespace {
    const THashMap<i64, TString> RubricNames = {
        {184106414L, "Гостиница"},
        {184106404L, "Санаторий"},
        {184106400L, "Дом Отдыха"},
        {20699506347L, "Хостел"},
        {184106426L, "Турбаза"},
        {184106420L, "Кемпинг"},
        {255921949L, "Отдых на ферме"},
        {150049871970L, "Апартаменты"},
        {197061821387L, "Жильё посуточно"},
    };

    constexpr static size_t IN_INDEX_SORT_FACTORS_SZ_EXPECTATION = 10000;
}

namespace {
    class TBucketRecordsStream {
    public:
        explicit TBucketRecordsStream(const TVector<NTravel::NGeoCounter::TFilteringPermalinkInfo>* bucket,
                                      const TVector<int>* permutation,
                                      NTravel::NGeoCounter::TPosition lowerLeft,
                                      NTravel::NGeoCounter::TPosition upperRight,
                                      TMaybe<NTravel::TPermalink> skippedPermalink,
                                      bool boosted)
            : Bucket(bucket)
            , Permutation(permutation)
            , Bbox(lowerLeft, upperRight)
            , Position(-1)
            , SkippedPermalink(skippedPermalink)
            , Boosted(boosted)
        {
        }

        bool MoveNext() {
            Position++;
            while (Position < static_cast<int>(Bucket->size())) {
                const auto& info = GetCurrent();
                const auto& pos = info.StaticPermalinkInfo.Position;

                if (Bbox.Contains(pos)) {
                    if (!SkippedPermalink.Defined() || info.StaticPermalinkInfo.Permalink != SkippedPermalink.GetRef()) {
                        return true;
                    }
                }
                Position++;
            }
            return false;
        }

        const NTravel::NGeoCounter::TFilteringPermalinkInfo& GetCurrent() const {
            if (Permutation) {
                return Bucket->at(Permutation->at(Position));
            } else {
                return Bucket->at(Position);
            }
        }

        bool IsEnd() const {
            return Position == static_cast<int>(Bucket->size());
        }

        bool IsBoosted() const {
            return Boosted;
        }

    private:
        const TVector<NTravel::NGeoCounter::TFilteringPermalinkInfo>* Bucket;
        const TVector<int>* Permutation;
        NTravel::NGeoCounter::TBoundingBox Bbox;
        int Position;
        TMaybe<NTravel::TPermalink> SkippedPermalink;
        bool Boosted;
    };

    void ProcessStreamsUnordered(const NTravel::NGeoCounter::TStringEncoder& stringEncoder,
                                 TVector<TBucketRecordsStream>* streamsPtr,
                                 size_t* processedCount,
                                 const std::function<bool(const NTravel::NGeoCounter::TFilteringPermalinkInfo&)>& handler) {
        for (auto& stream: *streamsPtr) {
            stream.MoveNext();
            while (!stream.IsEnd()) {
                const auto& info = stream.GetCurrent();
                ++(*processedCount);
                try {
                    if (!handler(info)) {
                        return;
                    }
                } catch (...) {
                    ERROR_LOG << "Failed to process permalink info. PermalinkInfo: " << info.StaticPermalinkInfo.ToDebugString(stringEncoder) << ", Error: "
                              << CurrentExceptionMessage() << Endl;
                }
                stream.MoveNext();
            }
        }
    }

    void ProcessStreams(const NTravel::NGeoCounter::TStringEncoder& stringEncoder,
                        int sortTypeId,
                        TVector<TBucketRecordsStream>* streamsPtr,
                        size_t* processedCount,
                        const std::function<bool(const NTravel::NGeoCounter::TFilteringPermalinkInfo&)>& handler) {
        auto cmp = [sortTypeId](const TBucketRecordsStream* const& lhs, const TBucketRecordsStream* const& rhs) {
            if (rhs->IsEnd()) {
                return false;
            }
            if (lhs->IsEnd()) {
                return true;
            }
            // Boosted streams should be processed first, so they are compared as greatest
            if (lhs->IsBoosted() && !rhs->IsBoosted()) {
                return false;
            }
            if (!lhs->IsBoosted() && rhs->IsBoosted()) {
                return true;
            }
            return lhs->GetCurrent().SortIndices.at(sortTypeId) > rhs->GetCurrent().SortIndices.at(sortTypeId);
        };
        TVector<TBucketRecordsStream*> streams;
        streams.reserve(streamsPtr->size());
        for (auto& stream: *streamsPtr) {
            streams.push_back(&stream);
        }

        for (auto& stream: streams) {
            stream->MoveNext();
        }
        int iterationsLeft = 100 * 1000 * 1000; // We have about 5_000_000 hotels, so 100_000_000 is enough
        std::make_heap(streams.begin(), streams.end(), cmp);
        while (!streams.empty() && !streams.front()->IsEnd()) {
            if (iterationsLeft-- <= 0) {
                break;
            }
            std::pop_heap(streams.begin(), streams.end(), cmp);
            auto& currentStream = streams.back();
            const auto& nextStream = streams.front(); // Can be same as currentStream, it's ok

            // there are rare cases when we have equal streams during rebuild, so using non-strict comparison here
            while (!currentStream->IsEnd() && (streams.size() == 1 || !cmp(currentStream, nextStream))) {
                if (iterationsLeft-- <= 0) {
                    break;
                }
                const auto& info = currentStream->GetCurrent();
                ++(*processedCount);
                try {
                    if (!handler(info)) {
                        return;
                    }
                } catch (...) {
                    ERROR_LOG
                        << "Failed to process permalink info. PermalinkInfo: " << info.StaticPermalinkInfo.ToDebugString(stringEncoder) << ", Error: "
                        << CurrentExceptionMessage()
                        << Endl;
                }
                currentStream->MoveNext();
            }
            std::push_heap(streams.begin(), streams.end(), cmp);
        }
        Y_ENSURE(iterationsLeft > 0, "Too many iterations while processing streams, probably a bug");
    }
}

namespace NTravel::NGeoCounter {

    TIndex::TIndex(
        const NTravelProto::NHotelFiltersConfig::TFilterInfoConfig& filterInfoConfig,
        const NTravelProto::NAppConfig::TYtTableCacheConfig& geoCounterRecordsTableConfig,
        const NTravelProto::NAppConfig::TYtTableCacheConfig& priceRecordsTableConfig,
        const NTravelProto::NAppConfig::TYtTableCacheConfig& hotelTraitsTableConfig,
        const NTravelProto::NAppConfig::TConfigYtQueueReader& offerBusConfig,
        const NTravelProto::NAppConfig::TYtTableCacheConfig& originalIdToPermalinkMapper,
        const NTravelProto::NAppConfig::TConfigStringCompressor& stringCompressorConfig,
        const NTravelProto::NGeoCounter::TConfig::TIndexOptions& indexOptionsConfig,
        const TBoyPartnerProvider& boyPartnerProvider,
        const TSortTypeRegistry& sortTypeRegistry,
        TStringEncoder& stringEncoder)
        : FilterInfoConfig_(filterInfoConfig)
        , StringEncoder_(stringEncoder)
        , FilterRegistry_(StringEncoder_)
        , GeoCounterRecordsTable_("GeoCounterRecords", geoCounterRecordsTableConfig)
        , PricesTable_("PriceRecords", priceRecordsTableConfig)
        , HotelTraitsTable_("HotelTraitsRecords", hotelTraitsTableConfig)
        , OfferBus_(offerBusConfig, "OfferBus")
        , OriginalIdToPermalinkMapper_("OriginalIdToPermalinkMapper", originalIdToPermalinkMapper)
        , BoyPartnerProvider_(boyPartnerProvider)
        , ExtraPermalinkInfoStringCompressor_("ExtraPermalinkInfoStringCompressor", stringCompressorConfig)
        , MaxPredefinedMatchesForDynamicFilters_(indexOptionsConfig.GetMaxPredefinedMatchesForDynamicFilters())
        , MinHotelsWithOfferBusDataForExtrapolation_(indexOptionsConfig.GetMinHotelsWithOfferBusDataForExtrapolation())
        , WaitBucketReleaseDelay_(TDuration::MilliSeconds(indexOptionsConfig.GetWaitBucketReleaseDelayMs()))
        , MaxWaitBucketRelease_(TDuration::MilliSeconds(indexOptionsConfig.GetMaxWaitBucketReleaseMs()))
        , MaxBucketsInReleaseQueue_(indexOptionsConfig.GetMaxBucketsInReleaseQueue())
        , StarsMapping_({
            {"unrated", 0},
            {"one", 1},
            {"two", 2},
            {"three", 3},
            {"four", 4},
            {"five", 5},
        })
        , SortTypeRegistry_(sortTypeRegistry)
    {
        InitBuckets(&Buckets_);

        InitPredefinedFilters();

        InitGeoCounterRecordsTable();
        InitPriceRecordsTable();
        InitHotelTraitsTable();

        InitOfferBus();

        OriginalIdToPermalinkMapper_.SetOnFinishHandler([this]() {
            if (OfferBusStarted_.TrySet()) {
                OfferBus_.Start();
            }
        });
    }

    void TIndex::RegisterCounters(NMonitor::TCounterSource& source) {
        GeoCounterRecordsTable_.RegisterCounters(source, "GeoCounterRecords");
        PricesTable_.RegisterCounters(source, "PriceRecords");
        HotelTraitsTable_.RegisterCounters(source, "HotelTraitsRecords");
        OfferBus_.RegisterCounters(source);
        OriginalIdToPermalinkMapper_.RegisterCounters(source);
        ExtraPermalinkInfoStringCompressor_.RegisterCounters(source);
        source.RegisterSource(&Counters_, "GeoCounterIndex");
    }

    void TIndex::Start() {
        Counters_.IsWaiting = 1;
        GeoCounterRecordsTable_.Start();
        PricesTable_.Start();
        HotelTraitsTable_.Start();
        OriginalIdToPermalinkMapper_.Start();
        ExtraPermalinkInfoStringCompressor_.Start();
    }

    void TIndex::Stop() {
        if (IsStopping_.TrySet()) {
            OfferBus_.Stop();
            ExtraPermalinkInfoStringCompressor_.Stop();
            OriginalIdToPermalinkMapper_.Stop();
            HotelTraitsTable_.Stop();
            PricesTable_.Stop();
            GeoCounterRecordsTable_.Stop();
            IndexUpdaterThread_->Join();
            CacheInvalidationThread_->Join();
        }
    }

    void TIndex::OnReady(std::function<void(void)> handler) {
        OnReady_ = std::move(handler);
    }

    bool TIndex::IsReady() const {
        return IsReady_;
    }

    TCountResults TIndex::GetCountsWithOfferBusData(TBoundingBox boundingBox,
                                                    TBookingRange bookingRange,
                                                    const TVector<TBasicFilterGroup>& selectedFiltersGroups,
                                                    const TVector<TAdditionalFilter>& additionalFilters,
                                                    bool disablePriceCounts) const {
        return GetCounts(boundingBox, bookingRange, selectedFiltersGroups, additionalFilters, disablePriceCounts, true);
    }

    TCountResults TIndex::GetCounts(TBoundingBox boundingBox,
                                    TBookingRange bookingRange,
                                    const TVector<TBasicFilterGroup>& selectedFiltersGroups,
                                    const TVector<TAdditionalFilter>& additionalFilters,
                                    bool disablePriceCounts,
                                    bool allowExtrapolation) const {
        DEBUG_LOG << "GetCounts request with bbox: " << boundingBox
                  << ", selected filter groups: [" << JoinVectorIntoString(selectedFiltersGroups, ", ")
                  << "], additional filters: [" << JoinVectorIntoString(additionalFilters, ", ")
                  << "]" << Endl;

        auto preprocessedFilters = PreprocessFilters(selectedFiltersGroups);

        THashMap<int, size_t> predefinedGroupIdToInd;
        for (size_t i = 0; i < preprocessedFilters.PredefinedSelectedFiltersGroups.size(); i++) {
            if (!predefinedGroupIdToInd.emplace(preprocessedFilters.PredefinedSelectedFiltersGroups[i]->Id, i).second) {
                ythrow yexception() << "Duplicate group id in request: " << StringEncoder_.Decode(preprocessedFilters.PredefinedSelectedFiltersGroups[i]->Id);
            }
        }

        THashMap<int, size_t> dynamicGroupIdToInd;
        for (size_t i = 0; i < preprocessedFilters.DynamicSelectedFiltersGroups.size(); i++) {
            if (!dynamicGroupIdToInd.emplace(preprocessedFilters.DynamicSelectedFiltersGroups[i]->Id, i).second) {
                ythrow yexception() << "Duplicate group id in request: " << StringEncoder_.Decode(preprocessedFilters.DynamicSelectedFiltersGroups[i]->Id);
            }
        }

        TBitMask groupExistsMask;
        for (size_t i = 0; i < PredefinedFilters_.size(); i++) {
            if (predefinedGroupIdToInd.contains(PredefinedFilters_[i].GroupId)) {
                groupExistsMask.set(i);
            }
        }

        TVector<size_t> dynamicAdditionalFilterIndices;
        for (size_t i = 0; i < additionalFilters.size(); i++) {
            if (!PredefinedFilterIndById_.contains(additionalFilters[i].Filter->GetUniqueId())) {
                if (dynamic_cast<TBasicFilterCompare*>(additionalFilters[i].Filter.get()) || dynamic_cast<TBasicFilterIntersect*>(additionalFilters[i].Filter.get())) {
                    WARNING_LOG << "Found not precalculated filter: " << StringEncoder_.Decode(additionalFilters[i].Filter->GetUniqueId()) << Endl;
                    Counters_.NNotPrecalculatedFilterWarns.Inc();
                }
                dynamicAdditionalFilterIndices.push_back(i);
            } else {
                if (dynamic_cast<TFilterByOfferBusData*>(additionalFilters[i].Filter.get())) {
                    ythrow yexception() << "Found precalculated filter of type TFilterByOfferBusData: " << StringEncoder_.Decode(additionalFilters[i].Filter->GetUniqueId()) << Endl;
                }
            }
        }

        struct TFilterCounts {
            TFilterCounts()
                : AllMatch(0)
                , PredefinedMatch(0)
                , AllMatchOC(0)
                , PredefinedMatchOC(0)
            {
            }

            int GetCount(const TIndex& index, bool allowExtrapolation) const {
                if (allowExtrapolation && PredefinedMatchOC < PredefinedMatch && PredefinedMatchOC > index.MinHotelsWithOfferBusDataForExtrapolation_) {
                    if (PredefinedMatchOC == 0) {
                        return 0;
                    } else {
                        return static_cast<int>(static_cast<i64>(PredefinedMatch) * AllMatchOC / PredefinedMatchOC);
                    }
                } else {
                    return AllMatch;
                }
            }

            int AllMatch;
            int PredefinedMatch;
            int AllMatchOC;
            int PredefinedMatchOC;
        };

        TVector<TFilterCounts> dynamicAdditionalFilterCounts(dynamicAdditionalFilterIndices.size());
        TVector<TFilterCounts> predefinedFilterCounts(PredefinedFilters_.size());
        TFilterCounts selectedFilterCounts;

        TVector<TPriceRange> priceRanges;
        TCountResults result;

        ProcessRecords(boundingBox, SortTypeRegistry_.GetDefaultSortType(), true, {}, [this, &groupExistsMask, &priceRanges, &bookingRange, &additionalFilters,
                                     &result, &preprocessedFilters, disablePriceCounts, &dynamicAdditionalFilterIndices,
                                     &dynamicAdditionalFilterCounts, &dynamicGroupIdToInd, &selectedFilterCounts,
                                     &predefinedFilterCounts](const TFilteringPermalinkInfo& info) {
            ++result.TotalCount;

            size_t countPassingPredefinedSelectedFiltersGroups = 0;
            TVector<bool> predefinedSelectedFiltersGroupsPasses(preprocessedFilters.PredefinedSelectedFiltersGroups.size());
            for (size_t i = 0; i < preprocessedFilters.PredefinedSelectedFiltersGroups.size(); i++) {
                for (size_t j = 0; j < preprocessedFilters.PredefinedSelectedFiltersGroups[i]->Filters.size(); j++) {
                    if (info.StaticPermalinkInfo.PredefinedFiltersResults[preprocessedFilters.SelectedFiltersPredefinedIndices[i][j]]) {
                        predefinedSelectedFiltersGroupsPasses[i] = true;
                        countPassingPredefinedSelectedFiltersGroups++;
                        break;
                    }
                }
            }

            auto isPassingAllPredefinedSelected = countPassingPredefinedSelectedFiltersGroups == preprocessedFilters.PredefinedSelectedFiltersGroups.size();
            auto oneNotPassingPredefinedGroup = countPassingPredefinedSelectedFiltersGroups + 1 == preprocessedFilters.PredefinedSelectedFiltersGroups.size();

            size_t countPassingDynamicSelectedFiltersGroups = 0;
            TVector<bool> dynamicSelectedFiltersGroupsPasses(preprocessedFilters.DynamicSelectedFiltersGroups.size());
            // If at least two groups are not passing, no additional filter (i.e. no single change) can fix the situation, so we even don't need dynamic filter values
            if (isPassingAllPredefinedSelected || oneNotPassingPredefinedGroup) {
                // We limit dynamic filter usage only by hotels, which pass all predefined filters (selectedFilterCounts.PredefinedMatch) to get values consistent with hints.
                // Because while calculating hints we don't know whether hotel doesn't pass only one predefined filter or more.
                if (selectedFilterCounts.PredefinedMatch < MaxPredefinedMatchesForDynamicFilters_) {
                    for (size_t i = 0; i < preprocessedFilters.DynamicSelectedFiltersGroups.size(); i++) {
                        for (const auto& filter : preprocessedFilters.DynamicSelectedFiltersGroups[i]->Filters) {
                            if (filter->IsPassingFilter(info)) {
                                dynamicSelectedFiltersGroupsPasses[i] = true;
                                countPassingDynamicSelectedFiltersGroups++;
                                break;
                            }
                        }
                    }
                } else {
                    // No time to calculate all of them, assuming all other dynamic filters passing. The only hope is extrapolation, it may override this assumption.
                    for (size_t i = 0; i < preprocessedFilters.DynamicSelectedFiltersGroups.size(); i++) {
                        dynamicSelectedFiltersGroupsPasses[i] = true;
                        countPassingDynamicSelectedFiltersGroups++;
                    }
                }
            }

            auto totalPassingGroups = countPassingPredefinedSelectedFiltersGroups + countPassingDynamicSelectedFiltersGroups;
            auto totalGroups = preprocessedFilters.PredefinedSelectedFiltersGroups.size() + preprocessedFilters.DynamicSelectedFiltersGroups.size();

            auto isPassingAllSelected = totalPassingGroups == totalGroups;
            auto oneNotPassingGroup = totalPassingGroups + 1 == totalGroups;
            if (isPassingAllSelected) {
                ++selectedFilterCounts.AllMatch;
            }
            if (isPassingAllPredefinedSelected) {
                ++selectedFilterCounts.PredefinedMatch;
            }
            if (selectedFilterCounts.PredefinedMatch <= MaxPredefinedMatchesForDynamicFilters_ && info.OfferBusData.Defined()) {
                if (isPassingAllSelected) {
                    ++selectedFilterCounts.AllMatchOC;
                }
                if (isPassingAllPredefinedSelected) {
                    ++selectedFilterCounts.PredefinedMatchOC;
                }
            }

            TBitMask someNotPassingPredefinedGroupMask;
            for (size_t i = 0; i < predefinedSelectedFiltersGroupsPasses.size(); i++) {
                if (!predefinedSelectedFiltersGroupsPasses[i]) {
                    auto maskIt = OnlyCurrGroupNotPassingMask_.find(preprocessedFilters.PredefinedSelectedFiltersGroups[i]->Id);
                    if (maskIt != OnlyCurrGroupNotPassingMask_.end()) {
                        someNotPassingPredefinedGroupMask = maskIt->second;
                        break;
                    }
                }
            }

            auto getCountsMask = [this, &info, &groupExistsMask, &someNotPassingPredefinedGroupMask](bool isPassingAllSelected, bool oneNotPassingGroup) { // todo (mpivko):
                TBitMask onlyCurrGroupNotPassingMask;
                if (oneNotPassingGroup) {
                    onlyCurrGroupNotPassingMask = someNotPassingPredefinedGroupMask;
                }
                const auto& filtersPasses = info.StaticPermalinkInfo.PredefinedFiltersResults;
                TBitMask isPassingAllSelectedMask = isPassingAllSelected ? ~TBitMask() : TBitMask();
                auto countsMask = (isPassingAllSelectedMask & filtersPasses) |
                                  ((PredefinedFiltersSingleMask_ | PredefinedFiltersOrMask_) & filtersPasses & onlyCurrGroupNotPassingMask) |
                                  (PredefinedFiltersOrMask_ & groupExistsMask & isPassingAllSelectedMask);
                return countsMask;
            };

            auto countsForPredefinedSelectedMask = getCountsMask(isPassingAllPredefinedSelected, oneNotPassingPredefinedGroup);
            for (size_t i = 0; i < PredefinedFilters_.size(); i++) {
                predefinedFilterCounts[i].PredefinedMatch += countsForPredefinedSelectedMask[i];
                if (predefinedFilterCounts[i].PredefinedMatch <= MaxPredefinedMatchesForDynamicFilters_ && info.OfferBusData.Defined()) {
                    predefinedFilterCounts[i].PredefinedMatchOC += countsForPredefinedSelectedMask[i];
                }
            }

            auto countsForAllSelectedMask = getCountsMask(isPassingAllSelected, oneNotPassingGroup);
            for (size_t i = 0; i < PredefinedFilters_.size(); i++) {
                predefinedFilterCounts[i].AllMatch += countsForAllSelectedMask[i];
                if (predefinedFilterCounts[i].PredefinedMatch <= MaxPredefinedMatchesForDynamicFilters_ && info.OfferBusData.Defined()) { // sic! we limit hotels by PredefinedMatch, as we do with selected filters
                    predefinedFilterCounts[i].AllMatchOC += countsForAllSelectedMask[i];
                }
            }

            for (size_t i = 0; i < dynamicAdditionalFilterIndices.size(); i++) {
                const auto& additionalFilter = additionalFilters[dynamicAdditionalFilterIndices[i]];
                switch (additionalFilter.Type) {
                    case TAdditionalFilter::Single: {
                        if (isPassingAllSelected || (oneNotPassingGroup &&
                            dynamicGroupIdToInd.contains(additionalFilter.GroupId) &&
                            !dynamicSelectedFiltersGroupsPasses[dynamicGroupIdToInd.at(additionalFilter.GroupId)])) {
                            if (selectedFilterCounts.PredefinedMatch <= MaxPredefinedMatchesForDynamicFilters_) {
                                if (additionalFilter.Filter->IsPassingFilter(info)) {
                                    ++dynamicAdditionalFilterCounts[i].AllMatch;
                                    if (info.OfferBusData.Defined()) {
                                        ++dynamicAdditionalFilterCounts[i].AllMatchOC;
                                    }
                                }
                            } else {
                                // Again no time to calculate all of them
                                ++dynamicAdditionalFilterCounts[i].AllMatch;
                            }
                        }
                        break;
                    }
                    case TAdditionalFilter::And: {
                        if (isPassingAllSelected) {
                            if (selectedFilterCounts.PredefinedMatch <= MaxPredefinedMatchesForDynamicFilters_) {
                                if (additionalFilter.Filter->IsPassingFilter(info)) {
                                    ++dynamicAdditionalFilterCounts[i].AllMatch;
                                    if (info.OfferBusData.Defined()) {
                                        ++dynamicAdditionalFilterCounts[i].AllMatchOC;
                                    }
                                }
                            } else {
                                // Again no time to calculate all of them
                                ++dynamicAdditionalFilterCounts[i].AllMatch;
                            }
                        }
                        break;
                    }
                    case TAdditionalFilter::Or: {
                        if (isPassingAllSelected || (oneNotPassingGroup &&
                            dynamicGroupIdToInd.contains(additionalFilter.GroupId) &&
                            !dynamicSelectedFiltersGroupsPasses[dynamicGroupIdToInd.at(additionalFilter.GroupId)])) {
                            if (selectedFilterCounts.PredefinedMatch <= MaxPredefinedMatchesForDynamicFilters_) {
                                if (additionalFilter.Filter->IsPassingFilter(info) ||
                                    (isPassingAllSelected && dynamicGroupIdToInd.contains(additionalFilter.GroupId))) {
                                    ++dynamicAdditionalFilterCounts[i].AllMatch;
                                    if (info.OfferBusData.Defined()) {
                                        ++dynamicAdditionalFilterCounts[i].AllMatchOC;
                                    }
                                }
                            } else {
                                // Again no time to calculate all of them
                                ++dynamicAdditionalFilterCounts[i].AllMatch;
                            }
                        }
                        break;
                    }
                }
            }

            if (info.PriceInfo.HasData() && !disablePriceCounts) {
                priceRanges.push_back(info.PriceInfo.GetPrice(bookingRange));
            }

            return true;
        });

        DEBUG_LOG << selectedFilterCounts.PredefinedMatchOC << " out of " << selectedFilterCounts.PredefinedMatch << " predefined matches has offerbus data" << Endl;
        DEBUG_LOG << selectedFilterCounts.AllMatchOC << " out of " << selectedFilterCounts.AllMatch << " all matches has offerbus data" << Endl;

        result.AdditionalFilterCounts.resize(additionalFilters.size());
        for (size_t i = 0; i < additionalFilters.size(); i++) {
            auto additionalFilterPredefinedIndexIt = PredefinedFilterIndById_.find(additionalFilters[i].Filter->GetUniqueId());
            if (additionalFilterPredefinedIndexIt != PredefinedFilterIndById_.end()) {
                result.AdditionalFilterCounts[i].UniqueId = additionalFilters[i].Filter->GetUniqueId();
                result.AdditionalFilterCounts[i].Count = predefinedFilterCounts[additionalFilterPredefinedIndexIt->second].GetCount(*this, allowExtrapolation);
            }
        }
        for (size_t i = 0; i < dynamicAdditionalFilterIndices.size(); i++) {
            auto index = dynamicAdditionalFilterIndices[i];
            auto uniqueId = StringEncoder_.Decode(additionalFilters[index].Filter->GetUniqueId());
            auto originalUniqueId = uniqueId.substr(1, uniqueId.length() - 2); // todo (mpivko): dirty hack :(
            result.AdditionalFilterCounts[index].UniqueId = StringEncoder_.Encode(originalUniqueId);
            dynamicAdditionalFilterCounts[i].PredefinedMatch = selectedFilterCounts.PredefinedMatch;
            dynamicAdditionalFilterCounts[i].PredefinedMatchOC = selectedFilterCounts.PredefinedMatchOC;
            result.AdditionalFilterCounts[index].Count = dynamicAdditionalFilterCounts[i].GetCount(*this, allowExtrapolation);
        }
        result.MatchedCount = selectedFilterCounts.GetCount(*this, allowExtrapolation);

        result.PriceResults = BuildPriceResults(bookingRange, priceRanges);

        Counters_.NRecordsAfterParameterFiltering.Update(result.MatchedCount);
        DEBUG_LOG << result.MatchedCount << " records passed filter" << Endl;

        TStringBuilder builder;
        builder << "GetCounts response: MatchedCount=" << result.MatchedCount
                << " TotalCount=" << result.TotalCount << " AdditionalFilterCounts=[";
        for (const auto& x : result.AdditionalFilterCounts) {
            builder << x.UniqueId << ": " << x.Count << ", ";
        }
        builder << "]" << Endl;
        DEBUG_LOG << builder;

        return result;
    }

    THotelsResults TIndex::GetHotels(TBoundingBox boundingBox,
                                     const TVector<TBasicFilterGroup>& selectedFiltersGroups,
                                     const TSortTypeWithContext& sortTypeWithContext,
                                     TBookingRange bookingRange,
                                     const TString& occupancy,
                                     TMaybe<TPermalink> topHotelPermalink,
                                     size_t skip,
                                     size_t limit) const {
        Y_UNUSED(bookingRange);
        Y_UNUSED(occupancy);
        Y_ENSURE(limit > 0, "Limit should be positive");
        DEBUG_LOG << "GetHotels request with bbox: " << boundingBox
                  << ", selected filter groups: [" << JoinVectorIntoString(selectedFiltersGroups, ", ")
                  << "]" << Endl;

        auto extraPermalinkInfos = ExtraPermalinkInfos_.GetData();
        auto featureInfo = FeaturesMetaInfos_.GetData();
        THotelsResults result{};

        auto starsFeatureId = StringEncoder_.Encode("star");
        auto categoryFeatureId = StringEncoder_.Encode("category_id");
        auto ratingFeatureId = StringEncoder_.Encode("rating");

        TMaybe<std::function<bool(const TFilteringPermalinkInfo&)>> sortThresholdFilter;

        const auto& realTimeSortInIndexInfo = sortTypeWithContext.SortType.GetRealTimeSortInIndex();
        THashMap<TPermalink, i64> realTimeInIndexSortFactorMap;
        if (realTimeSortInIndexInfo.Defined()) {
            realTimeInIndexSortFactorMap.reserve(IN_INDEX_SORT_FACTORS_SZ_EXPECTATION);
            TVector<i64> realTimeInIndexSortFactor;
            realTimeInIndexSortFactor.reserve(IN_INDEX_SORT_FACTORS_SZ_EXPECTATION);
            ProcessRecordsWithFilteringForGetHotels(boundingBox, selectedFiltersGroups, sortTypeWithContext.SortType, topHotelPermalink, 0, {}, {},
                                                    [&realTimeSortInIndexInfo, &realTimeInIndexSortFactor,
                                                     &realTimeInIndexSortFactorMap, &sortTypeWithContext](const TFilteringPermalinkInfo& filteringInfo) {
                realTimeInIndexSortFactor.push_back(realTimeSortInIndexInfo->GetFactorFunc(filteringInfo, sortTypeWithContext.Context));
                realTimeInIndexSortFactorMap.emplace(filteringInfo.StaticPermalinkInfo.Permalink, realTimeInIndexSortFactor.back());
            });

            if (realTimeInIndexSortFactor.size() > skip + limit) {
                std::nth_element(realTimeInIndexSortFactor.begin(), realTimeInIndexSortFactor.begin() + skip + limit - 1, realTimeInIndexSortFactor.end());
                i64 threshold = realTimeInIndexSortFactor.at(skip + limit - 1);
                sortThresholdFilter = [&realTimeInIndexSortFactorMap, threshold](const TFilteringPermalinkInfo& filteringInfo) {
                    return realTimeInIndexSortFactorMap.at(filteringInfo.StaticPermalinkInfo.Permalink) <= threshold;
                };
            }
        }

        ProcessRecordsWithFilteringForGetHotels(boundingBox, selectedFiltersGroups, sortTypeWithContext.SortType, topHotelPermalink, skip, limit, sortThresholdFilter,
                                                [this, &result, &starsFeatureId, &categoryFeatureId, &ratingFeatureId, &extraPermalinkInfos,
                                                 &featureInfo, &topHotelPermalink](const TFilteringPermalinkInfo& filteringInfo) {
            result.Hotels.push_back(BuildHotelResult(filteringInfo, starsFeatureId, categoryFeatureId, ratingFeatureId, extraPermalinkInfos, featureInfo, topHotelPermalink));
        });

        if (realTimeSortInIndexInfo.Defined()) {
            realTimeSortInIndexInfo->ReorderFunc(&result.Hotels, realTimeInIndexSortFactorMap);
        }

        return result;
    }

    void TIndex::ProcessRecordsWithFilteringForGetHotels(TBoundingBox boundingBox,
                                                         const TVector<TBasicFilterGroup>& selectedFiltersGroups,
                                                         const TSortType& sortType,
                                                         TMaybe<TPermalink> topHotelPermalink,
                                                         size_t skip,
                                                         TMaybe<size_t> limit,
                                                         TMaybe<std::function<bool(const TFilteringPermalinkInfo&)>> sortThresholdFilter,
                                                         const std::function<void(const TFilteringPermalinkInfo&)>& processHotel) const {
        auto preprocessedFilters = PreprocessFilters(selectedFiltersGroups);

        size_t skippedCnt = 0;
        size_t passedCnt = 0;

        ProcessRecords(boundingBox, sortType, false, topHotelPermalink,
                       [&preprocessedFilters, &skippedCnt, &passedCnt, skip, limit,
                        &sortThresholdFilter, &processHotel] (const TFilteringPermalinkInfo& info) {

            for (size_t i = 0; i < preprocessedFilters.PredefinedSelectedFiltersGroups.size(); i++) {
                bool currentFiltersGroupPasses = false;
                for (size_t j = 0; j < preprocessedFilters.PredefinedSelectedFiltersGroups[i]->Filters.size(); j++) {
                    if (info.StaticPermalinkInfo.PredefinedFiltersResults[preprocessedFilters.SelectedFiltersPredefinedIndices[i][j]]) {
                        currentFiltersGroupPasses = true;
                        break;
                    }
                }
                if (!currentFiltersGroupPasses) {
                    return true; // Skipping hotel which doesn't pass filter
                }
            }

            for (size_t i = 0; i < preprocessedFilters.DynamicSelectedFiltersGroups.size(); i++) {
                bool currentFiltersGroupPasses = false;
                for (const auto& filter : preprocessedFilters.DynamicSelectedFiltersGroups[i]->Filters) {
                    if (filter->IsPassingFilter(info)) {
                        currentFiltersGroupPasses = true;
                        break;
                    }
                }
                if (!currentFiltersGroupPasses) {
                    return true; // Skipping hotel which doesn't pass filter
                }
            }

            if (sortThresholdFilter.Defined()) {
                if (!sortThresholdFilter.GetRef()(info)) {
                    return true;
                }
            }

            if (skippedCnt < skip) {
                skippedCnt++;
                return true;
            }

            if (limit.Defined() && passedCnt >= limit.GetRef()) {
                return false;
            }
            passedCnt++;

            processHotel(info);

            return true;
        });
    }

    TIndex::TPreprocessedFilters TIndex::PreprocessFilters(const TVector<TBasicFilterGroup>& selectedFiltersGroups) const {
        TVector<const TBasicFilterGroup*> predefinedSelectedFiltersGroups;
        TVector<const TBasicFilterGroup*> dynamicSelectedFiltersGroups;
        for (const auto& selectedFiltersGroup : selectedFiltersGroups) {
            if (selectedFiltersGroup.Filters.empty()) {
                ythrow yexception() << "Empty selected filter group. Group id: " << StringEncoder_.Decode(selectedFiltersGroup.Id);
            }
            bool hasPredefined = false;
            bool hasDynamic = false;
            for (const auto& filter : selectedFiltersGroup.Filters) {
                if (PredefinedFilterIndById_.contains(filter->GetUniqueId())) {
                    if (dynamic_cast<TFilterByOfferBusData*>(filter.get())) {
                        ythrow yexception() << "Found precalculated filter of type TFilterByOfferBusData: " << StringEncoder_.Decode(filter->GetUniqueId()) << Endl;
                    }
                    hasPredefined = true;
                } else {
                    if (dynamic_cast<TBasicFilterCompare*>(filter.get()) || dynamic_cast<TBasicFilterIntersect*>(filter.get())) {
                        WARNING_LOG << "Found not precalculated filter: " << StringEncoder_.Decode(filter->GetUniqueId()) << Endl;
                        Counters_.NNotPrecalculatedFilterWarns.Inc();
                    }
                    hasDynamic = true;
                }
            }
            if (hasPredefined && hasDynamic) {
                ythrow yexception() << "Found selected filter group with both predefined and dynamic filters. Group id: " << StringEncoder_.Decode(selectedFiltersGroup.Id);
            }
            if (hasPredefined) {
                predefinedSelectedFiltersGroups.push_back(&selectedFiltersGroup);
            }
            if (hasDynamic) {
                dynamicSelectedFiltersGroups.push_back(&selectedFiltersGroup);
            }
        }

        TVector<TVector<int>> selectedFiltersPredefinedIndices(predefinedSelectedFiltersGroups.size());
        for (size_t i = 0; i < predefinedSelectedFiltersGroups.size(); i++) {
            const auto& group = predefinedSelectedFiltersGroups[i]->Filters;
            selectedFiltersPredefinedIndices[i].resize(group.size());
            for (size_t j = 0; j < group.size(); j++) {
                selectedFiltersPredefinedIndices[i][j] = PredefinedFilterIndById_.at(group[j]->GetUniqueId());
            }
        }
        return TPreprocessedFilters{
            predefinedSelectedFiltersGroups,
            dynamicSelectedFiltersGroups,
            selectedFiltersPredefinedIndices
        };
    }

    TMaybe<TString> TIndex::GetFirstFeatureValue(const TFilteringPermalinkInfo& info, int featureId) const {
        auto it = info.StaticPermalinkInfo.Features.find(featureId);
        if (it != info.StaticPermalinkInfo.Features.end()) {
            struct TVisitor {
                int FeatureId;
                const TStringEncoder& StringEncoder;

                TVisitor(int featureId, const TStringEncoder& stringEncoder)
                    : FeatureId(featureId)
                    , StringEncoder(stringEncoder) {
                }
                TMaybe<int> operator()(const std::monostate&) const {
                    return {};
                }
                TMaybe<int> operator()(const double&) const {
                    ythrow yexception() << StringEncoder.Decode(FeatureId) + " feature has double value";
                }
                TMaybe<int> operator()(const TFeature::Empty&) const {
                    return {};
                }
                TMaybe<int> operator()(const int& featureValue) const {
                    return featureValue;
                }
                TMaybe<int> operator()(const std::pair<int, int>& featureValue) const {
                    return featureValue.first;
                }
                TMaybe<int> operator()(const TIntrusivePtr<TFeature::Many>& featureValuePtr) const {
                    Y_ENSURE(featureValuePtr);
                    return featureValuePtr->Values.at(0);
                }
            };
            auto value = std::visit(TVisitor(featureId, StringEncoder_), it->second.Value);
            if (value.Empty()) {
                return {};
            }
            return StringEncoder_.Decode(value.GetRef());
        }
        return {};
    }

    THotelsResults::THotelResult TIndex::BuildHotelResult(const TFilteringPermalinkInfo& info,
                                                          int starsFeatureId,
                                                          int categoryFeatureId,
                                                          int ratingFeatureId,
                                                          const TAtomicSharedPtr<THashMap<TPermalink, TExtraPermalinkInfo>>& extraPermalinkInfos,
                                                          const TAtomicSharedPtr<THashMap<int, TFeatureMetaInfo>>& featureMetaInfo,
                                                          TMaybe<TPermalink> topHotelPermalink) const {
        auto currHotel = THotelsResults::THotelResult();

        currHotel.Permalink = info.StaticPermalinkInfo.Permalink;
        currHotel.Coordinates = info.StaticPermalinkInfo.Position;
        auto extraInfoIt = extraPermalinkInfos->find(currHotel.Permalink);
        if (extraInfoIt != extraPermalinkInfos->end()) {
            currHotel.Name = ExtraPermalinkInfoStringCompressor_.Decompress(extraInfoIt->second.Name);
            currHotel.Address = ExtraPermalinkInfoStringCompressor_.Decompress(extraInfoIt->second.Address);
            currHotel.ImageCount = extraInfoIt->second.ImageCount;
            currHotel.DisplayedLocationGeoId = extraInfoIt->second.DisplayedLocationGeoId;
            currHotel.ExtraInfo = extraInfoIt->second;
        } else {
            ERROR_LOG << "No extra permalink info for permalink: " << currHotel.Permalink << Endl;
        }

        auto stars = GetFirstFeatureValue(info, starsFeatureId);
        if (stars.Defined()) {
            auto starsIt = StarsMapping_.find(stars.GetRef());
            if (starsIt != StarsMapping_.end()) {
                currHotel.Stars = starsIt->second;
            }
        }
        auto category = GetFirstFeatureValue(info, categoryFeatureId);
        if (category.Defined()) {
            auto id = std::stoll(category.GetRef());
            auto nameIt = RubricNames.find(id);
            if (nameIt == RubricNames.end()) {
                currHotel.Rubric = THotelsResults::TRubric{id, ""};
                ERROR_LOG << "No name for rubric: " << id << Endl;
            } else {
                currHotel.Rubric = THotelsResults::TRubric{id, nameIt->second};
            }
        }
        currHotel.Rating = GetRating(info.StaticPermalinkInfo.Features, ratingFeatureId);
        currHotel.IsTopHotel = topHotelPermalink.Defined() && info.StaticPermalinkInfo.Permalink == topHotelPermalink.GetRef();

        for (const auto& feature : info.StaticPermalinkInfo.Features) {
            struct TVisitor {
                const TFeatureMetaInfo& FeatureMetaInfo;
                const TString& Id;
                const TStringEncoder& StringEncoder;

                TVisitor(const TFeatureMetaInfo& featureMetaInfo, const TString& id, const TStringEncoder& stringEncoder)
                    : FeatureMetaInfo(featureMetaInfo)
                    , Id(id)
                    , StringEncoder(stringEncoder) {
                }

                THotelsResults::THotelFeature operator()(const std::monostate&) const {
                    ythrow yexception() << "TFeature has no value";
                }
                THotelsResults::THotelFeature operator()(const double& value) const {
                    return THotelsResults::THotelFeature(Id, FeatureMetaInfo.Name, value);
                }
                THotelsResults::THotelFeature operator()(const TFeature::Empty&) const {
                    return THotelsResults::THotelFeature(Id, FeatureMetaInfo.Name, TVector<THotelsResults::TStringFeatureValue>{});
                }
                THotelsResults::THotelFeature operator()(const int& featureValue) const {
                    auto decodedValue = StringEncoder.Decode(featureValue);
                    if (FeatureMetaInfo.Type == TFeatureMetaInfo::EType::Boolean) {
                        return THotelsResults::THotelFeature(Id, FeatureMetaInfo.Name, decodedValue == "1");
                    }
                    if (FeatureMetaInfo.Type == TFeatureMetaInfo::EType::Numeric || FeatureMetaInfo.Type == TFeatureMetaInfo::EType::Range) {
                        return THotelsResults::THotelFeature(Id, FeatureMetaInfo.Name, {THotelsResults::TStringFeatureValue(decodedValue, decodedValue)});
                    }
                    return BuildTextValue({featureValue});
                }
                THotelsResults::THotelFeature operator()(const std::pair<int, int>& featureValue) const {
                    return BuildTextValue({featureValue.first, featureValue.second});
                }
                THotelsResults::THotelFeature operator()(const TIntrusivePtr<TFeature::Many>& featureValuePtr) const {
                    Y_ENSURE(featureValuePtr);
                    return BuildTextValue(featureValuePtr->Values);
                }

                THotelsResults::THotelFeature BuildTextValue(const TVector<int>& values) const {
                    TVector<THotelsResults::TStringFeatureValue> decodedValues;
                    decodedValues.reserve(values.size());
                    for (const auto& value: values) {
                        auto valueIt = FeatureMetaInfo.ValueNames.find(value);
                        if (valueIt == FeatureMetaInfo.ValueNames.end()) {
                            ERROR_LOG << "Value name not found for " << Id << " " << StringEncoder.Decode(value) << Endl;
                        } else {
                            decodedValues.emplace_back(StringEncoder.Decode(value), valueIt->second);
                        }
                    }
                    return THotelsResults::THotelFeature(Id, FeatureMetaInfo.Name, decodedValues);
                }
            };

            auto featureId = StringEncoder_.Decode(feature.first);
            auto featureInfoIt = featureMetaInfo->find(feature.first);
            if (featureInfoIt != featureMetaInfo->end()) {
                currHotel.Features.push_back(std::visit(TVisitor(featureInfoIt->second, featureId, StringEncoder_), feature.second.Value));
            }
        }

        return currHotel;
    }

    TMaybe<TOfferBusData> TIndex::GetPermalinkInfo(TPermalink permalink) const {
        with_lock (OfferBusDataMutex_) {
            if (OfferBusData_.contains(permalink)) {
                return OfferBusData_.at(permalink);
            }
            return {};
        }
    }

    TMaybe<TFilteringPermalinkInfo> TIndex::GetPermalinkFilteringInfo(TPermalink permalink) const {
        TPermalinkToBucketPtr permalinkToBucket;
        with_lock (BucketsMutex_) {
            permalinkToBucket = PermalinkToBucket_;
        }
        auto bucketIt = permalinkToBucket->find(permalink);
        if (bucketIt == permalinkToBucket->end()) {
            return {};
        }
        auto [i, j] = bucketIt->second;
        TBucketPtr bucket;
        with_lock (BucketsMutex_) {
            bucket = Buckets_.at(i).at(j);
        }
        auto indexIt = bucket->PermalinkToIndex.find(permalink);
        if (indexIt == bucket->PermalinkToIndex.end()) {
            return {};
        }
        return bucket->PermalinkInfos.at(indexIt->second);
    }

    TMaybe<TExtraPermalinkInfo> TIndex::GetExtraPermalinkInfo(TPermalink permalink) const {
        auto extraPermalinkInfos = ExtraPermalinkInfos_.GetData();
        auto infoIt = extraPermalinkInfos->find(permalink);
        if (infoIt == extraPermalinkInfos->end()) {
            return {};
        }
        return infoIt->second;
    }

    TCountResults::TPriceResults TIndex::BuildPriceResults(const TBookingRange& bookingRange, const TVector<TPriceRange>& priceRanges) const {
        TVector<ui32> minPrices;
        TVector<ui32> maxPrices;
        for (const auto& priceRange : priceRanges) {
            minPrices.push_back(priceRange.MinPrice);
            maxPrices.push_back(priceRange.MaxPrice);
        }
        std::sort(minPrices.begin(), minPrices.end());
        std::sort(maxPrices.begin(), maxPrices.end());

        auto nights = bookingRange.CheckOutDate - bookingRange.CheckInDate;
        auto minPriceEstimate = 0u; // https://st.yandex-team.ru/HOTELS-4694#5de7e4326808074705829a02
        auto maxPriceEstimate = 10000u * nights;

        TCountResults::TPriceResults priceResults{};

        if (minPriceEstimate < maxPriceEstimate) {
            priceResults.MinPriceEstimate = minPriceEstimate;
            priceResults.MaxPriceEstimate = maxPriceEstimate;

            int bucketCount = 20;
            ui32 bucketSize = Max((priceResults.MaxPriceEstimate / bucketCount) / 100, 1u) * 100;

            size_t minPricePos = 0;
            size_t maxPricePos = 0;
            for (int i = 0; i < bucketCount; i++) {
                ui32 lBorder = i * bucketSize;
                ui32 rBorder = i + 1 < bucketCount ? (i + 1) * bucketSize : std::numeric_limits<ui32>::max();
                while (minPricePos < minPrices.size() && minPrices[minPricePos] < rBorder) {
                    minPricePos++;
                }
                while (maxPricePos < maxPrices.size() && maxPrices[maxPricePos] < lBorder) {
                    maxPricePos++;
                }
                priceResults.HistogramBounds.push_back(lBorder);
                priceResults.HistogramCounts.push_back(minPricePos - maxPricePos);
            }
        }

        return priceResults;
    }

    void TIndex::ProcessRecords(TBoundingBox boundingBox,
                                const TSortType& sortType,
                                bool useUnorderedVersion,
                                TMaybe<TPermalink> topHotelPermalink,
                                const std::function<bool(const TFilteringPermalinkInfo&)>& handler) const {
        const auto& lowerLeft = boundingBox.LowerLeft;
        const auto& upperRight = boundingBox.UpperRight;

        auto start = TInstant::Now();
        auto [lowerLatBucket, leftLonBucket] = GetBucketIndex(lowerLeft);
        auto [upperLatBucket, rightLonBucket] = GetBucketIndex(upperRight);

        DEBUG_LOG << "lowerLeft: " << lowerLeft << ", upperRight: " << upperRight << Endl;

        DEBUG_LOG << "lowerLatBucket: " << lowerLatBucket << ", leftLonBucket: " << leftLonBucket
                  << ", upperLatBucket: " << upperLatBucket << ", rightLonBucket: " << rightLonBucket
                  << Endl;

        upperLatBucket = (upperLatBucket + 1) % BucketCount_;
        rightLonBucket = (rightLonBucket + 1) % BucketCount_;

        auto prices = Prices_.GetData();

        size_t totalCount = 0;
        size_t afterFilterCount = 0;
        size_t bucketCount = 0;

        TVector<TBucketRecordsStream> streams;

        TVector<NTravel::NGeoCounter::TFilteringPermalinkInfo> topPermalinkBucket;
        if (topHotelPermalink.Defined()) {
            auto topHotelPermalinkFilteringInfo = GetPermalinkFilteringInfo(topHotelPermalink.GetRef());
            if (topHotelPermalinkFilteringInfo.Defined()) {
                topPermalinkBucket.push_back(topHotelPermalinkFilteringInfo.GetRef());
                streams.push_back(TBucketRecordsStream(&topPermalinkBucket, nullptr, lowerLeft, upperRight, {}, true));
            }
        }
        TVector<TBucketPtr> usedBuckets;
        for (int latBucketOffset = 0; latBucketOffset < BucketCount_; latBucketOffset++) {
            int latBucket = (lowerLatBucket + latBucketOffset) % BucketCount_;
            if (latBucket == upperLatBucket && latBucketOffset != 0) {
                break;
            }
            for (int lonBucketOffset = 0; lonBucketOffset < BucketCount_; lonBucketOffset++) {
                int lonBucket = (leftLonBucket + lonBucketOffset) % BucketCount_;
                if (lonBucket == rightLonBucket && lonBucketOffset != 0) {
                    break;
                }
                TBucketPtr bucket;
                with_lock (BucketsMutex_) {
                    bucket = Buckets_.at(latBucket).at(lonBucket);
                }
                usedBuckets.push_back(bucket);
                if (sortType.Id == SortTypeRegistry_.GetDefaultSortType().Id) {
                    streams.push_back(TBucketRecordsStream(&bucket->PermalinkInfos, nullptr, lowerLeft, upperRight, topHotelPermalink, false));
                } else {
                    streams.push_back(TBucketRecordsStream(&bucket->PermalinkInfos, &bucket->Permutations.at(sortType.Id), lowerLeft, upperRight, topHotelPermalink, false));
                }
                totalCount += bucket->PermalinkInfos.size();
                bucketCount++;
            }
        }

        if (useUnorderedVersion) {
            ProcessStreamsUnordered(StringEncoder_, &streams, &afterFilterCount, handler);
        } else {
            ProcessStreams(StringEncoder_, sortType.Id, &streams, &afterFilterCount, handler);
        }

        Counters_.NRecordsScanned.Update(totalCount);
        Counters_.NRecordsAfterGeoFiltering.Update(afterFilterCount);
        Counters_.NBucketsScanned.Update(bucketCount);

        auto processTime = (TInstant::Now() - start).MilliSeconds();
        Counters_.ProcessTimeMs.Update(processTime);

        DEBUG_LOG << "Processed " << totalCount << " records in " << bucketCount << " buckets (" << afterFilterCount << " after geo filter) in " << processTime << "ms" << Endl;
    }

    void TIndex::InitBuckets(TBuckets* buckets) const {
        buckets->resize(BucketCount_);
        for (auto& row: (*buckets)) {
            row.resize(BucketCount_);
            for (int i = 0; i < BucketCount_; i++) {
                row[i] = MakeAtomicShared<TBucket>();
            }
        }
    }

    TAdditionalFilter::EType TIndex::ConvertFilterGroupTypeEnum(const NTravelProto::NHotelFiltersConfig::TFilterInfoConfig::TBasicFilterGroup::EBasicFilterGroupType& type) const {
        switch (type) {
#define TYPE(_PB_NAME_, _INNER_NAME_)                                                                                                                      \
    case NTravelProto::NHotelFiltersConfig::TFilterInfoConfig::TBasicFilterGroup::EBasicFilterGroupType::TFilterInfoConfig_TBasicFilterGroup_EBasicFilterGroupType_##_PB_NAME_: \
        return TAdditionalFilter::EType::_INNER_NAME_
            TYPE(Single, Single);
            TYPE(Or, Or);
            TYPE(And, And);
#undef TYPE
            default:
                ythrow yexception() << "Unknown enum value " << type;
        }
    }

    void TIndex::InitPredefinedFilters() {
        for (const auto& detailedFilters : FilterInfoConfig_.GetDetailedFilters()) {
            for (const auto& item : detailedFilters.GetItems()) {
                auto filter = FilterRegistry_.BuildFilterForPredefined(item.GetFilter());
                auto basicFilter = dynamic_cast<TBasicFilterBase*>(filter.get());
                if (!basicFilter) {
                    // Spkipping filters by price and by offer data
                    continue;
                }
                DEBUG_LOG << "Adding predefined filter (id=" << StringEncoder_.Decode(filter->GetUniqueId()) << ") " << filter << Endl;
                auto id = filter->GetUniqueId();
                if (PredefinedFilterIndById_.contains(id)) {
                    WARNING_LOG << "Duplicate predefined filter id: " << StringEncoder_.Decode(id) << Endl;
                    continue;
                }
                auto groupId = basicFilter->GetBusinessId(); // todo (mpivko): use true group id
                auto type = ConvertFilterGroupTypeEnum(detailedFilters.GetType());
                PredefinedFilters_.emplace_back(TAdditionalFilter(groupId, type, std::move(filter)));
                auto ind = PredefinedFilters_.size() - 1;
                if (ind >= PredefinedFiltersSingleMask_.size()) {
                    ythrow yexception() << "Too many filters. Maximum for current TBitMask is " << PredefinedFiltersSingleMask_.size();
                }
                if (!PredefinedFilterIndById_.emplace(id, ind).second) {
                    ythrow yexception() << "Failed to insert predefined filter: " << StringEncoder_.Decode(id);
                }
                switch (type) {
                    case TAdditionalFilter::EType::Single:
                        PredefinedFiltersSingleMask_.set(ind);
                        break;
                    case TAdditionalFilter::EType::Or:
                        PredefinedFiltersOrMask_.set(ind);
                        break;
                    case TAdditionalFilter::EType::And:
                        PredefinedFiltersAndMask_.set(ind);
                        break;
                }
            }
        }
        for (size_t i = 0; i < PredefinedFilters_.size(); i++) {
            const auto& groupId = PredefinedFilters_[i].GroupId;
            OnlyCurrGroupNotPassingMask_[groupId].set(i);
        }
    }

    void TIndex::InitGeoCounterRecordsTable() {
        auto conv = [](const NYT::TNode& node, NTravelProto::NGeoCounter::TGeoCounterRecord* proto) {
            *proto = NProtobufJson::Json2Proto<NTravelProto::NGeoCounter::TGeoCounterRecord>(
                node["Features"].AsString(),
                NProtobufJson::TJson2ProtoConfig().SetCastRobust(true));
            proto->SetPermalink(node["Permalink"].IntCast<ui64>());
            proto->MutablePosition()->SetLat(node["Lat"].AsDouble());
            proto->MutablePosition()->SetLon(node["Lon"].AsDouble());
            proto->SetName(node["Name"].IsNull() ? "" : node["Name"].AsString());
            proto->SetAddress(node["Address"].IsNull() ? "" : node["Address"].AsString());
            if (node["PhotoCount"].IsNull()) {
                proto->SetPhotoCount(0);
            } else {
                proto->SetPhotoCount(node["PhotoCount"].IntCast<int>());
            }
            if (node["GeoId"].IsNull()) {
                proto->SetGeoId(0);
            } else {
                proto->SetGeoId(node["GeoId"].IntCast<int>());
            }
            if (!node.HasKey("DisplayedLocationGeoId") || node["DisplayedLocationGeoId"].IsNull()) {
                proto->SetDisplayedLocationGeoId(0);
            } else {
                proto->SetDisplayedLocationGeoId(node["DisplayedLocationGeoId"].IntCast<int>());
            }
            if (node.HasKey("RankingFactors")) {
                auto protoWithRankingFactors = NProtobufJson::Json2Proto<NTravelProto::NGeoCounter::TGeoCounterRecord>(
                    node["RankingFactors"].AsString(),
                    NProtobufJson::TJson2ProtoConfig().SetCastRobust(true));
                proto->MutableRankingFactors()->CopyFrom(protoWithRankingFactors.GetRankingFactors());
            }
            if (node.HasKey("RealtimeRankingFloatFeatures")) {
                auto protoWithRealtimeRankingFloatFeatures = NProtobufJson::Json2Proto<NTravelProto::NGeoCounter::TGeoCounterRecord>(
                    node["RealtimeRankingFloatFeatures"].AsString(),
                    NProtobufJson::TJson2ProtoConfig().SetCastRobust(true));
                proto->MutableRealtimeRankingFloatFeatures()->CopyFrom(protoWithRealtimeRankingFloatFeatures.GetRealtimeRankingFloatFeatures());
            }
            if (node.HasKey("RealtimeRankingCatFeatures")) {
                auto protoWithRealtimeRankingCatFeatures = NProtobufJson::Json2Proto<NTravelProto::NGeoCounter::TGeoCounterRecord>(
                    node["RealtimeRankingCatFeatures"].AsString(),
                    NProtobufJson::TJson2ProtoConfig().SetCastRobust(true));
                proto->MutableRealtimeRankingCatFeatures()->CopyFrom(protoWithRealtimeRankingCatFeatures.GetRealtimeRankingCatFeatures());
            }
        };
        auto data = [this](const NTravelProto::NGeoCounter::TGeoCounterRecord& proto) {
            auto name = ExtraPermalinkInfoStringCompressor_.Compress(proto.GetName());
            auto address = ExtraPermalinkInfoStringCompressor_.Compress(proto.GetAddress());
            auto photoCount = proto.GetPhotoCount();
            auto geoId = proto.GetGeoId();
            auto displayedLocationGeoId = proto.GetDisplayedLocationGeoId();
            auto position = TPosition(proto.GetPosition().GetLat(), proto.GetPosition().GetLon());
            THashMap<int, TFeature> features;
            for (const auto& feature : proto.GetFeatures()) {
                auto encodedId = StringEncoder_.Encode(feature.GetId());
                auto type = TFeatureMetaInfo::EType::Other;
                if (feature.GetType() == "Bool") {
                    type = TFeatureMetaInfo::EType::Boolean;
                } else if (feature.GetType() == "Numeric") {
                    type = TFeatureMetaInfo::EType::Numeric;
                } else if (feature.GetType() == "Range") {
                    type = TFeatureMetaInfo::EType::Range;
                }
                auto [metaInfoIt, _] = FeaturesMetaInfos_.UnlockedAddToNewData(encodedId, TFeatureMetaInfo{type, feature.GetName(), {}});
                TVector<TString> values;
                for (const auto& value : feature.GetExportedValue()) {
                    if (value.HasTextValue()) {
                        if (feature.GetType() == "Enum") {
                            metaInfoIt->second.ValueNames.emplace(StringEncoder_.Encode(value.GetTextValue()), value.GetTextValueName());
                        }
                        values.push_back(value.GetTextValue());
                    }
                }
                features[encodedId] = FilterRegistry_.BuildFeature(feature.GetId(), values);
            }
            THashMap<int, double> rankingFactors;
            for (const auto& pbRankingFactor: proto.GetRankingFactors()) {
                if (pbRankingFactor.HasValue()) {
                    rankingFactors[StringEncoder_.Encode(pbRankingFactor.GetId())] = pbRankingFactor.GetValue();
                }
            }
            THashMap<int, float> realtimeRankingFloatFeatures;
            for (const auto& pbRealtimeRankingFloatFeature: proto.GetRealtimeRankingFloatFeatures()) {
                if (pbRealtimeRankingFloatFeature.HasValue()) {
                    realtimeRankingFloatFeatures[StringEncoder_.Encode(pbRealtimeRankingFloatFeature.GetId())] = pbRealtimeRankingFloatFeature.GetValue();
                }
            }
            THashMap<int, ui32> realtimeRankingCatFeatures;
            for (const auto& pbRealtimeRankingCatFeature: proto.GetRealtimeRankingCatFeatures()) {
                if (pbRealtimeRankingCatFeature.HasValue()) {
                    realtimeRankingCatFeatures[StringEncoder_.Encode(pbRealtimeRankingCatFeature.GetId())] = CalcCatFeatureHash(pbRealtimeRankingCatFeature.GetValue());
                }
            }
            HotelAltayData_.UnlockedAddToNewData(proto.GetPermalink(), THotelAltayData{name, address, photoCount, geoId, displayedLocationGeoId, position, features, rankingFactors,
                                                                                       realtimeRankingFloatFeatures, realtimeRankingCatFeatures});
        };
        auto finish = [this](bool ok, bool /*initial*/) {
            if (ok) {
                HotelAltayData_.CommitNewData();
                auto currHotelAltayData = HotelAltayData_.GetData();
                FeaturesMetaInfos_.CommitNewData();
                size_t newNBytes = GetHashMapByteSizeWithoutElementAllocations(*currHotelAltayData);
                for (const auto& [permalink, altayData] : *currHotelAltayData) {
                    newNBytes += GetHashMapByteSizeWithoutElementAllocations(altayData.Features);
                    for (const auto& [_, feature] : altayData.Features) {
                        newNBytes += GetByteSizeWithoutSizeof(feature);
                    }
                    newNBytes += GetHashMapByteSizeWithoutElementAllocations(altayData.RankingFactors);
                    newNBytes += GetHashMapByteSizeWithoutElementAllocations(altayData.RealtimeRankingFloatFeatures);
                    newNBytes += GetHashMapByteSizeWithoutElementAllocations(altayData.RealtimeRankingCatFeaturesCatBoostEncoded);
                }
                Counters_.NBytesHotelAltayData = newNBytes;
                UpdateIndexData(false);
                ExtraPermalinkInfoStringCompressor_.TriggerRebuild();
            } else {
                HotelAltayData_.ResetNewData();
                FeaturesMetaInfos_.ResetNewData();
            }
        };
        GeoCounterRecordsTable_.SetCallbacks(conv, data, finish);
    }

    void TIndex::InitPriceRecordsTable() {
        auto conv = [](const NYT::TNode& node, NTravelProto::NGeoCounter::TPriceRecord* proto) {
            proto->SetPermalink(node["permalink"].IntCast<ui64>());
            proto->SetMinPricePerNight(node["min_price_per_night"].IntCast<ui32>());
            proto->SetMaxPricePerNight(node["max_price_per_night"].IntCast<ui32>());
            if (!node["checkin"].IsNull()) {
                proto->SetCheckIn(node["checkin"].AsString());
            }
            if (!node["checkout"].IsNull()) {
                proto->SetCheckOut(node["checkout"].AsString());
            }
        };
        auto data = [this](const NTravelProto::NGeoCounter::TPriceRecord& proto) {
            auto [priceInfoIt, _] = Prices_.UnlockedAddToNewData(proto.GetPermalink(), TPriceInfo());
            TPriceInfo& priceInfo = priceInfoIt->second;
            auto range = TPriceRange(proto.GetMinPricePerNight(), proto.GetMaxPricePerNight());
            if (!proto.HasCheckIn() || proto.GetCheckIn().empty()) {
                priceInfo.UpdateDefaultPrice(range);
            } else {
                auto checkin = NOrdinalDate::FromString(proto.GetCheckIn());
                auto checkout = NOrdinalDate::FromString(proto.GetCheckOut());
                for (auto date = checkin; date < checkout; date++) {
                    priceInfo.UpdateDayPrice(date, range);
                }
            }
        };
        auto finish = [this](bool ok, bool /*initial*/) {
            if (ok) {
                Prices_.CommitNewData();
                auto currPrices = Prices_.GetData();
                size_t newNBytes = GetHashMapByteSizeWithoutElementAllocations(*currPrices);
                for (const auto& [permalink, priceInfo] : *currPrices) {
                    newNBytes += GetByteSizeWithoutSizeof(priceInfo);
                }
                Counters_.NBytesPrices = newNBytes;
                UpdateIndexData(false);
            } else {
                Prices_.ResetNewData();
            }
        };
        PricesTable_.SetCallbacks(conv, data, finish);
    }

    void TIndex::InitHotelTraitsTable() {
        auto conv = [](const NYT::TNode& node, NTravelProto::NGeoCounter::THotelTraitsRecord* proto) {
            proto->SetPermalink(node["permalink"].IntCast<ui64>());
            proto->MutableFeature()->SetId(node["feature"]["id"].AsString());
            if (node["feature"].HasKey("value")) {
                proto->MutableFeature()->SetValue(node["feature"]["value"].AsBool());
            } else if (node["feature"].HasKey("enum_id")) {
                proto->MutableFeature()->SetEnumId(node["feature"]["enum_id"].AsString());
            } else {
                throw yexception() << "Feature with id " << proto->MutableFeature()->GetId() << " has no value nor enum_id. Permalink: " << proto->GetPermalink() << Endl;
            }
        };
        auto data = [this](const NTravelProto::NGeoCounter::THotelTraitsRecord& proto) {
            auto [currPermalinkTraitsIt, _] = HotelTraits_.UnlockedAddToNewData(proto.GetPermalink(), THotelTraits());
            THotelTraits& currPermalinkTraits = currPermalinkTraitsIt->second;
            if (proto.GetFeature().GetId() == "free_cancellation") {
                currPermalinkTraits.FreeCancellation = proto.GetFeature().GetValue();
            }
            if (proto.GetFeature().GetId() == "breakfast_included") {
                currPermalinkTraits.BreakfastIncluded = proto.GetFeature().GetValue();
            }
            if (proto.GetFeature().GetId() == HotelPansionFeatureBusinessId) {
                if (currPermalinkTraits.PansionAliases.Empty()) {
                    currPermalinkTraits.PansionAliases = TVector<int>();
                }
                currPermalinkTraits.PansionAliases.GetRef().push_back(StringEncoder_.Encode(proto.GetFeature().GetEnumId()));
            }
        };
        auto finish = [this](bool ok, bool /*initial*/) {
            if (ok) {
                HotelTraits_.CommitNewData();
                auto currTraits = HotelTraits_.GetData();

                size_t newNBytes = GetHashMapByteSizeWithoutElementAllocations(*currTraits);
                for (const auto& [permalink, hotelTraits] : *currTraits) {
                    newNBytes += GetByteSizeWithoutSizeof(hotelTraits);
                }
                Counters_.NBytesHotelTraits = newNBytes;

                UpdateIndexData(false);
            } else {
                HotelTraits_.ResetNewData();
            }
        };
        HotelTraitsTable_.SetCallbacks(conv, data, finish);
    }

    void TIndex::InitOfferBus() {
        OfferBus_.SetReadinessNotifier([this]() {
            INFO_LOG << "OfferBus is ready" << Endl;
            OfferBusReady_.Set();
            UpdateIndexData(false);
        });

        OfferBus_.Subscribe(ru::yandex::travel::hotels::TSearcherMessage(), [this](const TYtQueueMessage& busMessage) {
            ru::yandex::travel::hotels::TSearcherMessage message;
            if (!message.ParseFromString(busMessage.Bytes)) {
                throw yexception() << "Failed to parse TSearcherMessage record";
            }

            const auto& request = message.GetRequest();
            const auto& response = message.GetResponse();

            if (response.HasOffers()) {
                auto boyPartnerIds = BoyPartnerProvider_.GetBoYPartnerIds();
                auto boyPartnerIdsSet = THashSet<NTravelProto::EPartnerId>(boyPartnerIds->begin(), boyPartnerIds->end());

                auto key = TOfferBusDataKey(TBookingRange(NOrdinalDate::FromString(request.GetCheckInDate()), NOrdinalDate::FromString(request.GetCheckOutDate())), request.GetOccupancy());
                auto mapping = OriginalIdToPermalinkMapper_.GetMapping(THotelId::FromProto(request.GetHotelId()));
                if (mapping) {
                    TPermalink permalink = *mapping;
                    with_lock (OfferBusDataMutex_) {
                        auto& offers = OfferBusData_[permalink].Offers[key][request.GetHotelId().GetPartnerId()];
                        offers.ExpirationTimestamp = Now() + TDuration::Seconds(response.GetOffers().HasCacheTimeSec() ? response.GetOffers().GetCacheTimeSec().value() : 300);
                        offers.IsBoY = boyPartnerIdsSet.contains(request.GetHotelId().GetPartnerId());
                        offers.Offers.clear();
                        for (const auto& offer : response.GetOffers().GetOffer()) {
                            offers.Offers.emplace_back(
                                offer.GetPrice().GetAmount(),
                                offer.HasFreeCancellation() ? TMaybe<bool>(offer.GetFreeCancellation().value()) : TMaybe<bool>(),
                                offer.GetPansion());
                        }
                    }
                }
            }

            return true;
        });
        OfferBus_.Ignore(ru::yandex::travel::hotels::TPingMessage());
        OfferBus_.Ignore(NTravelProto::TTravellineCacheEvent());
        OfferBus_.Ignore(NTravelProto::NOfferBus::TOfferInvalidationMessage());
    }

    void TIndex::UpdateIndexData(bool onlyOfferBusData) {
        if (!HotelAltayData_.IsReady() || !Prices_.IsReady() || !HotelTraits_.IsReady() || !OfferBusReady_) {
            INFO_LOG << "Skipping index data update, because not all data is ready" << Endl;
            return;
        }
        TIndexBuilder(*this).UpdateIndexData(onlyOfferBusData);
    }

    void TIndex::OnReady() {
        Counters_.IsWaiting = 0;
        Counters_.IsReady = 1;
        if (OnReady_) {
            OnReady_();
        }
        IndexUpdaterThread_ = SystemThreadFactory()->Run([this]() {
            while (!IsStopping_) {
                UpdateIndexData(true);
                Sleep(IndexUpdateDelay_);
            }
        });
        CacheInvalidationThread_ = SystemThreadFactory()->Run([this]() {
            while (!IsStopping_) {
                RemoveOldCacheRecords();
                Sleep(CacheInvalidationDelay_);
            }
        });
    }

    void TIndex::RemoveOldCacheRecords() {
        with_lock (OfferBusDataMutex_) {
            TProfileTimer timer;
            auto now = Now();
            for (auto offerBusDataIt = OfferBusData_.begin(); offerBusDataIt != OfferBusData_.end();) {
                auto& offers = offerBusDataIt->second.Offers;
                for (auto cacheItemsIt = offers.begin(); cacheItemsIt != offers.end();) {
                    auto& cacheItems = cacheItemsIt->second;
                    for (auto cacheItemIt = cacheItems.begin(); cacheItemIt != cacheItems.end();) {
                        if (cacheItemIt->second.ExpirationTimestamp < now) {
                            cacheItems.erase(cacheItemIt++);
                        } else {
                            ++cacheItemIt;
                        }
                    }
                    if (cacheItems.empty()) {
                        offers.erase(cacheItemsIt++);
                    } else {
                        ++cacheItemsIt;
                    }
                }
                if (offers.empty()) {
                    OfferBusData_.erase(offerBusDataIt++);
                } else {
                    ++offerBusDataIt;
                }
            }

            auto newNBytes = GetHashMapByteSizeWithoutElementAllocations(OfferBusData_);
            for (const auto& [k, v] : OfferBusData_) {
                newNBytes += v.SlowCalcTotalByteSize() - sizeof(v);
            }
            Counters_.NBytesOfferBusData = newNBytes;

            INFO_LOG << "Done cache invalidation in " << timer.Get() << Endl;
        }
    }

    std::tuple<int, int> TIndex::GetBucketIndex(TPosition pos) const {
        int latBucket = Min(Max(static_cast<int>(lround(floor((pos.Lat + 90) / 180 * BucketCount_))), 0), BucketCount_ - 1);
        int lonBucket = Min(Max(static_cast<int>(lround(floor((pos.Lon + 180) / 360 * BucketCount_))), 0), BucketCount_ - 1);
        return {latBucket, lonBucket};
    }

    TIndex::TIndexBuilder::TIndexBuilder(TIndex& index)
        : Index_(index)
    {
    }

    void TIndex::TIndexBuilder::UpdateIndexData(bool onlyOfferBusData) {
        TProfileTimer timer;
        with_lock (Index_.DataUpdateMutex_) {
            INFO_LOG << "Updating index data (onlyOfferBusData = " << onlyOfferBusData << ")" << Endl;

            auto hotelAltayData = Index_.HotelAltayData_.GetData();
            auto prices = Index_.Prices_.GetData();
            auto hotelTraits = Index_.HotelTraits_.GetData();

            UpdateExtraPermalinkInfos(*hotelAltayData);
            UpdatePermalinkToBucket(*hotelAltayData);
            TMaybe<THashMap<TPermalink, TSortIndices>> sortIndices{};
            TMaybe<TVector<TVector<TVector<TPermalink>>>> perBucketPermalinks{};
            if (!onlyOfferBusData) {
                sortIndices = BuildSortIndices(*hotelAltayData, *prices);
                perBucketPermalinks = BuildPermalinkBuckets(*hotelAltayData, sortIndices.GetRef());
            }

            i64 memoryBytes = onlyOfferBusData ? Index_.Counters_.NRecordBytes.Val() : 0;
            i64 recordsCount = 0;
            TQueue<TBucketPtr> oldBuckets;
            for (int i = 0; i < Index_.BucketCount_; i++) {
                for (int j = 0; j < Index_.BucketCount_; j++) {
                    {
                        TBucketPtr oldBucket;
                        with_lock (Index_.BucketsMutex_) {
                            oldBucket = Index_.Buckets_.at(i).at(j);
                        }

                        TBucketPtr newBucket;
                        if (onlyOfferBusData) {
                            newBucket = BuildSingleBucketOnlyOc(*oldBucket, &memoryBytes);
                        } else {
                            newBucket = BuildSingleBucketFull(*hotelAltayData, *prices, *hotelTraits,
                                                              perBucketPermalinks.GetRef().at(i).at(j),
                                                              sortIndices.GetRef(), &memoryBytes);
                        }

                        oldBuckets.push(oldBucket);
                        recordsCount += newBucket->PermalinkInfos.size();

                        with_lock (Index_.BucketsMutex_) {
                            std::swap(Index_.Buckets_[i][j], newBucket);
                        }
                    }

                    {
                        TProfileTimer waitTimer{};
                        while (oldBuckets.size() > Index_.MaxBucketsInReleaseQueue_ || (i + 1 == Index_.BucketCount_ && j + 1 == Index_.BucketCount_ && !oldBuckets.empty())) {
                            auto refCount = oldBuckets.front().RefCount();
                            auto tooLong = waitTimer.Get() > Index_.MaxWaitBucketRelease_;
                            if (refCount == 1 || tooLong) {
                                if (tooLong) {
                                    INFO_LOG << "Waiting for old buckets to release for too long, releasing them, refCount: " << refCount << Endl;
                                }
                                oldBuckets.pop();
                            } else {
                                DEBUG_LOG << "Waiting for old buckets to release, size: " << oldBuckets.size() << ", refCount: " << refCount << Endl;
                                Sleep(Index_.WaitBucketReleaseDelay_);
                            }
                        }
                    }
                }
            }

            Index_.Counters_.NRecords = recordsCount;
            Index_.Counters_.NRecordBytes = memoryBytes;

            if (Index_.IsReady_.TrySet()) {
                Index_.OnReady();
            }

            INFO_LOG << "Update of index data done in " << timer.Get() << Endl;
        }
    }

    void TIndex::TIndexBuilder::EnrichWithOcData(TBucket& bucket, i64* memoryBytes) {
        with_lock (Index_.OfferBusDataMutex_) {
            for (auto& permalinkInfo : bucket.PermalinkInfos) {
                (*memoryBytes) -= GetByteSizeWithoutSizeof(permalinkInfo);
                auto offerBusDataIt = Index_.OfferBusData_.find(permalinkInfo.StaticPermalinkInfo.Permalink);
                if (offerBusDataIt != Index_.OfferBusData_.end()) {
                    permalinkInfo.OfferBusData = offerBusDataIt->second;
                } else {
                    permalinkInfo.OfferBusData = TMaybe<TOfferBusData>();
                }
                (*memoryBytes) += GetByteSizeWithoutSizeof(permalinkInfo);
            }
        }
    }

    void TIndex::TIndexBuilder::UpdatePermalinkToBucket(const THashMap<TPermalink, THotelAltayData>& hotelAltayData) {
        TPermalinkToBucketPtr newPermalinkToBucket = MakeAtomicShared<TPermalinkToBucket>();
        newPermalinkToBucket->reserve(hotelAltayData.size());
        for (const auto& [permalink, hotelDataItem]: hotelAltayData) {
            auto [latBucket, lonBucket] = Index_.GetBucketIndex(hotelDataItem.Position);
            (*newPermalinkToBucket)[permalink] = std::make_pair(latBucket, lonBucket);
        }
        with_lock (Index_.BucketsMutex_) {
            std::swap(newPermalinkToBucket, Index_.PermalinkToBucket_);
        }
    }

    void TIndex::TIndexBuilder::UpdateExtraPermalinkInfos(const THashMap<TPermalink, THotelAltayData>& hotelAltayData) {
        Index_.ExtraPermalinkInfos_.ResetNewData();

        size_t allocatedBytes = 0;
        for (const auto& [permalink, hotelDataItem] : hotelAltayData) {
            Index_.ExtraPermalinkInfos_.UnlockedAddToNewData(permalink, TExtraPermalinkInfo{
                hotelDataItem.Name,
                hotelDataItem.Address,
                hotelDataItem.PhotoCount,
                hotelDataItem.GeoId,
                hotelDataItem.DisplayedLocationGeoId,
                hotelDataItem.RealtimeRankingFloatFeatures,
                hotelDataItem.RealtimeRankingCatFeaturesCatBoostEncoded,
            });

            allocatedBytes += GetHashMapByteSizeWithoutElementAllocations(hotelDataItem.RealtimeRankingFloatFeatures);
            allocatedBytes += GetHashMapByteSizeWithoutElementAllocations(hotelDataItem.RealtimeRankingCatFeaturesCatBoostEncoded);
        }

        Index_.ExtraPermalinkInfos_.CommitNewData(); // Old data is stored here till the next iteration (call of ResetNewData) to avoid freeing this memory in one of response handlers

        Index_.Counters_.NBytesExtraPermalinkInfos = GetHashMapByteSizeWithoutElementAllocations(*Index_.ExtraPermalinkInfos_.GetData()) + allocatedBytes;
    }

    THashMap<TPermalink, TSortIndices> TIndex::TIndexBuilder::BuildSortIndices(const THashMap<TPermalink, THotelAltayData>& hotelAltayData,
                                                                               const THashMap<TPermalink, TPriceInfo>& prices) {
        TPriceInfo emptyPrice{};
        TVector<THotelSortData> orderedHotelAltayData;
        orderedHotelAltayData.reserve(hotelAltayData.size());
        for (const auto& [permalink, hotelAltayDataRecord] : hotelAltayData) {
            auto priceIt = prices.find(permalink);
            auto priceInfoPtr = priceIt == prices.end() ? &emptyPrice : &priceIt->second;
            orderedHotelAltayData.emplace_back(THotelSortData{permalink, &hotelAltayDataRecord, priceInfoPtr});
        }

        THashMap<TPermalink, TSortIndices> sortIndices;
        for (const auto& sortType: Index_.SortTypeRegistry_.GetSortTypes()) {
            Sort(orderedHotelAltayData, sortType.GetCmpSortInIndex());
            for (int i = 0; i < static_cast<int>(orderedHotelAltayData.size()); i++) {
                sortIndices[orderedHotelAltayData[i].Permalink][sortType.Id] = i;
            }
        }

        return sortIndices;
    }

    TVector<TVector<TVector<TPermalink>>> TIndex::TIndexBuilder::BuildPermalinkBuckets(const THashMap<TPermalink, THotelAltayData>& hotelAltayData,
                                                                                       const THashMap<TPermalink, TSortIndices>& sortIndices) {
        TVector<TVector<TVector<TPermalink>>> perBucketPermalinks;
        perBucketPermalinks.resize(Index_.BucketCount_);
        for (int i = 0; i < Index_.BucketCount_; i++) {
            perBucketPermalinks.at(i).resize(Index_.BucketCount_);
        }

        for (const auto& [permalink, hotelDataItem] : hotelAltayData) {
            auto [latBucket, lonBucket] = Index_.GetBucketIndex(hotelDataItem.Position);
            perBucketPermalinks[latBucket][lonBucket].push_back(permalink);
        }

        auto defaultSortId = Index_.SortTypeRegistry_.GetDefaultSortType().Id;
        for (int i = 0; i < Index_.BucketCount_; i++) {
            for (int j = 0; j < Index_.BucketCount_; j++) {
                // Sorting for faster work with defaultSortId
                Sort(perBucketPermalinks.at(i).at(j),
                     [&sortIndices, defaultSortId](const TPermalink& lhs, const TPermalink& rhs) {
                         return sortIndices.at(lhs)[defaultSortId] < sortIndices.at(rhs)[defaultSortId];
                     });
            }
        }

        return perBucketPermalinks;
    }

    void TIndex::TIndexBuilder::EnrichWithStaticData(const THashMap<TPermalink, THotelAltayData>& hotelAltayData,
                                                     const THashMap<TPermalink, TPriceInfo>& prices,
                                                     const THashMap<TPermalink, THotelTraits>& hotelTraits,
                                                     const TVector<TPermalink>& currentBucketPermalinks,
                                                     const THashMap<TPermalink, TSortIndices>& sortIndices,
                                                     TBucket& bucket,
                                                     i64* memoryBytes) {
        bucket.PermalinkInfos.reserve(currentBucketPermalinks.size());
        bucket.PermalinkToIndex.reserve(currentBucketPermalinks.size());
        for (const auto& permalink: currentBucketPermalinks) {
            const auto& hotelAltayDataRecord = hotelAltayData.at(permalink);
            auto position = hotelAltayDataRecord.Position;
            auto features = hotelAltayDataRecord.Features;

            if (hotelTraits.contains(permalink)) { // todo (mpivko): remove hotelTraits?
                if (hotelTraits.at(permalink).BreakfastIncluded.Defined()) {
                    auto value = *hotelTraits.at(permalink).BreakfastIncluded.Get();
                    auto feature = TFeature(TVector<int>({Index_.StringEncoder_.Encode(value ? "1" : "0")}));
                    features[Index_.StringEncoder_.Encode("hotel_breakfast_included")] = feature;
                }
                if (hotelTraits.at(permalink).FreeCancellation.Defined()) {
                    auto value = *hotelTraits.at(permalink).FreeCancellation.Get();
                    auto feature = TFeature(TVector<int>({Index_.StringEncoder_.Encode(value ? "1" : "0")}));
                    features[Index_.StringEncoder_.Encode("hotel_free_cancellation")] = feature;
                }
                if (hotelTraits.at(permalink).PansionAliases.Defined()) {
                    auto value = *hotelTraits.at(permalink).PansionAliases.Get();
                    auto feature = TFeature(value);
                    features[Index_.StringEncoder_.Encode(HotelPansionFeatureBusinessId)] = feature;
                }
            }

            auto permalinkInfo = TStaticPermalinkInfo(permalink,
                                                      position,
                                                      std::move(features),
                                                      TBitMask());

            TBitMask predefinedFiltersResults;
            if (Index_.PredefinedFilters_.size() > predefinedFiltersResults.size()) {
                ythrow yexception() << "Too many predefined filters";
            }
            for (size_t i = 0; i < Index_.PredefinedFilters_.size(); i++) {
                if (Index_.PredefinedFilters_[i].Filter->IsPassingFilter(TFilteringPermalinkInfo{permalinkInfo, TPriceInfo(), TMaybe<TOfferBusData>(), {}})) {
                    predefinedFiltersResults.set(i);
                }
            }
            permalinkInfo.PredefinedFiltersResults = predefinedFiltersResults;

            bucket.PermalinkInfos.emplace_back(TFilteringPermalinkInfo{permalinkInfo,
                                                                       prices.contains(permalink) ? prices.at(permalink) : TPriceInfo(),
                                                                       TMaybe<TOfferBusData>(),
                                                                       sortIndices.at(permalink)});
            bucket.PermalinkToIndex[permalink] = static_cast<int>(bucket.PermalinkInfos.size()) - 1;

            (*memoryBytes) += GetByteSizeWithoutSizeof(bucket.PermalinkInfos.back());
        }
        bucket.PermalinkInfos.shrink_to_fit();

        bucket.Permutations.resize(Index_.SortTypeRegistry_.GetSortTypes().size());
        for (const auto& sortType : Index_.SortTypeRegistry_.GetSortTypes()) {
            auto& permutations = bucket.Permutations.at(sortType.Id);
            permutations.resize(bucket.PermalinkInfos.size());
            for (size_t i = 0; i < bucket.PermalinkInfos.size(); i++) {
                permutations[i] = static_cast<int>(i);
            }
            Sort(permutations, [&bucket, &sortType](const int& lhs, const int& rhs) {
                return bucket.PermalinkInfos.at(lhs).SortIndices[sortType.Id] < bucket.PermalinkInfos.at(rhs).SortIndices[sortType.Id];
            });
        }

        (*memoryBytes) += GetVectorByteSizeWithoutElementAllocations(bucket.PermalinkInfos);
        // todo (mpivko): calc size of other structures
    }

    TIndex::TBucketPtr TIndex::TIndexBuilder::BuildSingleBucketFull(const THashMap<TPermalink, THotelAltayData>& hotelAltayData,
                                                                    const THashMap<TPermalink, TPriceInfo>& prices,
                                                                    const THashMap<TPermalink, THotelTraits>& hotelTraits,
                                                                    const TVector<TPermalink>& currentBucketPermalinks,
                                                                    const THashMap<TPermalink, TSortIndices>& sortIndices,
                                                                    i64* memoryBytes) {
        auto newBucket = MakeAtomicShared<TBucket>();
        EnrichWithStaticData(hotelAltayData, prices, hotelTraits, currentBucketPermalinks, sortIndices, *newBucket, memoryBytes);
        EnrichWithOcData(*newBucket, memoryBytes);
        return newBucket;
    }

    TIndex::TBucketPtr TIndex::TIndexBuilder::BuildSingleBucketOnlyOc(const TBucket& oldBucket,
                                                                      i64* memoryBytes) {
        auto newBucket = MakeAtomicShared<TBucket>();
        *newBucket = oldBucket;
        EnrichWithOcData(*newBucket, memoryBytes);
        return newBucket;
    }

    static std::initializer_list<int> g_NRecordsScannedBuckets = {0, 1000, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 9000, 10000};
    static std::initializer_list<int> g_NBucketsScannedBuckets = {0, 1, 5, 10, 20, 30, 40, 50, 75, 100, 500, 750, 1000};
    static std::initializer_list<int> g_LatencyBuckets = {0, 1, 5, 10, 20, 50, 100, 250, 500, 1000};

    TIndex::TCounters::TCounters()
        : NRecordsScanned(g_NRecordsScannedBuckets)
        , NRecordsAfterGeoFiltering(g_NRecordsScannedBuckets)
        , NRecordsAfterParameterFiltering(g_NRecordsScannedBuckets)
        , NBucketsScanned(g_NBucketsScannedBuckets)
        , ProcessTimeMs(g_LatencyBuckets)
    {
    }

    void TIndex::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
        ct->insert(MAKE_COUNTER_PAIR(IsWaiting));
        ct->insert(MAKE_COUNTER_PAIR(IsReady));
        ct->insert(MAKE_COUNTER_PAIR(NRecords));
        ct->insert(MAKE_COUNTER_PAIR(NRecordBytes));

        ct->insert(MAKE_COUNTER_PAIR(NBytesOfferBusData));
        ct->insert(MAKE_COUNTER_PAIR(NBytesHotelAltayData));
        ct->insert(MAKE_COUNTER_PAIR(NBytesPrices));
        ct->insert(MAKE_COUNTER_PAIR(NBytesHotelTraits));
        ct->insert(MAKE_COUNTER_PAIR(NBytesExtraPermalinkInfos));

        NRecordsScanned.QueryCounters("NRecordsScanned", "", ct);
        NRecordsAfterGeoFiltering.QueryCounters("NRecordsAfterGeoFiltering", "", ct);
        NRecordsAfterParameterFiltering.QueryCounters("NRecordsAfterParameterFiltering", "", ct);
        NBucketsScanned.QueryCounters("NBucketsScanned", "", ct);
        ProcessTimeMs.QueryCounters("ProcessTimeMs", "", ct);

        ct->insert(MAKE_COUNTER_PAIR(NNotPrecalculatedFilterWarns));
    }
}
