#pragma once

#include "string_encoder.h"
#include "data.h"
#include "features_filtering.h"
#include "filter_registry.h"
#include "additional_filter.h"
#include "sorts.h"
#include "boy_partner_provider.h"
#include "reloadable_storage.h"

#include <travel/hotels/proto/offercache_api/commons.pb.h>
#include <travel/hotels/proto/hotel_filters_config/hotel_filters.pb.h>
#include <travel/hotels/proto/app_config/yt_table_cache.pb.h>
#include <travel/hotels/geocounter/proto/geocounter_data.pb.h>
#include <travel/hotels/geocounter/proto/config.pb.h>

#include <travel/hotels/lib/cpp/ages/ages.h>
#include <travel/hotels/lib/cpp/mon/tools.h>
#include <travel/hotels/lib/cpp/permalink_mappers/mappers/original_id_to_permalink_mapper.h>
#include <travel/hotels/lib/cpp/yt/table_cache.h>
#include <travel/hotels/lib/cpp/yt/queue_reader.h>

namespace NTravel::NGeoCounter {
    struct TCountResults {
        struct TAdditionalFilterCount {
            int UniqueId;
            int Count;
        };

        struct TPriceResults {
            ui32 MinPriceEstimate;
            ui32 MaxPriceEstimate;
            TVector<int> HistogramBounds;
            TVector<int> HistogramCounts;
        };

        TCountResults()
            : MatchedCount(0)
            , TotalCount(0)
        {
        }
        int MatchedCount;
        int TotalCount;
        TVector<TAdditionalFilterCount> AdditionalFilterCounts;
        TPriceResults PriceResults;
    };

    struct TFeatureMetaInfo {
        enum EType {
            Boolean = 0,
            Numeric = 1,
            Range = 2,
            Other = 3,
        };

        EType Type;
        TString Name;
        THashMap<int, TString> ValueNames;
    };

    class TIndex {
    public:
        explicit TIndex(const NTravelProto::NHotelFiltersConfig::TFilterInfoConfig& filterInfoConfig,
                        const NTravelProto::NAppConfig::TYtTableCacheConfig& geoCounterRecordsTableConfig,
                        const NTravelProto::NAppConfig::TYtTableCacheConfig& priceRecordsTableConfig,
                        const NTravelProto::NAppConfig::TYtTableCacheConfig& hotelTraitsTableConfig,
                        const NTravelProto::NAppConfig::TConfigYtQueueReader& offerBusConfig,
                        const NTravelProto::NAppConfig::TYtTableCacheConfig& originalIdToPermalinkMapper,
                        const NTravelProto::NAppConfig::TConfigStringCompressor& stringCompressorConfig,
                        const NTravelProto::NGeoCounter::TConfig::TIndexOptions& indexOptionsConfig,
                        const TBoyPartnerProvider& boyPartnerProvider,
                        const TSortTypeRegistry& sortTypeRegistry,
                        TStringEncoder& stringEncoder);

        void RegisterCounters(NMonitor::TCounterSource& source);
        void Start();
        void Stop();
        void OnReady(std::function<void(void)> handler);
        bool IsReady() const;

        TCountResults GetCountsWithOfferBusData(TBoundingBox boundingBox,
                                                TBookingRange bookingRange,
                                                const TVector<TBasicFilterGroup>& selectedFiltersGroups,
                                                const TVector<TAdditionalFilter>& additionalFilters,
                                                bool disablePriceCounts) const;

        TCountResults GetCounts(TBoundingBox boundingBox,
                                TBookingRange bookingRange,
                                const TVector<TBasicFilterGroup>& selectedFiltersGroups,
                                const TVector<TAdditionalFilter>& additionalFilters,
                                bool disablePriceCounts,
                                bool allowExtrapolation) const;

        THotelsResults GetHotels(TBoundingBox boundingBox,
                                 const TVector<TBasicFilterGroup>& selectedFiltersGroups,
                                 const TSortTypeWithContext& sortTypeWithContext,
                                 TBookingRange bookingRange,
                                 const TString& occupancy,
                                 TMaybe<TPermalink> topHotelPermalink,
                                 size_t skip,
                                 size_t limit) const;

        TMaybe<TOfferBusData> GetPermalinkInfo(TPermalink permalink) const;
        TMaybe<TFilteringPermalinkInfo> GetPermalinkFilteringInfo(TPermalink permalink) const;
        TMaybe<TExtraPermalinkInfo> GetExtraPermalinkInfo(TPermalink permalink) const;

    private:
        struct TCounters: public NMonitor::TCounterSource {
            TCounters();

            NMonitor::TCounter IsWaiting;
            NMonitor::TCounter IsReady;

            NMonitor::TCounter NRecords;
            NMonitor::TCounter NRecordBytes;

            NMonitor::TCounter NBytesOfferBusData;
            NMonitor::TCounter NBytesHotelAltayData;
            NMonitor::TCounter NBytesPrices;
            NMonitor::TCounter NBytesHotelTraits;
            NMonitor::TCounter NBytesExtraPermalinkInfos;

            NMonitor::TDerivHistogramCounter NRecordsScanned;
            NMonitor::TDerivHistogramCounter NRecordsAfterGeoFiltering;
            NMonitor::TDerivHistogramCounter NRecordsAfterParameterFiltering;
            NMonitor::TDerivHistogramCounter NBucketsScanned;

            NMonitor::TDerivHistogramCounter ProcessTimeMs;

            NMonitor::TDerivCounter NNotPrecalculatedFilterWarns;

            void QueryCounters(NMonitor::TCounterTable* ct) const override;
        };

        struct TPreprocessedFilters {
            TVector<const TBasicFilterGroup*> PredefinedSelectedFiltersGroups;
            TVector<const TBasicFilterGroup*> DynamicSelectedFiltersGroups;
            TVector<TVector<int>> SelectedFiltersPredefinedIndices;
        };

        struct TBucket {
            TVector<TFilteringPermalinkInfo> PermalinkInfos;
            TVector<TVector<int>> Permutations;
            THashMap<TPermalink, int> PermalinkToIndex;
        };

        typedef TAtomicSharedPtr<TBucket> TBucketPtr;
        typedef TVector<TVector<TBucketPtr>> TBuckets;
        typedef THashMap<TPermalink, std::pair<int, int>> TPermalinkToBucket;
        typedef TAtomicSharedPtr<TPermalinkToBucket> TPermalinkToBucketPtr;

        class TIndexBuilder {
        public:
            TIndexBuilder(TIndex& index);
            void UpdateIndexData(bool onlyOfferBusData);
        private:
            TIndex& Index_;

            void EnrichWithOcData(TBucket& bucket, i64* memoryBytes);

            void UpdatePermalinkToBucket(const THashMap<TPermalink, THotelAltayData>& hotelAltayData);

            void UpdateExtraPermalinkInfos(const THashMap<TPermalink, THotelAltayData>& hotelAltayData);

            THashMap<TPermalink, TSortIndices> BuildSortIndices(const THashMap<TPermalink, THotelAltayData>& hotelAltayData,
                                                                const THashMap<TPermalink, TPriceInfo>& prices);

            TVector<TVector<TVector<TPermalink>>> BuildPermalinkBuckets(const THashMap<TPermalink, THotelAltayData>& hotelAltayData,
                                                                        const THashMap<TPermalink, TSortIndices>& sortIndices);

            void EnrichWithStaticData(const THashMap<TPermalink, THotelAltayData>& hotelAltayData,
                                      const THashMap<TPermalink, TPriceInfo>& prices,
                                      const THashMap<TPermalink, THotelTraits>& hotelTraits,
                                      const TVector<TPermalink>& currentBucketPermalinks,
                                      const THashMap<TPermalink, TSortIndices>& sortIndices,
                                      TBucket& bucket,
                                      i64* memoryBytes);

            TBucketPtr BuildSingleBucketFull(const THashMap<TPermalink, THotelAltayData>& hotelAltayData,
                                             const THashMap<TPermalink, TPriceInfo>& prices,
                                             const THashMap<TPermalink, THotelTraits>& hotelTraits,
                                             const TVector<TPermalink>& currentBucketPermalinks,
                                             const THashMap<TPermalink, TSortIndices>& sortIndices,
                                             i64* memoryBytes);

            TBucketPtr BuildSingleBucketOnlyOc(const TBucket& oldBucket,
                                               i64* memoryBytes);
        };

        TCountResults::TPriceResults BuildPriceResults(const TBookingRange& bookingRange, const TVector<TPriceRange>& priceRanges) const;
        void ProcessRecords(TBoundingBox boundingBox, const TSortType& sortType, bool useUnorderedVersion, TMaybe<TPermalink> topHotelPermalink, const std::function<bool(const TFilteringPermalinkInfo&)>& handler) const;
        void ProcessRecordsWithFilteringForGetHotels(TBoundingBox boundingBox,
                                                     const TVector<TBasicFilterGroup>& selectedFiltersGroups,
                                                     const TSortType& sortType,
                                                     TMaybe<TPermalink> topHotelPermalink,
                                                     size_t skip,
                                                     TMaybe<size_t> limit,
                                                     TMaybe<std::function<bool(const TFilteringPermalinkInfo&)>> sortThresholdFilter,
                                                     const std::function<void(const TFilteringPermalinkInfo&)>& processHotel) const;
        void InitBuckets(TBuckets* buckets) const;

        std::tuple<int, int> GetBucketIndex(TPosition pos) const;
        TAdditionalFilter::EType ConvertFilterGroupTypeEnum(const NTravelProto::NHotelFiltersConfig::TFilterInfoConfig::TBasicFilterGroup::EBasicFilterGroupType& type) const;
        void InitPredefinedFilters();
        void InitGeoCounterRecordsTable();
        void InitPriceRecordsTable();
        void InitHotelTraitsTable();
        void InitOfferBus();
        void UpdateIndexData(bool onlyOfferBusData);
        void OnReady();
        void RemoveOldCacheRecords();
        THotelsResults::THotelResult BuildHotelResult(const TFilteringPermalinkInfo& info,
                                                      int starsFeatureId,
                                                      int categoryFeatureId,
                                                      int ratingFeatureId,
                                                      const TAtomicSharedPtr<THashMap<TPermalink, TExtraPermalinkInfo>>& extraPermalinkInfos,
                                                      const TAtomicSharedPtr<THashMap<int, TFeatureMetaInfo>>& featureMetaInfo,
                                                      TMaybe<TPermalink> topHotelPermalink) const;
        TMaybe<TString> GetFirstFeatureValue(const TFilteringPermalinkInfo& info, int featureId) const;
        TPreprocessedFilters PreprocessFilters(const TVector<TBasicFilterGroup>& selectedFiltersGroups) const;

        const int BucketCount_ = 30;
        const TDuration IndexUpdateDelay_ = TDuration::Seconds(5);
        const TDuration CacheInvalidationDelay_ = TDuration::Minutes(1);

        mutable TCounters Counters_;

        NTravelProto::NHotelFiltersConfig::TFilterInfoConfig FilterInfoConfig_;
        TStringEncoder& StringEncoder_;
        TFilterRegistry FilterRegistry_;

        TYtTableCache<NTravelProto::NGeoCounter::TGeoCounterRecord> GeoCounterRecordsTable_;
        TYtTableCache<NTravelProto::NGeoCounter::TPriceRecord> PricesTable_;
        TYtTableCache<NTravelProto::NGeoCounter::THotelTraitsRecord> HotelTraitsTable_;

        TYtQueueReader OfferBus_;
        TAtomicFlag OfferBusStarted_;
        TAtomicFlag OfferBusReady_;
        TMutex OfferBusDataMutex_;
        THashMap<TPermalink, TOfferBusData> OfferBusData_;
        THolder<IThreadFactory::IThread> IndexUpdaterThread_;
        THolder<IThreadFactory::IThread> CacheInvalidationThread_;
        NPermalinkMappers::TOriginalIdToPermalinkMapper OriginalIdToPermalinkMapper_;
        const TBoyPartnerProvider& BoyPartnerProvider_;

        TAtomicFlag IsStopping_;
        TAtomicFlag IsReady_;
        std::function<void(void)> OnReady_;

        TReloadableStorage<TPermalink, THotelAltayData> HotelAltayData_;
        TReloadableStorage<TPermalink, TPriceInfo> Prices_;
        TReloadableStorage<TPermalink, THotelTraits> HotelTraits_;
        TReloadableStorage<TPermalink, TExtraPermalinkInfo> ExtraPermalinkInfos_;
        TStringCompressor ExtraPermalinkInfoStringCompressor_;

        TReloadableStorage<int, TFeatureMetaInfo> FeaturesMetaInfos_;

        TMutex BucketsMutex_;
        TBuckets Buckets_;
        TPermalinkToBucketPtr PermalinkToBucket_;

        TMutex DataUpdateMutex_;

        TBitMask PredefinedFiltersSingleMask_;
        TBitMask PredefinedFiltersAndMask_;
        TBitMask PredefinedFiltersOrMask_;
        THashMap<int, size_t> PredefinedFilterIndById_;
        THashMap<size_t, TBitMask> OnlyCurrGroupNotPassingMask_;
        TVector<TAdditionalFilter> PredefinedFilters_;

        const int MaxPredefinedMatchesForDynamicFilters_;
        const int MinHotelsWithOfferBusDataForExtrapolation_;
        const TDuration WaitBucketReleaseDelay_;
        const TDuration MaxWaitBucketRelease_;
        const size_t MaxBucketsInReleaseQueue_;

        THashMap<TString, int> StarsMapping_;

        const TSortTypeRegistry& SortTypeRegistry_;
    };
}
