#include "regions_service.h"
#include "filter_registry.h"

#include <travel/hotels/lib/cpp/util/sizes.h>

#include <util/generic/serialized_enum.h>


namespace NTravel::NGeoCounter {
    TRegionsService::TRegionsService(
        const NTravelProto::NAppConfig::TYtTableCacheConfig& regionsTableConfig)
        : RegionsTable_("Regions", regionsTableConfig)
        , OldRegionsByTravelGeoId_(MakeAtomicShared<TRegionByTravelGeoId>())
        , RegionsByTravelGeoId_(MakeAtomicShared<TRegionByTravelGeoId>())
        , NewRegionsByTravelGeoId_(MakeAtomicShared<TRegionByTravelGeoId>())
    {
        InitRegionsTable();
    }

    void TRegionsService::RegisterCounters(NMonitor::TCounterSource& source) {
        RegionsTable_.RegisterCounters(source, "Regions");
        source.RegisterSource(&Counters_, "RegionsService");
    }

    void TRegionsService::Start() {
        Counters_.IsWaiting = 1;
        RegionsTable_.Start();
    }

    void TRegionsService::Stop() {
        if (IsStopping_.TrySet()) {
            RegionsTable_.Stop();
        }
    }

    void TRegionsService::SetOnReady(std::function<void(void)> handler) {
        OnReady_ = std::move(handler);
    }

    bool TRegionsService::IsReady() const {
        return RegionsTable_.IsReady();
    }

    TMaybe<TBoundingBox> TRegionsService::GetRegionBboxByTravelGeoId(int geoId) const {
        TAtomicSharedPtr<TRegionByTravelGeoId> currRegionsByTravelGeoId;
        with_lock (RegionsByTravelGeoIdMutex_) {
            currRegionsByTravelGeoId = RegionsByTravelGeoId_;
        }
        auto regionIt = currRegionsByTravelGeoId->find(geoId);
        if (regionIt == currRegionsByTravelGeoId->end()) {
            return {};
        }
        return regionIt->second.Bounds;
    }

    void TRegionsService::InitRegionsTable() {
        auto conv = [](const NYT::TNode& node, NTravelProto::NGeoCounter::TRegionRecord* proto) {
            if (!node["TravelGeoId"].IsNull()) {
                proto->SetTravelGeoId(node["TravelGeoId"].AsInt64());
            } else {
                proto->SetTravelGeoId(node["GeoId"].AsInt64()); // Backward compatibility
            }
            if (!node["GeoId"].IsNull()) {
                proto->SetGeoId(node["GeoId"].AsInt64());
            }
            proto->MutableBounds()->MutableLowerLeft()->SetLat(node["LowerLeftLat"].AsDouble());
            proto->MutableBounds()->MutableLowerLeft()->SetLon(node["LowerLeftLon"].AsDouble());
            proto->MutableBounds()->MutableUpperRight()->SetLat(node["UpperRightLat"].AsDouble());
            proto->MutableBounds()->MutableUpperRight()->SetLon(node["UpperRightLon"].AsDouble());
            if (!node["CenterLat"].IsNull() && !node["CenterLon"].IsNull()) {
                proto->MutableCenter()->SetLat(node["CenterLat"].AsDouble());
                proto->MutableCenter()->SetLon(node["CenterLon"].AsDouble());
            }
            if (!node["Geometry"].IsNull()) {
                proto->SetGeometry(node["Geometry"].AsString());
            }
        };
        auto data = [this](const NTravelProto::NGeoCounter::TRegionRecord& proto) {
            auto bounds = TBoundingBox(TPosition(proto.GetBounds().GetLowerLeft().GetLat(), proto.GetBounds().GetLowerLeft().GetLon()),
                                       TPosition(proto.GetBounds().GetUpperRight().GetLat(), proto.GetBounds().GetUpperRight().GetLon()));
            TMaybe<TPosition> center{};
            if (proto.HasCenter()) {
                center = TPosition(proto.GetCenter().GetLat(), proto.GetCenter().GetLon());
            }
            (*NewRegionsByTravelGeoId_).emplace(proto.GetTravelGeoId(), TRegion{proto.GetTravelGeoId(), bounds, center, proto.GetGeometry()});
        };
        auto finish = [this](bool ok, bool initial) {
            if (ok) {
                TAtomicSharedPtr<TRegionByTravelGeoId> currRegionsByTravelGeoId;
                with_lock (RegionsByTravelGeoIdMutex_) {
                    swap(NewRegionsByTravelGeoId_, RegionsByTravelGeoId_);
                    swap(OldRegionsByTravelGeoId_, NewRegionsByTravelGeoId_); // Will clear NewRegionsByTravelGeoId_ several lines later
                    currRegionsByTravelGeoId = RegionsByTravelGeoId_;
                }
                size_t newNBytes = GetHashMapByteSizeWithoutElementAllocations(*currRegionsByTravelGeoId);
                for (const auto& [geoId, regionData] : *currRegionsByTravelGeoId) {
                    newNBytes += GetByteSizeWithoutSizeof(regionData.Geometry);
                }
                Counters_.NRecords = currRegionsByTravelGeoId->size();
                Counters_.NRecordBytes = newNBytes;
                if (initial) {
                    OnReady();
                }
            }
            NewRegionsByTravelGeoId_ = MakeAtomicShared<TRegionByTravelGeoId>();
        };
        RegionsTable_.SetCallbacks(conv, data, finish);
    }

    void TRegionsService::OnReady() {
        Counters_.IsWaiting = 0;
        Counters_.IsReady = 1;
        if (OnReady_) {
            OnReady_();
        }
    }

    void TRegionsService::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
        ct->insert(MAKE_COUNTER_PAIR(IsWaiting));
        ct->insert(MAKE_COUNTER_PAIR(IsReady));
        ct->insert(MAKE_COUNTER_PAIR(NRecords));
        ct->insert(MAKE_COUNTER_PAIR(NRecordBytes));
    }
}
