#include "service.h"

#include <travel/hotels/lib/cpp/scheduler/scheduler.h>
#include <travel/hotels/lib/cpp/data/data.h>

#include <travel/hotels/lib/cpp/ages/ages.h>

#include <util/string/split.h>
#include <library/cpp/json/json_writer.h>

namespace NTravel::NGeoCounter {
    TService::TService(const NTravelProto::NGeoCounter::TConfig& config)
        : Config_(config)
        , BoyPartnerProvider_(config.GetYtConfigPartners())
        , FiltersConfigReader_()
        , RestartDetector_(config.GetOther().GetRestartDetectorStateFile())
        , FilterRegistry_(StringEncoder_)
        , UserSegmentsRegistry_(TUserSegmentsRegistry::BuildRegistry())
        , SortTypeRegistry_(TSortTypeRegistry::Build(StringEncoder_, config, UserSegmentsRegistry_))
        , Index_(FiltersConfigReader_.Read(),
                 config.GetGeoCounterRecordsTable(),
                 config.GetPricesTable(),
                 config.GetHotelTraitsTable(),
                 config.GetOfferBus(),
                 config.GetOriginalIdToPermalinkMapper(),
                 config.GetExtraPermalinkInfoCompressor(),
                 config.GetIndexOptions(),
                 BoyPartnerProvider_,
                 SortTypeRegistry_,
                 StringEncoder_)
        , GrpcServer_(NGrpc::TAsyncServerConfig(config.GetGrpcServer().GetBindAddress(), Config_.GetGrpcServer().GetReplyThreads()))
        , Counters_(*this)
        , CountersPerRequestType_({"request_type"})
        , OfferCacheClient_(config.GetOfferCacheClient())
        , PromoServiceClient_(config.GetPromoServiceClient())
        , RegionsService_(config.GetRegionsTable())
        , TvmService_(config.GetTvmService())
        , BigbClient_(TvmService_, config.GetBigbClient())
        , HotelSearchService_(OfferCacheClient_, PromoServiceClient_, Index_, StringEncoder_, RegionsService_, SortTypeRegistry_, FilterRegistry_, BigbClient_, UserSegmentsRegistry_)
        , ReqAnsLogger_("ReqAnsLogger", config.GetReqAnsLogger())
    {
        CountersPage_.AddToHttpService(MonWebService_);

        CountersPage_.RegisterSource(&Counters_, "Service");
        CountersPage_.RegisterSource(&CountersPerRequestType_, "PerRequestType");
        BoyPartnerProvider_.RegisterCounters(CountersPage_);
        RestartDetector_.RegisterCounters(CountersPage_);
        Http_.RegisterCounters(CountersPage_);
        Index_.RegisterCounters(CountersPage_);
        StringEncoder_.RegisterCounters(CountersPage_);
        GrpcServer_.RegisterCounters(CountersPage_, "GrpcServer");
        OfferCacheClient_.RegisterCounters(CountersPage_);
        PromoServiceClient_.RegisterCounters(CountersPage_);
        HotelSearchService_.RegisterCounters(CountersPage_);
        RegionsService_.RegisterCounters(CountersPage_);
        ReqAnsLogger_.RegisterCounters(CountersPage_);
        TvmService_.RegisterCounters(CountersPage_);
        BigbClient_.RegisterCounters(CountersPage_);
        CountersPage_.RegisterSource(&MemUsageCounters_, "MemUsage");

        Http_.AddHandler("/setlog", NHttp::Local(), this, &TService::OnSetLogLevel);
        Http_.AddHandler("/reopen_reqans", NHttp::Local(), this, &TService::OnReopenReqAnsLog);
        Http_.AddHandler("/shutdown", NHttp::Local(), this, &TService::OnShutdown);
        Http_.AddHandler("/ping", NHttp::ExternalWithoutTvm(), this, &TService::OnPing);
        Http_.AddHandler("/get_permalink_info", NHttp::ExternalWithoutTvm(), this, &TService::OnGetPermalinkInfo);

        GrpcServer_.SetReqAnsLogger(&ReqAnsLogger_, {"Ping"});
        REGISTER_GRPC_HANDLER(GrpcServer_, GetCounts, this, &TService::OnGrpcGetCounts);
        REGISTER_GRPC_HANDLER(GrpcServer_, GetHotels, this, &TService::OnGrpcGetHotels);
        REGISTER_GRPC_HANDLER(GrpcServer_, Ping, this, &TService::OnGrpcPing);

        BoyPartnerProvider_.AddOnUpdateHandler([this]() {
            Index_.Start();
        });

        Index_.OnReady([this]() {
            GrpcServer_.Start();
            Http_.Start(Config_.GetHttp());
        });
    }

    void TService::Start() {
        TScheduler::Instance().Start();
        MonWebService_.Start(Config_.GetOther().GetMonitoringPort());
        ReqAnsLogger_.Start();
        OfferCacheClient_.Start();
        PromoServiceClient_.Start();
        RegionsService_.Start();
        BoyPartnerProvider_.Start(); // It will trigger index start when ready, so it should be after all index deps
    }

    void TService::Wait() {
        StopEvent_.WaitI();
    }

    void TService::Stop() {
        RestartDetector_.ReportShutdown();
        Http_.Stop();
        GrpcServer_.Stop();
        RegionsService_.Stop();
        PromoServiceClient_.Stop();
        OfferCacheClient_.Stop();
        Index_.Stop();
        BoyPartnerProvider_.Stop();
        ReqAnsLogger_.Stop();
        MonWebService_.Stop();
        StopEvent_.Signal();
    }

    void TService::OnSetLogLevel(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
        if (auto v = httpReq.Query().Get("level")) {
            auto p = FromString<ELogPriority>(v);
            DoInitGlobalLog(CreateLogBackend(Config_.GetOther().GetMainLogFile(), p, true));
            responseCb(NHttp::TResponse::CreateText("Log level was changed to " + ToString(v) + "\n"));
        } else {
            responseCb(THttpResponse(HTTP_BAD_REQUEST));
        }
    }

    void TService::OnPing(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
        Y_UNUSED(httpReq);
        bool isReady = IsReady();
        responseCb(NHttp::TResponse::CreateText(isReady ? "OK" : "Not ready", isReady ? HTTP_OK : HTTP_BAD_GATEWAY));
    }

    void TService::OnReopenReqAnsLog(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
        Y_UNUSED(httpReq);
        INFO_LOG << "Reopening ReqAns log" << Endl;
        ReqAnsLogger_.Reopen();
        responseCb(THttpResponse(HTTP_OK));
    }

    void TService::OnShutdown(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
        Y_UNUSED(httpReq);
        if (!IsShuttingDown_.TrySet()) {
            responseCb(NHttp::TResponse::CreateText("Already shutting down\n"));
            return;
        }
        INFO_LOG << "Shutting down!" << Endl;
        SystemThreadFactory()->Run([this]() {
            Http_.Shutdown();
            GrpcServer_.Shutdown();
            auto dur = TDuration::Seconds(7);
            INFO_LOG << "Http & gRPC Shutdown called, wait " << dur << Endl;
            StopEvent_.WaitT(dur);
            Stop();
        });
        responseCb(NHttp::TResponse::CreateText("Shutdown procedure started\n"));
    }

    void TService::OnGetPermalinkInfo(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
        if (auto rawPermalink = httpReq.Query().Get("permalink")) {
            auto permalink = FromString<TPermalink>(rawPermalink);
            auto info = Index_.GetPermalinkInfo(permalink);
            if (info.Empty()) {
                responseCb(THttpResponse(HTTP_NOT_FOUND));
            } else {
                TString res;
                {
                    TStringOutput output(res);
                    NJson::TJsonWriter writer(&output, true, true);
                    auto offerBusData = *info.Get();
                    writer.OpenMap();
                    for (const auto&[key, offers] : info.Get()->Offers) {
                        writer.OpenMap(ToString(key));
                        for (const auto&[partner, cacheItem] : offers) {
                            writer.OpenMap(ToString(partner));
                            writer.Write("ExpirationTimestamp", cacheItem.ExpirationTimestamp.ToString());
                            writer.OpenArray("Offers");
                            for (const auto& offer : cacheItem.Offers) {
                                writer.OpenMap();
                                writer.Write("Price", offer.Price);
                                writer.Write("FreeCancellation", ToString(offer.FreeCancellation));
                                writer.Write("Pansion", ToString(offer.Pansion));
                                writer.CloseMap();
                            }
                            writer.CloseArray();
                            writer.CloseMap();
                        }
                        writer.CloseMap();
                    }
                    writer.CloseMap();
                }
                responseCb(NHttp::TResponse::CreateJson(res));
            }
        } else {
            responseCb(THttpResponse(HTTP_BAD_REQUEST));
        }
    }

    void TService::OnGrpcGetCounts(const NTravelProto::NGeoCounter::TGetCountsRequest& req,
                                   const NGrpc::TServerReqMetadata& srvMeta,
                                   const TGrpcServer::TResponseCb<NTravelProto::NGeoCounter::TGetCountsResponse>& responseCb) {
        DEBUG_LOG << "Got gRPC request callId " << srvMeta.CallId << " from '" << srvMeta.RemoteFQDN << Endl;

        auto counters = GetCountersPerRequestType("GetCounts");

        TBookingRange bookingRange(0, 0);
        TAges ages;
        TVector<TBasicFilterGroup> initialFilterGroups;
        TVector<TAdditionalFilter> additionalFilters;
        THashSet<TString> seenGroupIds;
        try {
            bookingRange = TBookingRange(NOrdinalDate::FromString(req.GetCheckInDate()), NOrdinalDate::FromString(req.GetCheckOutDate()));
            ages = TAges::FromAgesString(req.GetAges());
            for (const auto& initialFilterGroup : req.GetInitialFilterGroups()) {
                auto& group = initialFilterGroups.emplace_back();
                if (!seenGroupIds.insert(initialFilterGroup.GetUniqueId()).second) {
                    ythrow yexception() << "Duplicate group id in request: " << initialFilterGroup.GetUniqueId();
                }
                group.Id = StringEncoder_.Encode(initialFilterGroup.GetUniqueId());
                for (const auto& filter : initialFilterGroup.GetFilters()) {
                    group.Filters.push_back(FilterRegistry_.BuildFilter(filter, bookingRange, ages));
                }
            }
            for (const auto& additionalFilter : req.GetAdditionalFilters()) {
                additionalFilters.push_back(FilterRegistry_.BuildAdditionalFilter(additionalFilter, bookingRange, ages));
            }
        } catch (...) {
            NTravelProto::NGeoCounter::TGetCountsResponse resp;
            resp.MutableError()->SetMessage(CurrentExceptionMessage());
            ERROR_LOG << "Bad request: " << CurrentExceptionMessage() << Endl;
            counters->NBadRequests.Inc();
            responseCb(resp, NGrpc::TServerRespMetadata::BuildOk());
            return;
        }

        try {
            TCountResults counts;
            if (Config_.GetOther().GetUseOfferBusDataFilters() || req.GetUseOfferBusDataFilters()) { // todo (mpivko): maybe handle this while convering filters?
                counts = Index_.GetCountsWithOfferBusData(
                    TBoundingBox(
                        TPosition(req.GetLowerLeftLat(), req.GetLowerLeftLon()),
                        TPosition(req.GetUpperRightLat(), req.GetUpperRightLon())),
                    bookingRange,
                    initialFilterGroups,
                    additionalFilters,
                    req.GetDisablePriceCounts());
            } else {
                counts = Index_.GetCounts(
                    TBoundingBox(
                        TPosition(req.GetLowerLeftLat(), req.GetLowerLeftLon()),
                        TPosition(req.GetUpperRightLat(), req.GetUpperRightLon())),
                    bookingRange,
                    initialFilterGroups,
                    additionalFilters,
                    req.GetDisablePriceCounts(),
                    false);
            }

            NTravelProto::NGeoCounter::TGetCountsResponse resp;
            auto respCounts = resp.MutableCounts();
            respCounts->SetTotalCount(counts.TotalCount);
            respCounts->SetMatchedCount(counts.MatchedCount);
            for (const auto& additionalFilterCounts : counts.AdditionalFilterCounts) {
                auto pbAdditionalFilterCounts = respCounts->AddAdditionalFilterCounts();
                pbAdditionalFilterCounts->SetUniqueId(StringEncoder_.Decode(additionalFilterCounts.UniqueId));
                pbAdditionalFilterCounts->SetCount(additionalFilterCounts.Count);
            }
            if (!counts.PriceResults.HistogramBounds.empty()) {
                auto pbPriceCounts = respCounts->MutablePriceCounts();
                pbPriceCounts->SetMinPriceEstimate(counts.PriceResults.MinPriceEstimate);
                pbPriceCounts->SetMaxPriceEstimate(counts.PriceResults.MaxPriceEstimate);
                for (const auto& x : counts.PriceResults.HistogramBounds) {
                    pbPriceCounts->AddHistogramBounds(x);
                }
                for (const auto& x : counts.PriceResults.HistogramCounts) {
                    pbPriceCounts->AddHistogramCounts(x);
                }
            }
            responseCb(resp, NGrpc::TServerRespMetadata::BuildOk());
        } catch (...) {
            NTravelProto::NGeoCounter::TGetCountsResponse resp;
            resp.MutableError()->SetMessage(CurrentExceptionMessage());
            ERROR_LOG << "Error while processing request: " << CurrentExceptionMessage() << Endl;
            counters->NUnexpectedErrors.Inc();
            responseCb(resp, NGrpc::TServerRespMetadata::BuildOk());
            return;
        }
    }

    void TService::OnGrpcGetHotels(const NTravelProto::NGeoCounter::TGetHotelsRequest& req,
                                   const NGrpc::TServerReqMetadata& srvMeta,
                                   const TGrpcServer::TResponseCb<NTravelProto::NGeoCounter::TGetHotelsResponse>& responseCb) {
        DEBUG_LOG << "Got gRPC request callId " << srvMeta.CallId << " from '" << srvMeta.RemoteFQDN << Endl;

        auto counters = GetCountersPerRequestType("GetHotels");

        bool badRequest = false;
        TString badRequestError;
        NTravelProto::EErrorCode badRequestErrorCode = NTravelProto::EErrorCode::EC_OK;
        THotelSearchService::TSearchHotelsParsedRequestData parsedReqData;
        try {
            parsedReqData = HotelSearchService_.ParseSearchHotels(req, srvMeta.CallId);
        }  catch (const THotelSearchService::invalid_polling_token_exception& e) {
            badRequest = true;
            badRequestError = CurrentExceptionMessage();
            badRequestErrorCode = NTravelProto::EErrorCode::EC_INVALID_POLLING_TOKEN;
        } catch (...) {
            badRequest = true;
            badRequestError = CurrentExceptionMessage();
        }

        if (badRequest) {
            NTravelProto::TError error;
            error.SetMessage(badRequestError);
            if (badRequestErrorCode != NTravelProto::EErrorCode::EC_OK) {
                error.SetCode(badRequestErrorCode);
            }
            ERROR_LOG << "Bad request: " << badRequestError << Endl;
            counters->NBadRequests.Inc();
            responseCb(
                NTravelProto::NGeoCounter::TGetHotelsResponse(),
                NGrpc::TServerRespMetadata::BuildFailed(grpc::Status(grpc::StatusCode::INVALID_ARGUMENT, badRequestError), error)
            );
            return;
        }

        try {
            auto resp = HotelSearchService_.SearchHotels(parsedReqData);
            responseCb(resp, NGrpc::TServerRespMetadata::BuildOk());
        } catch (...) {
            auto message = CurrentExceptionMessage();
            NTravelProto::TError error;
            error.SetMessage(message);
            ERROR_LOG << "Error while processing request: " << message << Endl;
            counters->NUnexpectedErrors.Inc();
            responseCb(
                NTravelProto::NGeoCounter::TGetHotelsResponse(),
                NGrpc::TServerRespMetadata::BuildFailed(grpc::Status(grpc::StatusCode::INTERNAL, message), error)
            );
            return;
        }
    }

    void TService::OnGrpcPing(const NTravelProto::TPingRpcReq& req, const NGrpc::TServerReqMetadata& srvMeta, const TGrpcServer::TResponseCb<NTravelProto::TPingRpcRsp>& responseCb) {
        Y_UNUSED(req, srvMeta);
        NTravelProto::TPingRpcRsp resp;
        resp.SetIsReady(IsReady());
        responseCb(resp, NGrpc::TServerRespMetadata::BuildOk());
    }

    bool TService::IsReady() const {
        return Index_.IsReady() && BoyPartnerProvider_.IsReady() && RegionsService_.IsReady();
    }

    TAtomicSharedPtr<TService::TCountersPerRequestType> TService::GetCountersPerRequestType(const TString& requestType) {
        return CountersPerRequestType_.GetOrCreate({requestType});
    }

    TService::TCounters::TCounters(TService& service)
        : Service_(service)
    {
    }

    void TService::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
        IsReady = Service_.IsReady() ? 1 : 0;

        ct->insert(MAKE_COUNTER_PAIR(IsReady));
    }

    void TService::TCountersPerRequestType::QueryCounters(NMonitor::TCounterTable* ct) const {
        ct->insert(MAKE_COUNTER_PAIR(NBadRequests));
        ct->insert(MAKE_COUNTER_PAIR(NUnexpectedErrors));
    }
}
