#include "coordinate_utils.h"
#include "sorts.h"
#include "utils.h"

#include <catboost/libs/cat_feature/cat_feature.h>

#include <util/string/builder.h>

#include <utility>

namespace {
    constexpr static double RANKING_FACTORS_EPSILON = 1e-6;
}

namespace NTravel::NGeoCounter {
    TCatBoostModelEvaluator::TCatBoostModelEvaluator(const NTravelProto::NGeoCounter::TConfig::TCatBoostModel& config,
                                                     TStringEncoder& stringEncoder,
                                                     const TUserSegmentsRegistry& userSegmentsRegistry)
        : Config_(config)
        , Model_(ReadModel(config.GetModelPath()))
    {
        Y_ENSURE(Model_.GetNumFloatFeatures() == static_cast<size_t>(Config_.GetFloatFeatures().size()), "Model float features count ("
            + ToString(Model_.GetNumFloatFeatures()) + ") doesn't match config float features count ("
            + ToString(Config_.GetFloatFeatures().size()) + ")");
        Y_ENSURE(Model_.GetNumCatFeatures() == static_cast<size_t>(Config_.GetCatFeatures().size()), "Model cat features count ("
            + ToString(Model_.GetNumCatFeatures()) + ") doesn't match config cat features count ("
            + ToString(Config_.GetCatFeatures().size()) + ")");

        FloatFeatureGetters_.reserve(Config_.GetFloatFeatures().size());
        for (const auto& pbFloatFeatureConfig: Config_.GetFloatFeatures()) {
            if (pbFloatFeatureConfig.HasStaticFeatureSource()) {
                const auto& source = pbFloatFeatureConfig.GetStaticFeatureSource();
                auto featureId = stringEncoder.Encode(source.GetName());
                FloatFeatureGetters_.push_back([featureId, defaultValue = source.GetDefaultValue()](const THashMap<int, float>& hotelFloatFeatures,
                                                                                                    const TCryptaSegments& cryptaSegments) {
                    Y_UNUSED(cryptaSegments);
                    auto valueIt = hotelFloatFeatures.find(featureId);
                    if (valueIt == hotelFloatFeatures.end()) {
                        return defaultValue;
                    }
                    return valueIt->second;
                });
            } else if (pbFloatFeatureConfig.HasBigbSegmentWeightSource()) {
                const auto& source = pbFloatFeatureConfig.GetBigbSegmentWeightSource();
                auto segment = userSegmentsRegistry.GetSegmentById(source.GetKeywordId(), source.GetSegmentId());
                Y_ENSURE(segment.Defined(), "Unknown bigb segment (keyword_id=" + ToString(source.GetKeywordId())
                    + ", segment_id=" + ToString(source.GetSegmentId()) + ") (" + pbFloatFeatureConfig.GetFeatureName() + ")");
                auto key = std::make_pair(source.GetKeywordId(), source.GetSegmentId());
                FloatFeatureGetters_.push_back([key, defaultValue = source.GetDefaultValue()](const THashMap<int, float>& hotelFloatFeatures,
                                                                                              const TCryptaSegments& cryptaSegments) {
                    Y_UNUSED(hotelFloatFeatures);
                    auto weightIt = cryptaSegments.WeightedSegmentValues.find(key);
                    if (weightIt == cryptaSegments.WeightedSegmentValues.end()) {
                        return defaultValue;
                    } else {
                        return static_cast<float>(weightIt->second);
                    }
                });
            } else if (pbFloatFeatureConfig.HasBigbSegmentPresenceSource()) {
                const auto& source = pbFloatFeatureConfig.GetBigbSegmentPresenceSource();
                auto segment = userSegmentsRegistry.GetSegmentById(source.GetKeywordId(), source.GetSegmentId());
                Y_ENSURE(segment.Defined(), "Unknown bigb segment (keyword_id=" + ToString(source.GetKeywordId())
                    + ", segment_id=" + ToString(source.GetSegmentId()) + ") (" + pbFloatFeatureConfig.GetFeatureName() + ")");
                Y_ENSURE(IsIn({TUserSegmentsRegistry::EKeywordType::KT_UINT_VALUES,
                               TUserSegmentsRegistry::EKeywordType::KT_WEIGHTED_UINT_WITH_SELECTED_VALUES}, segment->Keyword.KeywordType),
                         "Can't use BigbSegmentPresenceSource for keyword " + ToString(source.GetKeywordId())
                             + ", it has incompatible type (" + pbFloatFeatureConfig.GetFeatureName() + ")");
                auto key = std::make_pair(source.GetKeywordId(), source.GetSegmentId());
                FloatFeatureGetters_.push_back([key, valueOnAbsent = source.GetValueOnAbsent(), valueOnPresent = source.GetValueOnPresent()]
                                                   (const THashMap<int, float>& hotelFloatFeatures, const TCryptaSegments& cryptaSegments) {
                    Y_UNUSED(hotelFloatFeatures);
                    auto segmentIt = cryptaSegments.SegmentValues.find(key);
                    if (segmentIt == cryptaSegments.SegmentValues.end()) {
                        return valueOnAbsent;
                    }
                    return valueOnPresent;
                });
            } else {
                throw yexception() << "Unknown source type in feature config for " << pbFloatFeatureConfig.GetFeatureName();
            }
        }

        CatFeatureGetters_.reserve(Config_.GetCatFeatures().size());
        for (const auto& pbCatFeatureConfig: Config_.GetCatFeatures()) {
            if (pbCatFeatureConfig.HasStaticFeatureSource()) {
                const auto& source = pbCatFeatureConfig.GetStaticFeatureSource();
                auto featureId = stringEncoder.Encode(source.GetName());
                auto defaultValue = CalcCatFeatureHash(source.GetDefaultValue());
                CatFeatureGetters_.push_back([featureId, defaultValue](const THashMap<int, ui32>& hotelCatFeaturesCatBoostEncoded,
                                                                       const TCryptaSegments& cryptaSegments,
                                                                       const THashMap<int, TVector<ui32>>& cryptaCatBoostEncodedSegmentsMap) {
                    Y_UNUSED(cryptaSegments);
                    Y_UNUSED(cryptaCatBoostEncodedSegmentsMap);
                    auto valueIt = hotelCatFeaturesCatBoostEncoded.find(featureId);
                    if (valueIt == hotelCatFeaturesCatBoostEncoded.end()) {
                        return defaultValue;
                    }
                    return valueIt->second;
                });
            } else if (pbCatFeatureConfig.HasBigbSegmentIdSource()) {
                const auto& source = pbCatFeatureConfig.GetBigbSegmentIdSource();
                auto keyword = userSegmentsRegistry.GetKeywordById(source.GetKeywordId());
                Y_ENSURE(keyword.Defined(), "Unknown bigb keyword (keyword_id=" + ToString(source.GetKeywordId()) + ")");
                Y_ENSURE(IsIn({TUserSegmentsRegistry::EKeywordType::KT_UINT_VALUES,
                               TUserSegmentsRegistry::EKeywordType::KT_WEIGHTED_UINT_WITH_SELECTED_VALUES}, keyword->KeywordType),
                         "Can't use BigbSegmentIdSource for keyword " + ToString(source.GetKeywordId())
                             + ", it has incompatible type (" + pbCatFeatureConfig.GetFeatureName() + ")");
                auto defaultValue = CalcCatFeatureHash(source.GetDefaultValue());
                CatFeatureGetters_.push_back([keywordId = source.GetKeywordId(), defaultValue](const THashMap<int, ui32>& hotelCatFeaturesCatBoostEncoded,
                                                                                               const TCryptaSegments& cryptaSegments,
                                                                                               const THashMap<int, TVector<ui32>>& cryptaCatBoostEncodedSegmentsMap) {
                    Y_UNUSED(hotelCatFeaturesCatBoostEncoded);
                    Y_UNUSED(cryptaSegments);
                    auto segmentsIt = cryptaCatBoostEncodedSegmentsMap.find(keywordId);
                    if (segmentsIt == cryptaCatBoostEncodedSegmentsMap.end() || segmentsIt->second.size() != 1) {
                        return defaultValue;
                    }
                    return segmentsIt->second.at(0);
                });
            } else if (pbCatFeatureConfig.HasBigbSegmentPresenceSource()) {
                const auto& source = pbCatFeatureConfig.GetBigbSegmentPresenceSource();
                auto segment = userSegmentsRegistry.GetSegmentById(source.GetKeywordId(), source.GetSegmentId());
                Y_ENSURE(segment.Defined(), "Unknown bigb segment (keyword_id=" + ToString(source.GetKeywordId())
                    + ", segment_id=" + ToString(source.GetSegmentId()) + ") (" + pbCatFeatureConfig.GetFeatureName() + ")");
                Y_ENSURE(IsIn({TUserSegmentsRegistry::EKeywordType::KT_UINT_VALUES,
                               TUserSegmentsRegistry::EKeywordType::KT_WEIGHTED_UINT_WITH_SELECTED_VALUES}, segment->Keyword.KeywordType),
                         "Can't use BigbSegmentPresenceSource for keyword " + ToString(source.GetKeywordId())
                             + ", it has incompatible type (" + pbCatFeatureConfig.GetFeatureName() + ")");
                auto key = std::make_pair(source.GetKeywordId(), source.GetSegmentId());
                auto valueOnAbsent = CalcCatFeatureHash(source.GetValueOnAbsent());
                auto valueOnPresent = CalcCatFeatureHash(source.GetValueOnPresent());
                CatFeatureGetters_.push_back([key, valueOnAbsent, valueOnPresent](const THashMap<int, ui32>& hotelCatFeaturesCatBoostEncoded,
                                                                                  const TCryptaSegments& cryptaSegments,
                                                                                  const THashMap<int, TVector<ui32>>& cryptaCatBoostEncodedSegmentsMap) {
                    Y_UNUSED(hotelCatFeaturesCatBoostEncoded);
                    Y_UNUSED(cryptaCatBoostEncodedSegmentsMap);
                    auto segmentIt = cryptaSegments.SegmentValues.find(key);
                    if (segmentIt == cryptaSegments.SegmentValues.end()) {
                        return valueOnAbsent;
                    }
                    return valueOnPresent;
                });
            } else {
                throw yexception() << "Unknown source type in feature config for " << pbCatFeatureConfig.GetFeatureName();
            }
        }
    }

    void TCatBoostModelEvaluator::ReorderTopHotels(TVector<THotelsResults::THotelResult>* hotels, const TCryptaSegments& cryptaSegments, bool applySort) const {
        THashMap<int, float> emptyHotelFloatFeatures;
        THashMap<int, ui32> emptyHotelCatFeatures;
        auto prefixSize = Min(hotels->size(), static_cast<size_t>(Config_.GetReorderTopSize()));
        THashMap<TPermalink, double> factors;
        TVector<double> results({0});
        TVector<float> modelFloatFeatures;
        modelFloatFeatures.reserve(FloatFeatureGetters_.size());
        THashMap<int, TVector<ui32>> cryptaCatBoostEncodedSegmentsMap;
        for (size_t i = 0; i < prefixSize; i++) {
            auto& hotel = hotels->at(i);
            const auto& hotelFloatFeatures = hotel.ExtraInfo.Defined() ? hotel.ExtraInfo.GetRef().RealtimeRankingFloatFeatures : emptyHotelFloatFeatures;
            const auto& hotelCatFeatures = hotel.ExtraInfo.Defined() ? hotel.ExtraInfo.GetRef().RealtimeRankingCatFeaturesCatBoostEncoded : emptyHotelCatFeatures;

            modelFloatFeatures.clear();
            for (const auto& floatFeatureGetter: FloatFeatureGetters_) {
                modelFloatFeatures.push_back(floatFeatureGetter(hotelFloatFeatures, cryptaSegments));
            }

            cryptaCatBoostEncodedSegmentsMap.clear();
            for (const auto& [keywordId, segmentId]: cryptaSegments.SegmentValues) {
                cryptaCatBoostEncodedSegmentsMap[keywordId].push_back(CalcCatFeatureHash(ToString(segmentId)));
            }

            TVector<int> modelCatFeatures;
            modelCatFeatures.reserve(CatFeatureGetters_.size());
            for (const auto& catFeatureGetter: CatFeatureGetters_) {
                modelCatFeatures.push_back(static_cast<int>(catFeatureGetter(hotelCatFeatures, cryptaSegments, cryptaCatBoostEncodedSegmentsMap)));
            }

            Model_.Calc(modelFloatFeatures, modelCatFeatures, results);
            auto prediction = results.at(0);
            factors[hotel.Permalink] = prediction;

            hotel.RealTimeRankingInfo.CatBoostUsed = true;
            hotel.RealTimeRankingInfo.CatBoostPrediction = prediction;
        }

        if (applySort) {
            Sort(hotels->begin(), hotels->begin() + prefixSize,
                 [&factors](const THotelsResults::THotelResult& lhs, const THotelsResults::THotelResult& rhs) {
                     if (lhs.IsTopHotel && !rhs.IsTopHotel) {
                         return true;
                     }
                     if (!lhs.IsTopHotel && rhs.IsTopHotel) {
                         return false;
                     }
                     return factors[lhs.Permalink] - factors[rhs.Permalink] > RANKING_FACTORS_EPSILON;
                 });
        }
    }

    TSortType::TSortContext::TSortContext(const TCryptaSegments& userCryptaSegments, TMaybe<TPosition> sortOrigin)
        : UserCryptaSegments(userCryptaSegments)
        , SortOrigin(sortOrigin)
    {
    }

    TSortType::TSortType(size_t id,
                         NTravelProto::NGeoCounter::ESortType sortType,
                         int priority,
                         const THashMap<int, TVector<int>>& cryptaSegmentFilters,
                         TSortType::SortInIndexCmpFunc cmpSortInIndex,
                         TMaybe<TSortType::SortBeforeRespCmpFunc> cmpSortBeforeResp,
                         TMaybe<TReorderTopRecordsParams> reorderTopRecordsParams,
                         TMaybe<TRealTimeSortInIndex> realTimeSortInIndex)
        : Id(id)
        , SortType(sortType)
        , Priority(priority)
        , CryptaSegmentFilters(cryptaSegmentFilters)
        , SortInIndexCmp(std::move(cmpSortInIndex))
        , SortBeforeRespCmp(cmpSortBeforeResp)
        , ReorderTopRecordsParams(reorderTopRecordsParams)
        , RealTimeSortInIndex(realTimeSortInIndex)
    {
    }

    TString TSortType::GetStringRepr() const {
        TVector<TString> segmentReprs;
        for (const auto& [keywordId, segmentValues]: CryptaSegmentFilters) {
            TStringBuilder innerBuilder{};
            TVector<TString> valuesReprs;
            for (const auto& value: segmentValues) {
                valuesReprs.push_back(ToString(value));
            }
            Sort(valuesReprs.begin(), valuesReprs.end());
            innerBuilder << ToString(keywordId) << ":" << JoinStrings(valuesReprs, ",");
            segmentReprs.push_back(innerBuilder);
        }
        Sort(segmentReprs.begin(), segmentReprs.end());

        TStringBuilder builder{};

        builder << "SortType=" << SortType << " ";
        builder << "Segments=" << JoinStrings(segmentReprs, "|");

        return builder;
    }

    bool TSortType::MatchesSegments(const TCryptaSegments& userCryptaSegments) const {
        for (const auto& [keywordId, segmentValues]: CryptaSegmentFilters) {
            for (const auto& value: segmentValues) {
                if (!userCryptaSegments.SegmentValues.contains(std::make_pair(keywordId, value))) {
                    return false;
                }
            }
        }
        return true;
    }

    const TSortType::SortInIndexCmpFunc& TSortType::GetCmpSortInIndex() const {
        return SortInIndexCmp;
    }

    const TMaybe<TSortType::SortBeforeRespCmpFunc>& TSortType::GetCmpSortBeforeResp() const {
        return SortBeforeRespCmp;
    }

    const TMaybe<TSortType::TReorderTopRecordsParams>& TSortType::GetReorderTopRecordsParams() const {
        return ReorderTopRecordsParams;
    }

    const TMaybe<TSortType::TRealTimeSortInIndex>& TSortType::GetRealTimeSortInIndex() const {
        return RealTimeSortInIndex;
    }

    TSortTypeWithContext::TSortTypeWithContext(const TSortType& sortType, TSortType::TSortContext context)
        : SortType(sortType)
        , Context(std::move(context))
    {
    }

    TSortType TSortTypeRegistry::GetDefaultSortType() const {
        return SortTypes.at(DefaultSortTypeIndex);
    }

    TVector<TSortType> TSortTypeRegistry::GetSortTypes() const {
        return SortTypes;
    }

    TSortType TSortTypeRegistry::GetSortType(NTravelProto::NGeoCounter::ESortType sortTypeEnum, const TCryptaSegments& userCryptaSegments) const {
        TMaybe<TSortType> bestSort{};
        for (const auto& sortType: SortTypes) {
            if (sortType.SortType == sortTypeEnum && sortType.MatchesSegments(userCryptaSegments) && (bestSort.Empty() || bestSort.GetRef().Priority < sortType.Priority)) {
                bestSort = sortType;
            }
        }
        if (bestSort.Empty()) {
            throw yexception() << "Cannot find sort type by enum '" << sortTypeEnum << "'";
        }
        return bestSort.GetRef();
    }

    TSortTypeRegistry::TSortTypeRegistry(const TVector<TSortType>& sortTypes, size_t defaultSortTypeIndex)
        : SortTypes(sortTypes)
        , DefaultSortTypeIndex(defaultSortTypeIndex)
    {
        auto seenIds = THashSet<int>();
        auto seenReprs = THashSet<TString>();
        for (const auto& sortType: sortTypes) {
            Y_ENSURE(sortType.Id >= 0 && static_cast<size_t>(sortType.Id) < g_SortTypeCount, "Invalid sort id, should be in [0; g_SortTypeCount)");
            Y_ENSURE(seenIds.emplace(sortType.Id).second, "Duplicate sort id: " + ToString(sortType.Id));
            auto repr = sortType.GetStringRepr();
            Y_ENSURE(seenReprs.emplace(repr).second, "Duplicate sort repr: " + repr);
        }
    }

    TSortTypeRegistry TSortTypeRegistry::Build(TStringEncoder& stringEncoder,
                                               const NTravelProto::NGeoCounter::TConfig& config,
                                               const TUserSegmentsRegistry& userSegmentsRegistry) {
        TVector<TSortType> sortTypes;

        auto getPreSortByFactor = [&stringEncoder](const TString& rankingFactorName) -> TSortType::SortInIndexCmpFunc {
            auto rankingFactor = stringEncoder.Encode(rankingFactorName);
            return [rankingFactor](const THotelSortData& lhs, const THotelSortData& rhs) {
                auto lit = lhs.HotelAltayData->RankingFactors.find(rankingFactor);
                auto rit = rhs.HotelAltayData->RankingFactors.find(rankingFactor);
                if (lit == lhs.HotelAltayData->RankingFactors.end()) {
                    return false;
                }
                if (rit == rhs.HotelAltayData->RankingFactors.end()) {
                    return true;
                }
                return lit->second - rit->second > RANKING_FACTORS_EPSILON;
            };
        };

        auto addSortByRankingFactor = [&sortTypes, &getPreSortByFactor](const TString& rankingFactorName,
                                                                        NTravelProto::NGeoCounter::ESortType sortTypeEnum,
                                                                        int priority,
                                                                        const THashMap<int, TVector<int>>& segmentFilters) {
            sortTypes.push_back(TSortType(sortTypes.size(), sortTypeEnum, priority, segmentFilters, getPreSortByFactor(rankingFactorName)));
        };

        // by rank
        size_t defaultSortTypeIndex = 0;
        addSortByRankingFactor(config.GetSortOptions().GetDefaultRankingFactorName(), NTravelProto::NGeoCounter::ST_ByRank, 0, {}); // default sort
        addSortByRankingFactor(config.GetSortOptions().GetExperimentalRankingFactorName(), NTravelProto::NGeoCounter::ST_ByRankExperimental, 0, {});

        addSortByRankingFactor("factor_group_1", NTravelProto::NGeoCounter::ST_ByRankExploration1, 0, {});
        addSortByRankingFactor("factor_group_2", NTravelProto::NGeoCounter::ST_ByRankExploration2, 0, {});
        addSortByRankingFactor("factor_group_3", NTravelProto::NGeoCounter::ST_ByRankExploration3, 0, {});
        addSortByRankingFactor("factor_group_4", NTravelProto::NGeoCounter::ST_ByRankExploration4, 0, {});
        addSortByRankingFactor("factor_group_5", NTravelProto::NGeoCounter::ST_ByRankExploration5, 0, {});

        auto convertSegment = [](const TUserSegmentsRegistry::TKeyword& keyword, const TVector<TString>& values) {
            THashMap<TString, int> possibleValues;
            for (const auto& possibleSegment: keyword.PossibleSegments) {
                possibleValues.emplace(possibleSegment.SegmentName, possibleSegment.SegmentId);
            }

            TVector<int> result;
            for (const auto& value: values) {
                auto it = possibleValues.find(value);
                Y_ENSURE(it != possibleValues.end(), "Unknown segment value " + value + " for keyword " + keyword.KeywordName);
                result.push_back(it->second);
            }
            return result;
        };

        auto ageKeyword = userSegmentsRegistry.GetKeywordByIdOrFail(TUserSegmentsRegistry::AgeKeywordId);
        auto incomeKeyword = userSegmentsRegistry.GetKeywordByIdOrFail(TUserSegmentsRegistry::IncomeKeywordId);
        addSortByRankingFactor("age={#N/A}+income={#N/A}", NTravelProto::NGeoCounter::ST_ByRankWithPersonalization, 0, {});
        for (const auto& age_segment: TVector<TVector<TString>>{{"0_17", "18_24"}, {"25_34"}, {"35_44"}, {"45_54"}, {"55_99"}}) {
            for (const auto& income_segment: TVector<TVector<TString>>{{"A", "B1"}, {"B2"}, {"C1", "C2"}}) {
                THashMap<int, TVector<int>> segmentFilters;

                segmentFilters[ageKeyword.KeywordId] = convertSegment(ageKeyword, age_segment);
                segmentFilters[incomeKeyword.KeywordId] = convertSegment(incomeKeyword, income_segment);

                TStringBuilder name{};
                name << "age={" << JoinStrings(age_segment, ",") << "}+income={" << JoinStrings(income_segment, ",") << "}";

                addSortByRankingFactor(name, NTravelProto::NGeoCounter::ST_ByRankWithPersonalization, 1, segmentFilters);
            }
        }

        if (config.GetSortOptions().HasCatBoostModel()) {
            auto modelConfig = config.GetSortOptions().GetCatBoostModel();
            auto modelEvaluator = std::make_shared<TCatBoostModelEvaluator>(modelConfig, stringEncoder, userSegmentsRegistry);
            sortTypes.push_back(TSortType(sortTypes.size(), NTravelProto::NGeoCounter::ESortType::ST_ByRankWithCatBoost, 0, {},
                                          getPreSortByFactor(config.GetSortOptions().GetCatBoostPreRankingFactorName()), {},
                                          TSortType::TReorderTopRecordsParams{
                                              modelConfig.GetReorderTopSize(),
                                              [modelEvaluator](TVector<THotelsResults::THotelResult>* hotels, const TSortType::TSortContext& sortContext) {
                                                  modelEvaluator->ReorderTopHotels(hotels, sortContext.UserCryptaSegments, true);
                                              }}));

            sortTypes.push_back(TSortType(sortTypes.size(), NTravelProto::NGeoCounter::ESortType::ST_ByRankWithCatBoostFake, 0, {},
                                          getPreSortByFactor(config.GetSortOptions().GetDefaultRankingFactorName()), {},
                                          TSortType::TReorderTopRecordsParams{
                                              modelConfig.GetReorderTopSize(),
                                              [modelEvaluator](TVector<THotelsResults::THotelResult>* hotels, const TSortType::TSortContext& sortContext) {
                                                  modelEvaluator->ReorderTopHotels(hotels, sortContext.UserCryptaSegments, false);
                                              }}));
        } else {
            // For correct handling requests from testing api to prod geocounter
            addSortByRankingFactor(config.GetSortOptions().GetDefaultRankingFactorName(), NTravelProto::NGeoCounter::ST_ByRankWithCatBoostFake, 0, {});
        }

        auto getPreSortByPrice = [](std::function<bool(TPriceVal, TPriceVal)> cmp) -> TSortType::SortInIndexCmpFunc {
            return [cmp](const THotelSortData& lhs, const THotelSortData& rhs) {
                const auto& lhsPrice = lhs.PriceInfo->DefaultPrice;
                const auto& rhsPrice = rhs.PriceInfo->DefaultPrice;
                if (lhsPrice.Empty()) {
                    return false;
                }
                if (rhsPrice.Empty()) {
                    return true;
                }
                return cmp(lhsPrice.GetRef().MinPrice, rhsPrice.GetRef().MinPrice);
            };
        };

        auto getPostSortByPrice = [](std::function<bool(TPriceVal, TPriceVal)> cmp) -> TSortType::SortBeforeRespCmpFunc {
            return [cmp](const THotelsResults::THotelResult& lhs, const THotelsResults::THotelResult& rhs) {
                const auto& lhsOfferInfo = lhs.OfferInfo;
                const auto& rhsOfferInfo = rhs.OfferInfo;
                if (lhsOfferInfo.Empty()) {
                    return false;
                }
                if (rhsOfferInfo.Empty()) {
                    return true;
                }
                return cmp(lhsOfferInfo.GetRef().Price, rhsOfferInfo.GetRef().Price);
            };
        };

        // by price asc
        auto less = [](const TPriceVal& lhs, const TPriceVal& rhs) { return lhs < rhs; };
        sortTypes.push_back(TSortType(sortTypes.size(), NTravelProto::NGeoCounter::ESortType::ST_ByPriceAsc, 0, {}, getPreSortByPrice(less), getPostSortByPrice(less)));

        // by price desc
        auto greater = [](const TPriceVal& lhs, const TPriceVal& rhs) { return lhs > rhs; };
        sortTypes.push_back(TSortType(sortTypes.size(), NTravelProto::NGeoCounter::ESortType::ST_ByPriceDesc, 0, {}, getPreSortByPrice(greater), getPostSortByPrice(greater)));

        auto ratingFeatureId = stringEncoder.Encode("rating");
        // by rating
        sortTypes.push_back(TSortType(sortTypes.size(), NTravelProto::NGeoCounter::ESortType::ST_ByRating, 0, {}, [ratingFeatureId](const THotelSortData& lhs, const THotelSortData& rhs) {
            const auto& lhsRating = GetRating(lhs.HotelAltayData->Features, ratingFeatureId);
            const auto& rhsRating = GetRating(rhs.HotelAltayData->Features, ratingFeatureId);
            if (lhsRating.Empty()) {
                return false;
            }
            if (rhsRating.Empty()) {
                return true;
            }
            return lhsRating.GetRef() > rhsRating.GetRef();
        }));

        // by distance
        sortTypes.push_back(TSortType(sortTypes.size(), NTravelProto::NGeoCounter::ESortType::ST_ByDistance, 0, {},
                                      getPreSortByFactor(config.GetSortOptions().GetDefaultRankingFactorName()), {}, {},
                                      TSortType::TRealTimeSortInIndex {
            [](const TFilteringPermalinkInfo& filteringInfo, const TSortType::TSortContext& context) {
                Y_ENSURE(context.SortOrigin.Defined(), "Sort origin is required for sort by distance");
                auto distance = GetDistanceMeters(context.SortOrigin.GetRef(), filteringInfo.StaticPermalinkInfo.Position);
                return static_cast<i64>(distance);
            },
            [](TVector<THotelsResults::THotelResult>* hotels, const THashMap<TPermalink, i64>& factors) {
                Sort(hotels->begin(), hotels->end(),
                     [&factors](const THotelsResults::THotelResult& lhs, const THotelsResults::THotelResult& rhs) {
                         if (lhs.IsTopHotel && !rhs.IsTopHotel) {
                             return true;
                         }
                         if (!lhs.IsTopHotel && rhs.IsTopHotel) {
                             return false;
                         }
                         return factors.at(lhs.Permalink) < factors.at(rhs.Permalink);
                     });
            }
        }));

        Y_ENSURE(sortTypes.size() <= g_SortTypeCount, "Too many sorts, increase g_SortTypeCount");

        return TSortTypeRegistry(sortTypes, defaultSortTypeIndex);
    }
}
