package ru.yandex.travel.hotels.administrator

import io.grpc.stub.StreamObserver
import org.apache.commons.lang3.StringUtils
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.transaction.TransactionStatus
import org.springframework.transaction.support.TransactionTemplate
import ru.yandex.travel.commons.grpc.ServerUtils
import ru.yandex.travel.grpc.GrpcService
import ru.yandex.travel.hotels.administrator.configuration.BillingServiceProperties
import ru.yandex.travel.hotels.administrator.entity.*
import ru.yandex.travel.hotels.administrator.grpc.proto.*
import ru.yandex.travel.hotels.administrator.grpc.proto.AdministratorAdminServiceInterfaceGrpc.AdministratorAdminServiceInterfaceImplBase
import ru.yandex.travel.hotels.administrator.repository.AgreementRepository
import ru.yandex.travel.hotels.administrator.repository.HotelConnectionRepository
import ru.yandex.travel.hotels.administrator.repository.HotelConnectionUpdateRepository
import ru.yandex.travel.hotels.administrator.repository.LegalDetailsRepository
import ru.yandex.travel.hotels.administrator.service.BillingService
import ru.yandex.travel.hotels.administrator.service.Meters
import ru.yandex.travel.hotels.administrator.service.StarTrekService
import ru.yandex.travel.hotels.administrator.workflow.proto.*
import ru.yandex.travel.integration.balance.model.BillingPerson
import ru.yandex.travel.workflow.WorkflowMessageSender
import ru.yandex.travel.workflow.WorkflowProcessService
import ru.yandex.travel.workflow.entities.Workflow
import ru.yandex.travel.workflow.repository.WorkflowRepository
import java.util.*
import java.util.function.Function


/*
Outdated service with no authorization (for cli-usage).
Should be eventually replaced by AdministratorInternalService
 */
@GrpcService(authenticateService = true)
class AdministratorAdminService(
        private val transactionTemplate: TransactionTemplate,
        private val hotelConnectionRepository: HotelConnectionRepository,
        private val legalDetailsRepository: LegalDetailsRepository,
        private val workflowProcessService: WorkflowProcessService,
        private val workflowRepository: WorkflowRepository,
        private val agreementRepository: AgreementRepository,
        private val hotelConnectionUpdateRepository: HotelConnectionUpdateRepository,
        private val starTrekService: StarTrekService,
        private val billingService: BillingService,
        private val billingServiceProperties: BillingServiceProperties,
        private val workflowMessageSender: WorkflowMessageSender,
        private val meters: Meters
    ) : AdministratorAdminServiceInterfaceImplBase() {

    val log: Logger = LoggerFactory.getLogger(javaClass)

    override fun changeHotelConnectionStatus(request: TChangeHotelConnectionStatusReq,
                                             responseObserver: StreamObserver<TChangeHotelConnectionStatusRsp>) {
        synchronouslyWithTx(request, responseObserver, ERROR_CHANGE_CONNECTION_STATUS_TAG, { req: TChangeHotelConnectionStatusReq ->
            val hotelConn = hotelConnectionRepository.findByPartnerIdAndHotelCode(req.partnerId, req.hotelCode)
                    ?: return@synchronouslyWithTx TChangeHotelConnectionStatusRsp.newBuilder()
                            .setResult(EChangeHotelConnectionStatusResult.CCS_HOTEL_NOT_FOUND).build()

            hotelConn.state = req.connectionState
            if (req.unpublishedReason != EInternalUnpublishedReason.IUR_UNKNOWN) {
                hotelConn.unpublishedReason = UnpublishedReason.convertFromProto(req.unpublishedReason)
            }
            workflowProcessService.scheduleEvent(hotelConn.workflow.id,
                    TNotifyConnectionStatusChanged.newBuilder().build())
            return@synchronouslyWithTx TChangeHotelConnectionStatusRsp.newBuilder()
                    .setResult(EChangeHotelConnectionStatusResult.CCS_SUCCESS).build()
        })
    }

    override fun createLegalDetailsFromBalance(request: TCreateLegalDetailsReq, responseObserver: StreamObserver<TCreateLegalDetailsRsp>) {
        synchronouslyWithTx(request, responseObserver, ERROR_CREATE_LEGAL_DETAILS, { req: TCreateLegalDetailsReq ->
            val clientId = req.billingClientId
            val contractId = req.billingContractId
            val personId = req.billingPersonId

            legalDetailsRepository.findByBalanceClientIdAndBalanceContractIdAndBalancePersonId(
                    clientId, contractId, personId)
                ?: return@synchronouslyWithTx TCreateLegalDetailsRsp.newBuilder()
                        .setResult(ECreateLegalDetailsResult.CLD_CONFLICT).build()

            val responseBuilder = TCreateLegalDetailsRsp.newBuilder()
            val billingClient = billingService.getClient(clientId)
            val billingContract = billingService.getContract(clientId, contractId)
            val billingPerson = billingService.getPerson(personId)
            if (billingClient == null || billingContract == null || billingPerson == null) {
                responseBuilder.result = ECreateLegalDetailsResult.CLD_BALANCE_MISSING
                return@synchronouslyWithTx responseBuilder.build()
            }
            if (billingContract.personId != personId || billingPerson.clientId != clientId) {
                responseBuilder.result = ECreateLegalDetailsResult.CLD_BALANCE_UNBOUND
                return@synchronouslyWithTx responseBuilder.build()
            }
            var legalDetails = LegalDetails()
            legalDetails.id = UUID.randomUUID()
            legalDetails.inn = billingPerson.inn
            legalDetails.kpp = billingPerson.kpp
            legalDetails.bic = billingPerson.bik
            legalDetails.paymentAccount = billingPerson.account
            legalDetails.legalName = billingPerson.name
            legalDetails.fullLegalName = billingPerson.longName
            legalDetails.legalAddress = extractLegalAddress(billingPerson)
            legalDetails.legalAddressUnified = false
            legalDetails.offerAccepted = true
            legalDetails.balanceClientId = clientId
            legalDetails.registeredInBalance = true
            legalDetails.state = ELegalDetailsState.DS_REGISTERED
            legalDetails.balanceContractId = contractId
            legalDetails.partnerId = req.partnerId
            legalDetails.postCode = billingPerson.postCode
            legalDetails.postAddress = billingPerson.postSuffix
            legalDetails.phone = billingPerson.phone
            legalDetails.balancePersonId = personId
            legalDetails.balanceExternalContractId = billingContract.externalId
            legalDetails.registeredAt = billingContract.dt.toInstant(billingServiceProperties.billingZoneOffset)
            legalDetails.managedByAdministrator = false
            legalDetails.stTicket = starTrekService.createOrUpdateLegalDetailsTicket(legalDetails)
            legalDetails = legalDetailsRepository.saveAndFlush(legalDetails)

            val workflow = Workflow.createWorkflowForEntity(legalDetails, KnownWorkflow.GENERIC_SUPERVISOR.uuid)
            workflowRepository.saveAndFlush(workflow)
            val agreement = Agreement()
            agreement.id = UUID.randomUUID()
            agreement.inn = billingPerson.inn
            agreement.agreementType = AgreementType.AGREEMENT
            agreement.partnerId = req.partnerId
            agreement.active = true
            agreement.blocked = false
            agreement.kpp = billingPerson.kpp
            agreement.bic = billingPerson.bik
            agreement.paymentAccount = billingPerson.account
            agreement.balanceClientId = clientId
            agreement.balanceContractId = contractId
            agreement.balancePersonId = personId
            agreement.balanceExternalContractId = billingContract.externalId

            agreementRepository.saveAndFlush(agreement)
            responseBuilder.result = ECreateLegalDetailsResult.CLD_CREATED
            responseBuilder.id = legalDetails.id.toString()
            responseBuilder.build()
        })
    }

    private fun extractLegalAddress(billingPerson: BillingPerson): String {
        if (StringUtils.isNotBlank(billingPerson.legalAddress)) {
            return billingPerson.legalAddress
        }
        val legalAddress = StringBuilder()
        legalAddress.append(billingPerson.legalAddressPostcode)
        if (StringUtils.isNotBlank(billingPerson.legalAddressCity)) {
            legalAddress.append(", ").append(billingPerson.legalAddressCity)
        }
        if (StringUtils.isNotBlank(billingPerson.legalAddressStreet)) {
            legalAddress.append(", ").append(billingPerson.legalAddressStreet)
        }
        if (StringUtils.isNotBlank(billingPerson.legalAddressHome)) {
            legalAddress.append(", ").append(billingPerson.legalAddressHome)
        }
        return legalAddress.toString()
    }

    override fun updateLegalDetailsFromBalance(request: TUpdateLegalDetailsReq, responseObserver: StreamObserver<TUpdateLegalDetailsRsp>) {
        synchronouslyWithTx(request, responseObserver, ERROR_CREATE_LEGAL_DETAILS, { req: TUpdateLegalDetailsReq ->
            val legalDetails = legalDetailsRepository.findById(UUID.fromString(req.id)).orElse(null)
                    ?: return@synchronouslyWithTx TUpdateLegalDetailsRsp.newBuilder()
                            .setResult(EUpdateLegalDetailsResult.ULD_NOT_FOUND)
                            .build()
            val clientId = legalDetails.balanceClientId
            val contractId = legalDetails.balanceContractId
            val personId = legalDetails.balancePersonId
            if (clientId == null || contractId == null || personId == null) {
                return@synchronouslyWithTx TUpdateLegalDetailsRsp.newBuilder()
                        .setResult(EUpdateLegalDetailsResult.ULD_INCOMPLETE)
                        .build()
            }
            val billingContract = billingService.getContract(clientId, contractId)
            val billingPerson = billingService.getPerson(personId)
            legalDetails.inn = billingPerson.inn
            legalDetails.kpp = billingPerson.kpp
            legalDetails.bic = billingPerson.bik
            legalDetails.paymentAccount = billingPerson.account
            legalDetails.legalName = billingPerson.name
            legalDetails.fullLegalName = billingPerson.longName
            legalDetails.legalAddress = extractLegalAddress(billingPerson)
            legalDetails.postCode = billingPerson.postCode
            legalDetails.postAddress = billingPerson.postSuffix
            legalDetails.phone = billingPerson.phone
            legalDetails.balanceExternalContractId = billingContract.externalId
            legalDetails.registeredAt = billingContract.dt.toInstant(billingServiceProperties.billingZoneOffset)
            legalDetails.stTicket = starTrekService.createOrUpdateLegalDetailsTicket(legalDetails)
            legalDetailsRepository.saveAndFlush(legalDetails)
            TUpdateLegalDetailsRsp.newBuilder()
                    .setResult(EUpdateLegalDetailsResult.ULD_UPDATED)
                    .build()
        })
    }

    override fun acceptHotelConnectionUpdate(request: TAcceptConnectionUpdateReq,
                                             responseObserver: StreamObserver<TAcceptConnectionUpdateRsp>) {
        synchronouslyWithTx(request, responseObserver, ERROR_ACCEPT_UPDATE_TAG, { req: TAcceptConnectionUpdateReq ->
            val connectionUpdateId = UUID.fromString(req.updateId)
            val connectionUpdate = hotelConnectionUpdateRepository.findById(connectionUpdateId)
            if (connectionUpdate.isPresent) {
                workflowMessageSender.scheduleEvent(connectionUpdate.get().workflow.id,
                        TApplyConnectionUpdate.newBuilder()
                                .setBankChangeInplaceMode(req.bankChangeInplaceMode)
                                .build())
                return@synchronouslyWithTx TAcceptConnectionUpdateRsp.newBuilder()
                        .setStatus(EAcceptUpdateStatus.AUS_ACCEPTED)
                        .build()
            } else {
                return@synchronouslyWithTx TAcceptConnectionUpdateRsp.newBuilder()
                        .setStatus(EAcceptUpdateStatus.AUS_NOT_FOUND)
                        .build()
            }
        })
    }

    override fun rejectHotelConnectionUpdate(request: TRejectConnectionUpdateReq,
                                             responseObserver: StreamObserver<TRejectConnectionUpdateRsp>) {
        synchronouslyWithTx(request, responseObserver, ERROR_REJECT_UPDATE_TAG, { req: TRejectConnectionUpdateReq ->
            val connectionUpdateId = UUID.fromString(req.updateId)
            val connectionUpdate = hotelConnectionUpdateRepository.findById(connectionUpdateId)
            if (connectionUpdate.isPresent) {
                workflowMessageSender.scheduleEvent(connectionUpdate.get().workflow.id,
                        TRejectConnectionUpdate.newBuilder().build())
                return@synchronouslyWithTx TRejectConnectionUpdateRsp.newBuilder()
                        .setStatus(ERejectUpdateStatus.RUS_REJECTED)
                        .build()
            } else {
                return@synchronouslyWithTx TRejectConnectionUpdateRsp.newBuilder()
                        .setStatus(ERejectUpdateStatus.RUS_NOT_FOUND)
                        .build()
            }
        })
    }

    private fun <ReqT, RspT> synchronouslyWithTx(request: ReqT, observer: StreamObserver<RspT>, errorTag: String,
                                                 handler: Function<ReqT, RspT>) {
        ServerUtils.synchronously(log, request, observer,
                { rq: ReqT ->
                    try {
                        return@synchronously transactionTemplate.execute { ignored: TransactionStatus? -> handler.apply(rq) }
                    } catch (e: Exception) {
                        meters.incrementCounter(errorTag)
                        throw e
                    }
                })
    }

    companion object {
        private const val ERROR_CHANGE_CONNECTION_STATUS_TAG = "GrpcChangeConnectionStatus"
        private const val ERROR_CREATE_LEGAL_DETAILS = "GrpcCreateLegalDetailsFromBalance"
        private const val ERROR_UPDATE_LEGAL_DETAILS = "GrpcUpdateLegalDetailsFromBalance"
        private const val ERROR_GET_HOTEL_CONNECTION_DETAILS = "GrpcGetHotelConnectionDetails"
        private const val ERROR_ACCEPT_UPDATE_TAG = "GrpcAcceptHotelConnectionUpdate"
        private const val ERROR_REJECT_UPDATE_TAG = "GrpcRejectHotelConnectionUpdate"
    }

    init {
        meters.initCounter(ERROR_CHANGE_CONNECTION_STATUS_TAG)
        meters.initCounter(ERROR_CREATE_LEGAL_DETAILS)
        meters.initCounter(ERROR_UPDATE_LEGAL_DETAILS)
        meters.initCounter(ERROR_GET_HOTEL_CONNECTION_DETAILS)
    }
}
