package ru.yandex.travel.hotels.administrator

import io.grpc.Status
import io.grpc.stub.StreamObserver
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.transaction.TransactionStatus
import org.springframework.transaction.support.TransactionTemplate
import ru.yandex.travel.commons.grpc.ServerUtils
import ru.yandex.travel.grpc.GrpcService
import ru.yandex.travel.hotels.administrator.entity.*
import ru.yandex.travel.hotels.administrator.grpc.proto.*
import ru.yandex.travel.hotels.administrator.grpc.proto.AdministratorInternalServiceInterfaceGrpc.AdministratorInternalServiceInterfaceImplBase
import ru.yandex.travel.hotels.administrator.repository.HotelConnectionRepository
import ru.yandex.travel.hotels.administrator.service.AuthService
import ru.yandex.travel.hotels.administrator.service.Meters
import ru.yandex.travel.hotels.administrator.workflow.proto.*
import ru.yandex.travel.workflow.WorkflowMessageSender
import java.util.*
import java.util.function.Function


@GrpcService(authenticateService = true, authenticateUser = true)
class AdministratorInternalService(
        private val workflowMessageSender: WorkflowMessageSender,
        private val transactionTemplate: TransactionTemplate,
        private val hotelConnectionRepository: HotelConnectionRepository,
        private val authService: AuthService,
        private val meters: Meters,
    ) : AdministratorInternalServiceInterfaceImplBase() {

    val log: Logger = LoggerFactory.getLogger(javaClass)

    override fun getHotelConnection(request: GetHotelConnectionReq, responseObserver: StreamObserver<GetHotelConnectionRsp>) {
        synchronouslyWithTx(request, responseObserver, ERROR_GET_HOTEL_CONNECTION, { req: GetHotelConnectionReq ->
            authorizeUserAction(AdminAction.AA_GET_HOTEL_CONNECTION)

            val hotelConn = hotelConnectionRepository.findByPartnerIdAndHotelCode(req.partnerId, req.hotelCode)
                ?: return@synchronouslyWithTx GetHotelConnectionRsp.newBuilder()
                    .setConnectionState(EHotelConnectionState.CS_UNKNOWN).build()

            val resp = GetHotelConnectionRsp.newBuilder()
            resp.connectionState = hotelConn.state
            resp.paperAgreement = hotelConn.paperAgreement
            resp.addAllAccountantEmails(hotelConn.accountantEmails)
            resp.addAllContractPersonEmails(hotelConn.contractPersonEmails)

            hotelConn.hotelName?.let {resp.hotelName = it}
            hotelConn.permalink?.let {resp.permalink = it}
            hotelConn.stTicket?.let {resp.stTicket = it}
            hotelConn.unpublishedReason?.let {resp.unpublishedReason = UnpublishedReason.convertToProto(it)}
            hotelConn.legalDetails?.let { ld ->
                ld.balanceClientId?.let {resp.billingClientId = it}
                ld.balanceContractId?.let {resp.billingContractId = it}
            }

            return@synchronouslyWithTx resp.build()
        })
    }

    override fun changeHotelConnectionState(request: ChangeHotelConnectionStateReq,
                                             responseObserver: StreamObserver<ChangeHotelConnectionStateRsp>) {
        synchronouslyWithTx(request, responseObserver, ERROR_CHANGE_HOTEL_CONNECTION_STATE) {req: ChangeHotelConnectionStateReq ->
            authorizeUserAction(AdminAction.AA_CHANGE_HOTEL_CONNECTION_STATE)

            val hotelConn = hotelConnectionRepository.findByPartnerIdAndHotelCode(req.partnerId, req.hotelCode)
                ?: run{ throw Status.NOT_FOUND.withDescription(
                    "Hotel connection not found: partner ${req.partnerId}, code: ${req.hotelCode}")
                    .asRuntimeException() }

            if (hotelConn.state == req.connectionState) {
                throw Status.FAILED_PRECONDITION.withDescription(
                    "Hotel connection is in a '${req.connectionState}' state already ").asRuntimeException()
            }

            val prevState = hotelConn.state
            hotelConn.state = req.connectionState
            if (req.unpublishedReason != EInternalUnpublishedReason.IUR_UNKNOWN) {
                hotelConn.unpublishedReason = UnpublishedReason.convertFromProto(req.unpublishedReason)
            }
            workflowMessageSender.scheduleEvent(
                hotelConn.workflow.id, TNotifyConnectionStatusChanged.newBuilder().build())

            return@synchronouslyWithTx ChangeHotelConnectionStateRsp.newBuilder()
                .setPreviousState(prevState)
                .setNewState(hotelConn.state).build()
        }
    }

    private fun authorizeUserAction(action: AdminAction) {
        if (!authService.userCanDoAction(action)) {
            throw Status.PERMISSION_DENIED.withDescription("User is not authorized for $action").asRuntimeException()
        }
    }

    private fun <ReqT, RspT> synchronouslyWithTx(request: ReqT, observer: StreamObserver<RspT>, errorTag: String,
                                                 handler: Function<ReqT, RspT>) {
        ServerUtils.synchronously(log, request, observer,
            { rq: ReqT ->
                try {
                    return@synchronously transactionTemplate.execute { ignored: TransactionStatus? -> handler.apply(rq) }
                } catch (ex: Exception) {
                    meters.incrementCounter(errorTag)
                    observer.onError(ex)
                    throw ex
                }
            })
    }

    companion object {
        private const val ERROR_GET_HOTEL_CONNECTION = "GrpcGetHotelConnection"
        private const val ERROR_CHANGE_HOTEL_CONNECTION_STATE = "GrpcChangeHotelConnectionState"
    }

    init {
        meters.initCounter(ERROR_GET_HOTEL_CONNECTION)
        meters.initCounter(ERROR_CHANGE_HOTEL_CONNECTION_STATE)
    }
}
