package ru.yandex.travel.hotels.administrator;

import java.util.List;
import java.util.UUID;
import java.util.function.Function;

import io.grpc.stub.StreamObserver;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.hotels.administrator.entity.HotelConnection;
import ru.yandex.travel.hotels.administrator.entity.KnownWorkflow;
import ru.yandex.travel.hotels.administrator.grpc.proto.AdministratorInterfaceGrpc;
import ru.yandex.travel.hotels.administrator.grpc.proto.EHotelStatus;
import ru.yandex.travel.hotels.administrator.grpc.proto.EUnpublishedReason;
import ru.yandex.travel.hotels.administrator.grpc.proto.TAcceptAgreementReq;
import ru.yandex.travel.hotels.administrator.grpc.proto.TAcceptAgreementRsp;
import ru.yandex.travel.hotels.administrator.grpc.proto.TAgreementStatusRsp;
import ru.yandex.travel.hotels.administrator.grpc.proto.TCheckAgreementReq;
import ru.yandex.travel.hotels.administrator.grpc.proto.THotelDetailsChangedReq;
import ru.yandex.travel.hotels.administrator.grpc.proto.THotelDetailsChangedRsp;
import ru.yandex.travel.hotels.administrator.grpc.proto.THotelStatusByPermalinkReq;
import ru.yandex.travel.hotels.administrator.grpc.proto.THotelStatusReq;
import ru.yandex.travel.hotels.administrator.grpc.proto.THotelStatusRsp;
import ru.yandex.travel.hotels.administrator.repository.HotelConnectionRepository;
import ru.yandex.travel.hotels.administrator.service.AgreementService;
import ru.yandex.travel.hotels.administrator.service.Meters;
import ru.yandex.travel.hotels.administrator.util.HotelConnectionComparator;
import ru.yandex.travel.hotels.administrator.workflow.proto.EConnectionStepState;
import ru.yandex.travel.hotels.administrator.workflow.proto.EHotelConnectionState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TActualizeHotelConnection;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Slf4j
@GrpcService(authenticateService = true)
public class AdministratorService extends AdministratorInterfaceGrpc.AdministratorInterfaceImplBase {

    private static final String ERROR_HOTEL_DETAILS_CHANGED_TAG = "GrpcHotelDetailsChanged";
    private static final String ERROR_HOTEL_STATUS_TAG = "GrpcHotelStatus";
    private static final String ERROR_AGREEMENT_STATUS_TAG = "GrpcAgreementStatus";
    private static final String ERROR_ACCEPT_AGREEMENT_TAG = "GrpcAcceptAgreement";

    private final TransactionTemplate transactionTemplate;
    private final HotelConnectionRepository hotelConnectionRepository;
    private final WorkflowRepository workflowRepository;
    private final WorkflowMessageSender workflowMessageSender;
    private final AgreementService agreementService;
    private final Meters meters;

    public AdministratorService(TransactionTemplate transactionTemplate,
                                HotelConnectionRepository hotelConnectionRepository,
                                WorkflowRepository workflowRepository, WorkflowMessageSender workflowMessageSender,
                                AgreementService agreementService, Meters meters) {
        this.transactionTemplate = transactionTemplate;
        this.hotelConnectionRepository = hotelConnectionRepository;
        this.workflowRepository = workflowRepository;
        this.workflowMessageSender = workflowMessageSender;
        this.agreementService = agreementService;
        this.meters = meters;
        this.meters.initCounter(ERROR_HOTEL_STATUS_TAG);
        this.meters.initCounter(ERROR_HOTEL_DETAILS_CHANGED_TAG);
        this.meters.initCounter(ERROR_AGREEMENT_STATUS_TAG);
        this.meters.initCounter(ERROR_ACCEPT_AGREEMENT_TAG);
    }

    @Override
    public void hotelDetailsChanged(THotelDetailsChangedReq request,
                                    StreamObserver<THotelDetailsChangedRsp> responseObserver) {
        synchronouslyWithTx(request, responseObserver, ERROR_HOTEL_DETAILS_CHANGED_TAG, req -> {
            String hotelCode = req.getHotelCode();
            EPartnerId partnerId = req.getPartnerId();
            HotelConnection hotelConnection = hotelConnectionRepository.findByPartnerIdAndHotelCode(partnerId,
                    hotelCode);
            if (hotelConnection == null) {
                UUID connectionId = UUID.randomUUID();
                hotelConnection = new HotelConnection();
                hotelConnection.setHotelCode(hotelCode);
                hotelConnection.setPartnerId(partnerId);
                hotelConnection.setId(connectionId);
                hotelConnection.setState(EHotelConnectionState.CS_NEW);
                hotelConnection.setPaperAgreement(false);
                hotelConnection.setAvailableAtPartner(true);
                Workflow connectionWorkflow = Workflow.createWorkflowForEntity(hotelConnection,
                        KnownWorkflow.GENERIC_SUPERVISOR.getUuid());
                workflowRepository.saveAndFlush(connectionWorkflow);
                hotelConnectionRepository.saveAndFlush(hotelConnection);
            }
            workflowMessageSender.scheduleEvent(hotelConnection.getWorkflow().getId(),
                    TActualizeHotelConnection.newBuilder().build());
            return THotelDetailsChangedRsp.newBuilder().build();
        });
    }

    @Override
    public void hotelStatus(THotelStatusReq request, StreamObserver<THotelStatusRsp> responseObserver) {
        synchronouslyWithTx(request, responseObserver, ERROR_HOTEL_STATUS_TAG, req -> {
            HotelConnection hotelConnection =
                    hotelConnectionRepository.findByPartnerIdAndHotelCode(req.getPartnerId(), req.getHotelCode());
            THotelStatusRsp.Builder hotelStatusResponseBuilder = THotelStatusRsp.newBuilder()
                    .setHotelCode(req.getHotelCode());
            if (hotelConnection == null) {
                return hotelStatusResponseBuilder
                        .setHotelStatus(EHotelStatus.H_NOT_FOUND)
                        .setUnpublishedReason(EUnpublishedReason.UR_NONE)
                        .build();
            } else {
                if (hotelConnection.getPermalink() != null) {
                    hotelStatusResponseBuilder.setPermalink(hotelConnection.getPermalink());
                }
                hotelStatusResponseBuilder.setUnpublishedReason(EUnpublishedReason.UR_NONE);
                switch (hotelConnection.getState()) {
                    case CS_NEW:
                    case CS_PUBLISHING:
                        hotelStatusResponseBuilder.setHotelStatus(EHotelStatus.H_PUBLISHING);
                        if (existsConnectionStepWithManualTicket(hotelConnection)
                                || !hotelConnection.getWorkflow().getState().equals(EWorkflowState.WS_RUNNING)) {
                            hotelStatusResponseBuilder.setUnpublishedReason(EUnpublishedReason.UR_DELAYED);
                        }
                        break;
                    case CS_PUBLISHED:
                        hotelStatusResponseBuilder.setHotelStatus(EHotelStatus.H_PUBLISHED);
                        if (hotelConnection.getLegalDetails() != null) {
                            if (hotelConnection.getLegalDetails().getBalanceExternalContractId() != null) {
                                hotelStatusResponseBuilder.setAgreementId(hotelConnection.getLegalDetails().getBalanceExternalContractId());
                            }
                            if (hotelConnection.getLegalDetails().getRegisteredAt() != null) {
                                hotelStatusResponseBuilder.setAgreementFrom(ProtoUtils.fromInstant(hotelConnection.getLegalDetails().getRegisteredAt()));
                            }
                        }
                        break;
                    case CS_MANUAL_VERIFICATION:
                        hotelStatusResponseBuilder.setHotelStatus(EHotelStatus.H_UNPUBLISHED);
                        hotelStatusResponseBuilder.setUnpublishedReason(EUnpublishedReason.UR_UPDATING);
                        break;
                    case CS_UNPUBLISHED:
                        hotelStatusResponseBuilder.setHotelStatus(EHotelStatus.H_UNPUBLISHED);
                        hotelStatusResponseBuilder.setUnpublishedReason(defineUnpublishedReason(hotelConnection));
                        break;
                    default:
                        throw new RuntimeException("Unexpected hotel connection state: " + hotelConnection.getState());
                }
                return hotelStatusResponseBuilder.build();
            }
        });
    }

    @Override
    public void hotelStatusByPermalink(THotelStatusByPermalinkReq request,
                                       StreamObserver<THotelStatusRsp> responseObserver) {
        synchronouslyWithTx(request, responseObserver, ERROR_HOTEL_STATUS_TAG, req -> {
            List<HotelConnection> hotelConnections = hotelConnectionRepository.findByPermalink(req.getPermalink());

            if (hotelConnections.isEmpty()) {
                return THotelStatusRsp
                        .newBuilder()
                        .setPermalink(req.getPermalink())
                        .setHotelStatus(EHotelStatus.H_NOT_FOUND)
                        .build();
            }

            hotelConnections.sort(HotelConnectionComparator.INSTANCE);

            HotelConnection hc = hotelConnections.get(0);

            THotelStatusRsp.Builder hotelStatusResponseBuilder = THotelStatusRsp.newBuilder()
                    .setHotelCode(hc.getHotelCode())
                    .setPartnerId(hc.getPartnerId())
                    .setPermalink(hc.getPermalink());
            hotelStatusResponseBuilder.setUnpublishedReason(EUnpublishedReason.UR_NONE);
            switch (hc.getState()) {
                case CS_NEW:
                case CS_PUBLISHING:
                    hotelStatusResponseBuilder.setHotelStatus(EHotelStatus.H_PUBLISHING);
                    if (existsConnectionStepWithManualTicket(hc)
                            || !hc.getWorkflow().getState().equals(EWorkflowState.WS_RUNNING)) {
                        hotelStatusResponseBuilder.setUnpublishedReason(EUnpublishedReason.UR_DELAYED);
                    }
                    break;
                case CS_PUBLISHED:
                    hotelStatusResponseBuilder.setHotelStatus(EHotelStatus.H_PUBLISHED);
                    if (hc.getLegalDetails() != null) {
                        if (hc.getLegalDetails().getBalanceExternalContractId() != null) {
                            hotelStatusResponseBuilder.setAgreementId(hc.getLegalDetails().getBalanceExternalContractId());
                        }
                        if (hc.getLegalDetails().getRegisteredAt() != null) {
                            hotelStatusResponseBuilder.setAgreementFrom(ProtoUtils.fromInstant(hc.getLegalDetails().getRegisteredAt()));
                        }
                    }
                    break;
                case CS_MANUAL_VERIFICATION:
                    hotelStatusResponseBuilder.setHotelStatus(EHotelStatus.H_UNPUBLISHED);
                    hotelStatusResponseBuilder.setUnpublishedReason(EUnpublishedReason.UR_UPDATING);
                    break;
                case CS_UNPUBLISHED:
                    hotelStatusResponseBuilder.setHotelStatus(EHotelStatus.H_UNPUBLISHED);
                    if (hc.getLegalDetails() != null) {
                        if (hc.getLegalDetails().getBalanceExternalContractId() != null) {
                            hotelStatusResponseBuilder.setAgreementId(hc.getLegalDetails().getBalanceExternalContractId());
                        }
                        if (hc.getLegalDetails().getRegisteredAt() != null) {
                            hotelStatusResponseBuilder.setAgreementFrom(ProtoUtils.fromInstant(hc.getLegalDetails().getRegisteredAt()));
                        }
                    }
                    hotelStatusResponseBuilder.setUnpublishedReason(defineUnpublishedReason(hc));
                    break;
                default:
                    throw new RuntimeException("Unexpected hotel connection state: " + hc.getState());
            }
            return hotelStatusResponseBuilder.build();
        });
    }

    private boolean existsConnectionStepWithManualTicket(HotelConnection hotelConnection) {
        if (hotelConnection.getConnectionSteps().stream()
                .anyMatch(step -> step.getState().equals(EConnectionStepState.CSS_WAIT_FOR_TICKET_RESOLUTION))) {
            return true;
        }
        if (hotelConnection.getLegalDetails() != null && hotelConnection.getLegalDetails().getConnectionSteps().stream()
                .anyMatch(step -> step.getState().equals(EConnectionStepState.CSS_WAIT_FOR_TICKET_RESOLUTION))) {
            return true;
        }
        return false;
    }

    @Override
    public void agreementStatus(TCheckAgreementReq request, StreamObserver<TAgreementStatusRsp> responseObserver) {
        synchronouslyWithTx(request, responseObserver, ERROR_AGREEMENT_STATUS_TAG,
                req -> TAgreementStatusRsp.newBuilder()
                        .setAgreementStatus(agreementService.checkAgreement(req.getInn()))
                        .build());
    }

    @Override
    public void acceptAgreement(TAcceptAgreementReq request, StreamObserver<TAcceptAgreementRsp> responseObserver) {
        synchronouslyWithTx(request, responseObserver, ERROR_ACCEPT_AGREEMENT_TAG, agreementService::acceptAgreement);
    }

    private EUnpublishedReason defineUnpublishedReason(HotelConnection hotelConnection) {
        if (hotelConnection.getLegalDetails() != null && !hotelConnection.getLegalDetails().isOfferAccepted()) {
            return EUnpublishedReason.UR_NO_AGREEMENT;
        } else {
            return EUnpublishedReason.UR_SUSPENDED;
        }
    }

    private <ReqT, RspT> void synchronouslyWithTx(ReqT request, StreamObserver<RspT> observer, String errorTag,
                                                  Function<ReqT, RspT> handler) {
        ServerUtils.synchronously(log, request, observer,
                rq -> {
                    try {
                        return transactionTemplate.execute(ignored -> handler.apply(rq));
                    } catch (Exception e) {
                        meters.incrementCounter(errorTag);
                        throw e;
                    }
                });
    }
}
