package ru.yandex.travel.hotels.administrator

import io.grpc.stub.StreamObserver
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.transaction.support.TransactionTemplate
import ru.yandex.travel.commons.grpc.ServerUtils
import ru.yandex.travel.grpc.GrpcService
import ru.yandex.travel.hotels.administrator.grpc.proto.*
import ru.yandex.travel.hotels.administrator.service.AuthService
import java.util.function.Function


private val rolesInfoResp = RolesInfoResponse.newBuilder().addAllRoles(listOf(
    IdmRole.newBuilder()
        .setType(AdminRole.AR_DEVELOPER)
        .setNames(IdmNames.newBuilder()
            .setNameEn("Developer")
            .setNameRu("Разработчик").build()).build(),
    IdmRole.newBuilder()
        .setType(AdminRole.AR_BIZDEV)
        .setNames(IdmNames.newBuilder()
            .setNameEn("Bizdev")
            .setNameRu("Биздев").build()).build())).build()


@GrpcService(authenticateService = true)
class IdmApiGrpcImpl @Autowired constructor(
    private val authService: AuthService,
    private val transactionTemplate: TransactionTemplate,
) : IdmApiGrpc.IdmApiImplBase() {

    private val log = LoggerFactory.getLogger(javaClass)

    override fun rolesInfo(request: RolesInfoRequest, responseObserver: StreamObserver<RolesInfoResponse>) {
        synchronouslyWithTx(request, responseObserver, {rolesInfoResp})
    }

    override fun addRole(request: AddRoleRequest, responseObserver: StreamObserver<AddRoleResponse>) {
        synchronouslyWithTx(request, responseObserver, {
            val added = authService.grantAuthorization(it.login, it.passportUid, it.role)
            AddRoleResponse.newBuilder().setAdded(added).build()
        })
    }

    override fun removeRole(request: RemoveRoleRequest, responseObserver: StreamObserver<RemoveRoleResponse>) {
        synchronouslyWithTx(request, responseObserver, {
            val removed = authService.revokeAuthorization(it.login, it.role)
            RemoveRoleResponse.newBuilder().setRemoved(removed).build()
        })
    }

    override fun getAllRoles(request: GetAllRolesRequest, responseObserver: StreamObserver<GetAllRolesResponse>) {
        synchronouslyWithTx(request, responseObserver, {
            val roles = authService.getAllRoles().map {
                IdmRoleUser.newBuilder()
                    .setType(it.role)
                    .setLogin(it.login).build()
            }
            GetAllRolesResponse.newBuilder().addAllUsers(roles).build()
        })
    }

    private fun <ReqT, RspT> synchronouslyWithTx(request: ReqT, observer: StreamObserver<RspT>,
                                                 handler: Function<ReqT, RspT>) {
        ServerUtils.synchronously(log, request, observer,
            { rq -> return@synchronously transactionTemplate.execute { handler.apply(rq) }})
    }
}
