package ru.yandex.travel.hotels.administrator.service;

import java.io.IOException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import NSprav.UnifierReplyOuterClass;
import com.google.common.base.Preconditions;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;

import ru.yandex.altay.model.SignalOuterClass;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
@RequiredArgsConstructor
public class AddressUnificationService {

    private static final String ORIGIN = "ytravel";

    private final AsyncHttpClientWrapper client;

    private final String baseUrl;

    public CompletableFuture<String> unifyAddress(String addressLine) {
        RequestBuilder httpReq = new RequestBuilder()
                .setUrl(baseUrl + "/unify")
                .addQueryParam("format", "proto")
                .addQueryParam("origin", ORIGIN)
                .addQueryParam("address", addressLine);
        return client.executeRequest(httpReq).thenApply(httpResponse -> {
            checkStatus(httpResponse);
            try {
                UnifierReplyOuterClass.UnifierReply unifierReply =
                        UnifierReplyOuterClass.UnifierReply.parseFrom(httpResponse.getResponseBodyAsBytes());
                if (!unifierReply.getSuccess()) {
                    log.warn("Address unifier was unable to unify the following address: {}", addressLine);
                    return null;
                } else {
                    Preconditions.checkState(!unifierReply.getAddressList().isEmpty(), "There should be at least 1 address");
                    if (unifierReply.getAddressCount() > 1) {
                        log.warn("{} addresses were returned instead of 1. The 1st address will be used. Address: {}",
                                unifierReply.getAddressCount(),
                                addressLine);
                    }
                    SignalOuterClass.Address address = unifierReply.getAddress(0);
                    if (address.getPrecision().equals(SignalOuterClass.Precision.EXACT)) {
                        return address.getOneLine();
                    } else {
                        log.warn("Precision was {} instead of EXACT. Address: {}", address.getPrecision(), addressLine);
                        return null;
                    }
                }
            } catch (InvalidProtocolBufferException e) {
                throw new RuntimeException("Unable to deserialize Unifier proto response", e);
            }
        });
    }

    public String unifyAddressSync(String addressLine) {
        try {
            return unifyAddress(addressLine).get();
        } catch (InterruptedException e) {
            log.error("AddressUnify call interrupted", e);
            Thread.currentThread().interrupt(); // preserved interruption status
            throw new RetryableException(e);
        } catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof HttpException
                    || cause instanceof IOException
                    || cause instanceof TimeoutException
            ) {
                throw new RetryableException(e);
            } else {
                throw new RuntimeException(e);
            }
        }
    }

    private void checkStatus(Response response) {
        if (response.getStatusCode() != HttpStatus.SC_200_OK) {
            throw new HttpException(response.getStatusCode(),
                    String.format("UGC returned %s, message '%s'", response.getStatusCode(), response.getResponseBody()));
        }
    }

}
