package ru.yandex.travel.hotels.administrator.service;

import java.util.List;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.administrator.entity.Agreement;
import ru.yandex.travel.hotels.administrator.entity.AgreementType;
import ru.yandex.travel.hotels.administrator.grpc.proto.EAcceptAgreementStatus;
import ru.yandex.travel.hotels.administrator.grpc.proto.EAgreementStatusType;
import ru.yandex.travel.hotels.administrator.grpc.proto.TAcceptAgreementReq;
import ru.yandex.travel.hotels.administrator.grpc.proto.TAcceptAgreementRsp;
import ru.yandex.travel.hotels.administrator.repository.AgreementRepository;

@Service
@RequiredArgsConstructor
public class AgreementService {

    private final AgreementRepository agreementRepository;

    public EAgreementStatusType checkAgreement(String inn) {
        List<Agreement> agreements = agreementRepository.findAllByInnAndActive(inn, true);
        if (agreements.isEmpty()) {
            return EAgreementStatusType.AS_NOT_FOUND;
        }
        if (agreements.stream().anyMatch(agreement -> !agreement.isBlocked())) {
            return EAgreementStatusType.AS_FOUND;
        } else {
            return EAgreementStatusType.AS_BLOCKED;
        }
    }

    public boolean existsPaperAgreementForInn(String inn) {
        return agreementRepository.existsAgreementByInnAndActiveAndAgreementType(inn, true, AgreementType.AGREEMENT);
    }

    public TAcceptAgreementRsp acceptAgreement(TAcceptAgreementReq acceptRequest) {
        List<Agreement> agreements = agreementRepository.findAllByInnAndActive(acceptRequest.getInn(), true);
        if (agreements.isEmpty()) {
            Agreement agreement = Agreement.builder()
                    .id(UUID.randomUUID())
                    .active(true)
                    .blocked(false)
                    .partnerId(acceptRequest.getPartnerId())
                    .inn(acceptRequest.getInn())
                    .hotelCode(acceptRequest.getHotelCode())
                    .agreementType(AgreementType.OFFER)
                    .contactName(acceptRequest.getContactName())
                    .contactPosition(acceptRequest.getContactPosition())
                    .contactPhone(acceptRequest.getContactPhone())
                    .contactEmail(acceptRequest.getContactEmail())
                    .build();
            agreementRepository.saveAndFlush(agreement);
            return TAcceptAgreementRsp.newBuilder()
                    .setStatus(EAcceptAgreementStatus.AAS_SUCCESS)
                    .build();
        }
        if (agreements.stream().anyMatch(Agreement::isBlocked)) {
            return TAcceptAgreementRsp.newBuilder()
                    .setStatus(EAcceptAgreementStatus.AAS_FAILURE)
                    .setMessage("Предоставленный ИНН находится в черном списке")
                    .build();
        } else {
            return TAcceptAgreementRsp.newBuilder()
                    .setStatus(EAcceptAgreementStatus.AAS_FAILURE)
                    .setMessage("Предоставленный ИНН уже имеет активный договор или оферту")
                    .build();
        }
    }

}
