package ru.yandex.travel.hotels.administrator.service

import com.google.common.base.Preconditions
import com.google.common.base.Strings
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.travel.credentials.UserCredentials
import ru.yandex.travel.hotels.administrator.entity.UserRole
import ru.yandex.travel.hotels.administrator.grpc.proto.AdminAction
import ru.yandex.travel.hotels.administrator.grpc.proto.AdminRole
import ru.yandex.travel.hotels.administrator.repository.UserRoleRepository


private val actionsByRole: Map<AdminRole, Set<AdminAction>> = mapOf(
    AdminRole.AR_DEVELOPER to setOf(
        AdminAction.AA_GET_HOTEL_CONNECTION,
        AdminAction.AA_CHANGE_HOTEL_CONNECTION_STATE),
    AdminRole.AR_BIZDEV to setOf(
        AdminAction.AA_GET_HOTEL_CONNECTION,
        AdminAction.AA_CHANGE_HOTEL_CONNECTION_STATE),
)

@Service
open class AuthService @Autowired constructor(
    private val userRoleRepository: UserRoleRepository,
) {
    fun userCanDoAction(action: AdminAction): Boolean {
        val userCredentials = UserCredentials.get()
        checkAuthPreconditions(userCredentials)

        val roles = getUserRoles(userCredentials.passportId)
        return AdminRole.AR_SUPERUSER in roles || action in getActionsForRoles(roles)
    }

    private fun getUserRoles(passportUid: String): Set<AdminRole> {
        return userRoleRepository.findAllByPassportUid(passportUid).map { it.role }.toSet()
    }

    private fun getActionsForRoles(roles: Collection<AdminRole>): Set<AdminAction> {
        return actionsByRole
            .filter { it.key in roles }.values
            .reduceOrNull{resultActions, actions -> resultActions + actions} ?: emptySet()
    }

    private fun checkAuthPreconditions(userCredentials: UserCredentials) {
        Preconditions.checkArgument(
            !Strings.isNullOrEmpty(userCredentials.passportId),
            "userCredentials.passportId must not be null or empty"
        )
    }

    fun grantAuthorization(login: String, passportUid: String, role: AdminRole): Boolean {
        return if (userRoleRepository.findByPassportUidAndRole(passportUid, role) != null) {
            false
        } else {
            val userRole = UserRole(
                login = login,
                passportUid = passportUid,
                role = role,
            )
            userRoleRepository.saveAndFlush(userRole)
            true
        }
    }

    fun revokeAuthorization(login: String, role: AdminRole): Boolean {
        return userRoleRepository.removeAllByLoginAndRole(login, role) > 0
    }

    fun getAllRoles(): Collection<UserRole> {
        return userRoleRepository.findAll()
    }
}
