package ru.yandex.travel.hotels.administrator.service;

import java.time.Instant;
import java.time.LocalDate;
import java.util.List;
import java.util.Set;
import java.util.function.Supplier;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.http.apiclient.HttpApiRetryableException;
import ru.yandex.travel.hotels.administrator.configuration.BillingServiceProperties;
import ru.yandex.travel.hotels.administrator.entity.HotelConnection;
import ru.yandex.travel.hotels.administrator.entity.LegalDetails;
import ru.yandex.travel.integration.balance.BillingApiClient;
import ru.yandex.travel.integration.balance.BillingClientContract;
import ru.yandex.travel.integration.balance.model.BillingClient;
import ru.yandex.travel.integration.balance.model.BillingCommissionType;
import ru.yandex.travel.integration.balance.model.BillingContract;
import ru.yandex.travel.integration.balance.model.BillingOfferConfirmationType;
import ru.yandex.travel.integration.balance.model.BillingPaymentType;
import ru.yandex.travel.integration.balance.model.BillingPerson;
import ru.yandex.travel.integration.balance.model.BillingUrPerson;
import ru.yandex.travel.integration.balance.responses.BillingCreateContractResponse;

import static java.util.stream.Collectors.toSet;

@Service
@RequiredArgsConstructor
@Slf4j
public class BillingService {
    private static final String DEFAULT_CONTRACT_CURRENCY = "RUB";
    private static final long DEFAULT_FIRM_ID = 1; // Yandex
    private static final String EMAIL_SEPARATOR = "; ";

    private final BillingServiceProperties properties;
    private final BillingApiClient client;

    public long createClient(LegalDetails details) {
        return wrapRetryableApiExceptions("createClient",
                () -> client.createClient(properties.getOperatorId(), BillingClient.builder()
                        .name(details.getLegalName())
                        .build()));
    }

    public void updateClient(LegalDetails details) {
        wrapRetryableApiExceptionsVoid("updateClient",
                () -> client.updateClient(properties.getOperatorId(), BillingClient.builder()
                        .clientId(details.getBalanceClientId())
                        .name(details.getLegalName())
                        .build()));
    }

    public BillingClient getClient(long clientId) {
        return wrapRetryableApiExceptions("getClient",
                () -> client.getClient(clientId));
    }

    public BillingClientContract getContract(long clientId, long contractId) {
        return wrapRetryableApiExceptions("getContract",
                () -> client.getClientContracts(clientId)
                        .stream()
                        .filter(contract -> contract.getId().equals(contractId))
                        .findFirst()
                        .orElse(null));
    }

    public BillingPerson getPerson(long personId) {
        return wrapRetryableApiExceptions("getPerson",
                () -> client.getPerson(personId));
    }


    public long createPerson(long clientId, LegalDetails details) {
        Set<String> emails = details.getHotelConnections().stream()
                .map(HotelConnection::getAccountantEmail)
                .collect(toSet());
        return wrapRetryableApiExceptions("createPerson",
                () -> client.createPerson(properties.getOperatorId(), BillingUrPerson.builder()
                        .clientId(clientId)
                        .name(details.getLegalName())
                        .longName(details.getFullLegalName())
                        .inn(details.getInn())
                        .kpp(details.getKpp())
                        .account(details.getPaymentAccount())
                        .phone(details.getPhone())
                        .email(String.join(EMAIL_SEPARATOR, emails))
                        .postCode(details.getPostCode())
                        .postAddress(details.getPostAddress())
                        .legalAddress(details.getLegalAddress())
                        .bik(details.getBic())
                        .account(details.getPaymentAccount())
                        .build()));
    }

    public void updatePerson(LegalDetails details) {
        Set<String> emails = details.getHotelConnections().stream()
                .map(HotelConnection::getAccountantEmail)
                .collect(toSet());
        wrapRetryableApiExceptionsVoid("updatePerson",
                () -> client.updatePerson(properties.getOperatorId(), BillingUrPerson.builder()
                        .personId(details.getBalancePersonId())
                        .clientId(details.getBalanceClientId())
                        .name(details.getLegalName())
                        .longName(details.getFullLegalName())
                        .inn(details.getInn())
                        .kpp(details.getKpp())
                        .account(details.getPaymentAccount())
                        .phone(details.getPhone())
                        .email(String.join(EMAIL_SEPARATOR, emails))
                        .postCode(details.getPostCode())
                        .postAddress(details.getPostAddress())
                        .legalAddress(details.getLegalAddress())
                        .bik(details.getBic())
                        .account(details.getPaymentAccount())
                        .build()));
    }

    public BillingCreateContractResponse createAcceptedOffer(long clientId, long personId) {
        LocalDate today = Instant.now().atZone(BillingApiClient.BILLING_TIMEZONE).toLocalDate();
        return wrapRetryableApiExceptions("createContract",
                () -> client.createContract(properties.getOperatorId(), BillingContract.builder()
                        .clientId(clientId)
                        .personId(personId)
                        .currency(DEFAULT_CONTRACT_CURRENCY)
                        .firmId(DEFAULT_FIRM_ID)
                        .services(List.of(properties.getServiceId()))
                        .paymentType(BillingPaymentType.POSTPAID)
                        .managerCode(properties.getManagerId())
                        .commission(BillingCommissionType.OFFER)
                        .offerConfirmationType(BillingOfferConfirmationType.NO)
                        .paymentTerm(properties.getPaymentTermInDays())
                        .startDate(today)
                        .build()
                ));
    }

    public boolean checkRuBankAccount(String bic, String account) {
        return wrapRetryableApiExceptions("checkRuBankAccount",
                () -> client.checkRUBankAccount(bic, account).getStatus() == 0);
    }

    private void wrapRetryableApiExceptionsVoid(String requestDescription, Runnable action) {
        wrapRetryableApiExceptions(requestDescription, () -> {
            action.run();
            return null;
        });
    }

    private <T> T wrapRetryableApiExceptions(String requestDescription, Supplier<T> action) {
        try {
            log.debug("Calling Billing API: {}", requestDescription);
            return action.get();
        } catch (HttpApiRetryableException e) {
            throw new RetryableServiceException("Billing API request has failed with a retryable exception; " +
                    "call: " + requestDescription, e);
        }
    }
}
