package ru.yandex.travel.hotels.administrator.service;

import java.util.Collections;
import java.util.Optional;
import java.util.UUID;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.tables.types.NativeProtobufEntryType;
import ru.yandex.travel.hotels.administrator.configuration.ClusterizationProperties;
import ru.yandex.travel.hotels.administrator.configuration.YtConnectionProperties;
import ru.yandex.travel.hotels.administrator.export.proto.ClusterizationRequest;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.yt.util.YtHelper;
import ru.yandex.travel.yt.util.YtProtoUtils;

@Slf4j
public class ClusterizationService {

    private static final String FIELD_GROUPING_KEY = "hotels_administrator";
    private static final long FIELD_PRIORITY = Long.MAX_VALUE;
    private static final String FIELD_REQUIRED_STAGES = "clusterization";
    private static final String FIELD_ACTUALIZATION_START_REASON = "main_attribute_actualization";
    private static final String FIELD_CLUSTERIZATION_START_REASON = "boy_hotels";
    private static final String FIELD_ASSIGNEE_SKILL = "advanced";

    private final ClusterizationProperties serviceProperties;
    private final YtConnectionProperties connectionProperties;
    private Yt clusterizationRequestClient;

    public ClusterizationService(
            ClusterizationProperties serviceProperties,
            YtConnectionProperties connectionProperties
    ) {
        this.serviceProperties = serviceProperties;
        this.connectionProperties = connectionProperties;
        String proxy = serviceProperties.getClusterizationRequestProxy();
        if (clusterizationRequestClient == null) {
            clusterizationRequestClient = YtUtils.http(proxy, connectionProperties.getToken());
        }
    }

    @TransactionMandatory
    public void requestClusterization(long permalink, String hotelName, String ticketLink) {
        log.info("Requesting clusterization for {}", permalink);
        YtHelper.doInTx(clusterizationRequestClient, connectionProperties.getTransactionDuration(), txId -> {
            YPath table = YPath
                    .simple(serviceProperties.getClusterizationRequestPath())
                    .child(UUID.randomUUID().toString())
                    .child("permalinks");

            YPath requestTable = YtHelper.createTable(
                    clusterizationRequestClient,
                    txId,
                    false,
                    table,
                    YtProtoUtils.getTableSchemaForMessage(ClusterizationRequest.getDefaultInstance())
            );

            var clusterizationRequestEntryType = new NativeProtobufEntryType<ClusterizationRequest>(
                    ClusterizationRequest.newBuilder()
            );

            ClusterizationRequest request = ClusterizationRequest
                    .newBuilder()
                    .setPermalink(permalink)
                    .setGroupingKey(FIELD_GROUPING_KEY)
                    .setPriority(FIELD_PRIORITY)
                    .setRequiredStages(FIELD_REQUIRED_STAGES)
                    .setActualizationStartReason(FIELD_ACTUALIZATION_START_REASON)
                    .setClusterizationStartReason(FIELD_CLUSTERIZATION_START_REASON)
                    .setHotelName(hotelName)
                    .setAssigneeSkill(FIELD_ASSIGNEE_SKILL)
                    .setRequirements(ticketLink)
                    .build();
            log.info("Sending clusterization request {}", request);

            var records = Collections.singletonList(request);

            clusterizationRequestClient.tables().write(
                    Optional.of(txId),
                    false,
                    requestTable,
                    clusterizationRequestEntryType,
                    records.iterator()
            );
        });
    }
}
