package ru.yandex.travel.hotels.administrator.service;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import javax.persistence.EntityNotFoundException;

import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.administrator.cache.HotelClusteringDictionary;
import ru.yandex.travel.hotels.administrator.configuration.HotelConnectionProperties;
import ru.yandex.travel.hotels.administrator.entity.ConnectionStep;
import ru.yandex.travel.hotels.administrator.entity.VerifyClusteringStep;
import ru.yandex.travel.hotels.administrator.repository.CallGeoSearchStepRepository;
import ru.yandex.travel.hotels.administrator.repository.ClusterizationStepRepository;
import ru.yandex.travel.hotels.administrator.repository.ConnectionStepRepository;
import ru.yandex.travel.hotels.administrator.workflow.proto.EConnectionStepState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TClusterizationFinished;
import ru.yandex.travel.hotels.administrator.workflow.proto.TTicketClosed;
import ru.yandex.travel.hotels.administrator.workflow.proto.TTicketOld;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowProcessService;

@Slf4j
@EnableConfigurationProperties({HotelConnectionProperties.class})
@Service
public class ConnectionStepService {

    private static final String ERROR_COUNTER_TAG = "TicketAwaitingResolutionTaskProcessor";

    private final StarTrekService starTrekService;
    private final HotelClusteringDictionary hotelClusteringDictionary;
    private final ConnectionStepRepository stepRepository;
    private final CallGeoSearchStepRepository callGeoSearchStepRepository;
    private final ClusterizationStepRepository clusterizationStepRepository;
    private final WorkflowProcessService workflowProcessService;
    private final Meters meters;
    private final HotelConnectionProperties hotelConnectionProperties;

    public ConnectionStepService(
            StarTrekService starTrekService,
            HotelClusteringDictionary hotelClusteringDictionary,
            ConnectionStepRepository stepRepository,
            CallGeoSearchStepRepository callGeoSearchStepRepository,
            ClusterizationStepRepository clusterizationStepRepository,
            WorkflowProcessService workflowProcessService,
            Meters meters,
            HotelConnectionProperties hotelConnectionProperties
    ) {
        this.starTrekService = starTrekService;
        this.hotelClusteringDictionary = hotelClusteringDictionary;
        this.stepRepository = stepRepository;
        this.callGeoSearchStepRepository = callGeoSearchStepRepository;
        this.clusterizationStepRepository = clusterizationStepRepository;
        this.workflowProcessService = workflowProcessService;
        this.meters = meters;
        this.meters.initCounter(ERROR_COUNTER_TAG);
        this.hotelConnectionProperties = hotelConnectionProperties;
    }

    public List<UUID> getUnprocessedStartedStepIdsWithClosedTickets(Set<UUID> exclusions, int resultSize) {
        return stepRepository.getAwaitingStepsInStartedStateAndNotSentAnswer(
                EConnectionStepState.CSS_WAIT_FOR_TICKET_RESOLUTION,
                EWorkflowState.WS_RUNNING,
                exclusions,
                PageRequest.of(0, resultSize));
    }

    public long countUnprocessedStartedStepsWithClosedTickets(Set<UUID> exclusions) {
        return stepRepository.countAwaitingStepsInStartedStateAndNotSentAnswer(
                EConnectionStepState.CSS_WAIT_FOR_TICKET_RESOLUTION,
                EWorkflowState.WS_RUNNING,
                exclusions);
    }

    @TransactionMandatory
    public void processStTicketClosedForStep(UUID stepId) {
        try {
            var tooOld = false;
            ConnectionStep step = stepRepository.getOne(stepId);
            try {
                var lastCallAt = callGeoSearchStepRepository.getOne(stepId).getLastGeoSearchCallAt();
                tooOld = lastCallAt != null && Instant.now().minus(hotelConnectionProperties.getGeoSearchCallRetryDelay()).isAfter(lastCallAt);
            } catch (EntityNotFoundException ignored) {
            }
            step.setCheckedAt(Instant.now());
            if (starTrekService.isTicketClosed(step.getStTicket())) {
                step.setTicketResultFetched(true);
                String lastComment = starTrekService.getLastComment(step.getStTicket());
                TTicketClosed.Builder eventBuilder = TTicketClosed.newBuilder();
                if (lastComment != null) {
                    eventBuilder.setLastComment(lastComment);
                }
                workflowProcessService.scheduleEvent(step.getWorkflow().getId(),
                        eventBuilder.build());
            } else if (tooOld) {
                step.setTicketResultFetched(true);
                workflowProcessService.scheduleEvent(step.getWorkflow().getId(),
                        TTicketOld.newBuilder().build());
            }
        } catch (Exception e) {
            meters.incrementCounter(ERROR_COUNTER_TAG);
            throw e;
        }
    }

    public List<UUID> getUnprocessedClusterizationIds(Set<UUID> exclusions, int resultSize) {
        return clusterizationStepRepository.getAwaitingStepsInStartedStateAndNotSentAnswer(
                EConnectionStepState.CSS_WAIT_FOR_CLUSTERIZATION,
                EWorkflowState.WS_RUNNING,
                exclusions,
                PageRequest.of(0, resultSize));
    }

    public long countUnprocessedClusterizationIds(Set<UUID> exclusions) {
        return clusterizationStepRepository.countAwaitingStepsInStartedStateAndNotSentAnswer(
                EConnectionStepState.CSS_WAIT_FOR_CLUSTERIZATION,
                EWorkflowState.WS_RUNNING,
                exclusions);
    }

    @TransactionMandatory
    public void processClusterizationFinished(UUID stepId) {
        try {
            VerifyClusteringStep step = clusterizationStepRepository.getOne(stepId);
            step.setCheckedAt(Instant.now());
            if (hotelClusteringDictionary.isHotelClusteringVerified(step.getPartnerId(), step.getHotelCode())) {
                // we have to materialize control transfer from the task processor back to the workflow handlers
                // to prevent duplicate TClusterizationFinished events, so we moving the step out of the polling state
                step.setState(EConnectionStepState.CSS_COMMIT_CLUSTERIZATION);
                workflowProcessService.scheduleEvent(
                        step.getWorkflow().getId(),
                        TClusterizationFinished.newBuilder().build()
                );
            }
        } catch (Exception e) {
            meters.incrementCounter(ERROR_COUNTER_TAG);
            throw e;
        }
    }
}
