package ru.yandex.travel.hotels.administrator.service;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.Metrics;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.hotels.administrator.configuration.GaugeServiceProperties;
import ru.yandex.travel.hotels.administrator.repository.HotelConnectionRepository;
import ru.yandex.travel.hotels.administrator.repository.LegalDetailsRepository;
import ru.yandex.travel.hotels.administrator.task.BillingContractFlagsSynchronizer;
import ru.yandex.travel.hotels.administrator.workflow.proto.EHotelConnectionState;
import ru.yandex.travel.hotels.administrator.workflow.proto.ELegalDetailsState;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Service
@Slf4j
@EnableConfigurationProperties(GaugeServiceProperties.class)
public class GaugeService implements InitializingBean, DisposableBean {

    private final static String GAUGE_PREFIX = "administrator";
    private static final String ERROR_COUNTER_TAG = "GaugeService";

    private final HotelConnectionRepository hotelConnectionRepository;
    private final LegalDetailsRepository legalDetailsRepository;
    private final TransactionTemplate transactionTemplate;
    private final WorkflowRepository workflowRepository;
    private final BillingContractFlagsSynchronizer billingContractFlagsSynchronizer;
    private final Meters meters;

    private final GaugeServiceProperties properties;
    private final AtomicReference<Map<EHotelConnectionState, Double>> hotelConnectionGauges = new AtomicReference<>(new HashMap<>());
    private final AtomicReference<Map<ELegalDetailsState, Double>> legalDetailsGauges = new AtomicReference<>(new HashMap<>());
    private final AtomicReference<Map<EWorkflowState, Double>> workflowGauges = new AtomicReference<>(new HashMap<>());
    private final AtomicLong billingFlagsSyncDelay = new AtomicLong(0);

    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(
            new ThreadFactoryBuilder()
                    .setNameFormat("gauge-updater")
                    .setDaemon(true)
                    .build());

    public GaugeService(HotelConnectionRepository hotelConnectionRepository,
                        LegalDetailsRepository legalDetailsRepository, TransactionTemplate transactionTemplate,
                        WorkflowRepository workflowRepository,
                        Optional<BillingContractFlagsSynchronizer> billingContractFlagsSynchronizer,
                        Meters meters, GaugeServiceProperties properties) {
        this.hotelConnectionRepository = hotelConnectionRepository;
        this.legalDetailsRepository = legalDetailsRepository;
        this.transactionTemplate = transactionTemplate;
        this.workflowRepository = workflowRepository;
        this.billingContractFlagsSynchronizer = billingContractFlagsSynchronizer.orElse(null);
        this.meters = meters;
        this.properties = properties;
        this.meters.initCounter(ERROR_COUNTER_TAG);
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        for (EHotelConnectionState hotelConnectionState: EHotelConnectionState.values()) {
            Gauge.builder(GAUGE_PREFIX + "." + "hotelConnection",
                        () -> hotelConnectionGauges.get().getOrDefault(hotelConnectionState, 0.0))
                    .tag("state", hotelConnectionState.toString())
                    .register(Metrics.globalRegistry);
        }

        for (ELegalDetailsState eLegalDetailsState: ELegalDetailsState.values()) {
            Gauge.builder(GAUGE_PREFIX + "." + "legalDetails",
                    () -> legalDetailsGauges.get().getOrDefault(eLegalDetailsState, 0.0))
                    .tag("state", eLegalDetailsState.toString())
                    .register(Metrics.globalRegistry);
        }

        for (EWorkflowState eWorkflowState: EWorkflowState.values()) {
            Gauge.builder(GAUGE_PREFIX + "." + "wf",
                    () -> workflowGauges.get().getOrDefault(eWorkflowState, 0.0))
                    .tag("state", eWorkflowState.toString())
                    .register(Metrics.globalRegistry);
        }
        Gauge.builder(GAUGE_PREFIX + ".billingFlagsSyncDelay", billingFlagsSyncDelay::get)
                .register(Metrics.globalRegistry);

        executor.scheduleWithFixedDelay(this::updateFromDb, 0, properties.getUpdateInterval().toMillis(),
                TimeUnit.MILLISECONDS);
    }

    private void updateFromDb() {
        try {
            transactionTemplate.execute(ignored -> {
                updateHotelConnections();
                updateLegalDetails();
                updateWorkflows();
                updateFinancialStats();
                return null;
            });
        } catch (Exception e) {
            log.error("Unable to update db metrics", e);
            meters.incrementCounter(ERROR_COUNTER_TAG);
        }
    }

    private void updateHotelConnections() {
        Map<EHotelConnectionState, Double> newGaugeMap = new HashMap<>();
        hotelConnectionRepository.countHotelConnectionsByState().forEach(row ->
                newGaugeMap.put((EHotelConnectionState) row[0], ((Long) row[1]).doubleValue()));
        hotelConnectionGauges.set(newGaugeMap);
    }

    private void updateLegalDetails() {
        Map<ELegalDetailsState, Double> newGaugeMap = new HashMap<>();
        legalDetailsRepository.countLegalDetailsByState().forEach(row ->
                newGaugeMap.put((ELegalDetailsState) row[0], ((Long) row[1]).doubleValue()));
        legalDetailsGauges.set(newGaugeMap);
    }

    private void updateWorkflows() {
        Map<EWorkflowState, Double> newGaugeMap = new HashMap<>();
        workflowRepository.countWorkflowByState().forEach(row ->
                newGaugeMap.put((EWorkflowState) row[0], ((Long) row[1]).doubleValue()));
        workflowGauges.set(newGaugeMap);
    }

    private void updateFinancialStats() {
        if (billingContractFlagsSynchronizer != null) {
            billingFlagsSyncDelay.set(billingContractFlagsSynchronizer.getCurrentProcessingDelay().toMillis());
        }
    }

    @Override
    public void destroy() throws Exception {
        executor.shutdown();
    }
}
