package ru.yandex.travel.hotels.administrator.service;

import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.PessimisticLockingFailureException;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.administrator.configuration.HotelConnectionProperties;
import ru.yandex.travel.hotels.administrator.entity.HotelConnection;
import ru.yandex.travel.hotels.administrator.entity.HotelConnectionUpdate;
import ru.yandex.travel.hotels.administrator.entity.HotelTaxType;
import ru.yandex.travel.hotels.administrator.entity.KnownWorkflow;
import ru.yandex.travel.hotels.administrator.entity.LegalDetails;
import ru.yandex.travel.hotels.administrator.entity.LegalDetailsUpdate;
import ru.yandex.travel.hotels.administrator.entity.LegalDetailsUpdateState;
import ru.yandex.travel.hotels.administrator.repository.HotelConnectionRepository;
import ru.yandex.travel.hotels.administrator.repository.HotelConnectionUpdateRepository;
import ru.yandex.travel.hotels.administrator.repository.LegalDetailsRepository;
import ru.yandex.travel.hotels.administrator.repository.LegalDetailsUpdateRepository;
import ru.yandex.travel.hotels.administrator.service.partners.PartnerServiceProvider;
import ru.yandex.travel.hotels.administrator.service.partners.model.BankAccountDetailsDTO;
import ru.yandex.travel.hotels.administrator.service.partners.model.ContactType;
import ru.yandex.travel.hotels.administrator.service.partners.model.HotelContactDTO;
import ru.yandex.travel.hotels.administrator.service.partners.model.HotelDetailsDTO;
import ru.yandex.travel.hotels.administrator.workflow.proto.EHotelConnectionState;
import ru.yandex.travel.hotels.administrator.workflow.proto.EHotelConnectionUpdateState;
import ru.yandex.travel.hotels.administrator.workflow.proto.ELegalDetailsState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TRegisterLegalDetails;
import ru.yandex.travel.hotels.administrator.workflow.proto.TUpdateLegalDetails;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.commons.proto.EVat;
import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Service
@Slf4j
@RequiredArgsConstructor
public class HotelConnectionService {

    private static final Pattern LEGAL_NAME_PATTERN = Pattern.compile("^[0-9А-Яа-яa-zA-Z\\s\"-./ёЁ:«»!№]+$");

    private final HotelConnectionRepository hotelConnectionRepository;

    private final WorkflowRepository workflowRepository;

    private final PartnerServiceProvider partnerServiceProvider;

    private final LegalDetailsRepository legalDetailsRepository;

    private final LegalDetailsUpdateRepository legalDetailsUpdateRepository;

    private final HotelConnectionUpdateRepository hotelConnectionUpdateRepository;

    private final AddressUnificationService addressUnificationService;

    private final AgreementService agreementService;

    private final StarTrekService starTrekService;

    private final BillingService billingService;

    private final HotelConnectionProperties hotelConnectionProperties;

    public UpdateResult actualizeHotelConnection(HotelConnection hotelConnection,
                                                 StateContext<EHotelConnectionState, HotelConnection> context) {
        HotelDetailsDTO hotelDetails = partnerServiceProvider.getPartnerService(hotelConnection.getPartnerId())
                .getHotelDetails(hotelConnection.getHotelCode());
        // we receive hotel notifications only after a hotel has accepted the offer, they can't 'un-accept' it later
        partnerServiceProvider.getPartnerService(hotelConnection.getPartnerId()).verifyHotelDetails(hotelDetails);
        validateLegalName(hotelDetails.getBankAccountDetails().getPersonLegalName());
        validateLegalName(hotelDetails.getBankAccountDetails().getBranchName());
        validateBankAccountDetails(hotelDetails.getBankAccountDetails());

        HotelConnectionUpdate connectionUpdate = createHotelConnectionUpdate(hotelConnection, hotelDetails);

        UpdateResult result = verifyUpdateResult(hotelConnection, connectionUpdate);
        if (result.getType().equals(UpdateResult.ResultType.SUCCESS)) {
            applyHotelConnectionUpdate(hotelConnection, connectionUpdate, context, true);
        } else {
            connectionUpdate = persistHotelConnectionUpdate(connectionUpdate);
            result.setHotelConnectionUpdate(connectionUpdate);
            result.setBoundHotels(findBoundHotels(hotelConnection));
            result.setRelevantHotels(findRelevantHotels(result.getOldLegalDetails(), result.getNewLegalDetails(),
                    result.getBoundHotels(), hotelConnection));
        }
        hotelConnectionRepository.saveAndFlush(hotelConnection);
        return result;
    }

    private List<UpdateResult.HotelData> findBoundHotels(HotelConnection hotelConnection) {
        if (hotelConnection.getLegalDetails() == null) {
            return List.of();
        }
        Set<HotelConnection> foundHotelConnections =
                hotelConnectionRepository.findAllByLegalDetails(hotelConnection.getLegalDetails());
        foundHotelConnections.remove(hotelConnection);
        return foundHotelConnections.stream()
                .map(connection -> new UpdateResult.HotelData(
                        connection,
                        hotelConnectionUpdateRepository
                                .findFirstByHotelConnectionAndState(connection, EHotelConnectionUpdateState.HCU_NEW)
                                .map(HotelConnectionUpdate::getStTicket)
                                .orElse(null))
                ).collect(Collectors.toList());
    }

    private List<UpdateResult.HotelData> findRelevantHotels(LegalDetails oldLegalDetails, LegalDetails newLegalDetails,
                                                            List<UpdateResult.HotelData> boundHotels,
                                                            HotelConnection hotelConnection) {
        Set<String> innList = new HashSet<>();
        if (oldLegalDetails != null) {
            innList.add(oldLegalDetails.getInn());
        }
        if (newLegalDetails != null) { //new LegalDetails should be always not null, check just in case
            innList.add(newLegalDetails.getInn());
        }
        Set<HotelConnection> relevantHotels = hotelConnectionRepository.findAllWithAnyInn(innList);
        relevantHotels.removeAll(boundHotels.stream().map(UpdateResult.HotelData::getHotelConnection).collect(Collectors.toList()));
        relevantHotels.remove(hotelConnection);
        return relevantHotels.stream()
                .map(connection -> new UpdateResult.HotelData(
                        connection,
                        hotelConnectionUpdateRepository
                                .findFirstByHotelConnectionAndState(connection, EHotelConnectionUpdateState.HCU_NEW)
                                .map(HotelConnectionUpdate::getStTicket)
                                .orElse(null))
                ).collect(Collectors.toList());
    }

    private HotelConnectionUpdate persistHotelConnectionUpdate(HotelConnectionUpdate hotelConnectionUpdate) {
        hotelConnectionUpdate = hotelConnectionUpdateRepository.saveAndFlush(hotelConnectionUpdate);
        Workflow connectionUpdateWorkflow = Workflow.createWorkflowForEntity(hotelConnectionUpdate,
                KnownWorkflow.GENERIC_SUPERVISOR.getUuid());
        workflowRepository.saveAndFlush(connectionUpdateWorkflow);

        return hotelConnectionUpdate;
    }

    public ApplyUpdateResult applyHotelConnectionUpdate(HotelConnection hotelConnection,
                                                        HotelConnectionUpdate connectionUpdate,
                                                        MessagingContext<?> context, boolean bankChangeInplaceMode) {
        hotelConnection.setHotelDetailsDTO(connectionUpdate.getHotelDetailsDto());

        // each hotel can have its own accountant email and its update doesn't require a manager's approve
        hotelConnection.setAccountantEmail(connectionUpdate.getAccountantEmail());
        hotelConnection.setReservationPhone(connectionUpdate.getReservationPhone());
        hotelConnection.setAddress(connectionUpdate.getAddress());
        hotelConnection.setCityName(connectionUpdate.getCityName());
        hotelConnection.setTaxType(connectionUpdate.getTaxType());
        hotelConnection.setVatType(connectionUpdate.getVatType());

        hotelConnection.setExternalHotelId(connectionUpdate.getExternalHotelId());

        hotelConnection.setContractPersonName(connectionUpdate.getContractPersonName());
        hotelConnection.setContractPersonPosition(connectionUpdate.getContractPersonPosition());
        hotelConnection.setContractPersonPhone(connectionUpdate.getContractPersonPhone());
        hotelConnection.setContractPersonEmail(connectionUpdate.getContractPersonEmail());
        ApplyUpdateResult applyUpdateResult = actualizeLegalDetails(hotelConnection, connectionUpdate, context,
                bankChangeInplaceMode);
        starTrekService.updateHotelConnectionTicket(hotelConnection);
        return applyUpdateResult;
    }

    private HotelConnectionUpdate createHotelConnectionUpdate(HotelConnection hotelConnection,
                                                              HotelDetailsDTO hotelDetails) {
        BankAccountDetailsDTO bankAccountDetails = hotelDetails.getBankAccountDetails();
        String unifiedPostAddress =
                addressUnificationService.unifyAddressSync(hotelDetails.getFactAddress().getFullAddress());
        String unifiedLegalAddress =
                addressUnificationService.unifyAddressSync(bankAccountDetails.getAddressDetails().getFullAddress());
        HotelContactDTO contactInfo = getContactInfo(hotelDetails.getContactInfo(), ContactType.CONTRACT);
        return HotelConnectionUpdate.builder()
                .id(UUID.randomUUID())
                .hotelConnection(hotelConnection)
                .hotelDetailsDto(hotelDetails)
                .hotelCode(hotelDetails.getHotelCode())
                .externalHotelId(hotelDetails.getExternalHotelId())
                .partnerId(hotelConnection.getPartnerId())
                .originalHotelConnectionState(hotelConnection.getState())
                .accountantEmail(getAccountantEmail(hotelDetails.getContactInfo()))
                .reservationPhone(getReservationPhone(hotelDetails.getContactInfo()))
                .address(unifiedPostAddress != null ? unifiedPostAddress :
                        hotelDetails.getFactAddress().getFullAddress())
                .cityName(hotelDetails.getFactAddress().getCityName())
                .contractPersonName(contactInfo.getName())
                .contractPersonPosition(contactInfo.getPosition())
                .contractPersonPhone(contactInfo.getPhone())
                .contractPersonEmail(contactInfo.getEmail())
                .taxType(bankAccountDetails.getTaxType())
                .vatType(mapTaxTypeToEVat(bankAccountDetails.getTaxType()))
                .inn(bankAccountDetails.getInn())
                .kpp(StringUtils.trimToNull(bankAccountDetails.getKpp()))
                .bic(bankAccountDetails.getBic())
                .correspondingAccount(bankAccountDetails.getCorrespondingAccount())
                .paymentAccount(bankAccountDetails.getCurrentAccount())
                .bankName(bankAccountDetails.getBankName())
                .legalName(bankAccountDetails.getPersonLegalName())
                .fullLegalName(StringUtils.isNotBlank(bankAccountDetails.getBranchName()) ?
                        bankAccountDetails.getBranchName() : bankAccountDetails.getPersonLegalName())
                .legalPostCode(bankAccountDetails.getAddressDetails().getPostalCode())
                .legalAddress(unifiedLegalAddress != null ? unifiedLegalAddress :
                        bankAccountDetails.getAddressDetails().getFullAddress())
                .originalLegalAddress(bankAccountDetails.getAddressDetails().getFullAddress())
                .legalAddressUnified(unifiedLegalAddress != null)
                .postCode(hotelDetails.getFactAddress().getPostalCode())
                .postAddress(unifiedPostAddress != null ? unifiedPostAddress :
                        hotelDetails.getFactAddress().getFullAddress())
                .offerAccepted(true)
                .legalPhone(bankAccountDetails.getPhone())
                .state(EHotelConnectionUpdateState.HCU_NEW)
                .build();
    }

    private void validateLegalName(String legalName) {
        if (legalName != null) {
            if (!LEGAL_NAME_PATTERN.matcher(legalName).matches()) {
                throw new IllegalArgumentException("Illegal name: " + legalName);
            }
        }
    }

    private UpdateResult verifyUpdateResult(HotelConnection hotelConnection, HotelConnectionUpdate connectionUpdate) {
        String hotelCode = connectionUpdate.getHotelCode();
        String inn = connectionUpdate.getInn();
        LegalDetails originalLegalDetails = hotelConnection.getLegalDetails();
        LegalDetails existingLegalDetails = findExistingLegalDetails(connectionUpdate);
        LegalDetails newLegalDetails = convertToLegalDetails(connectionUpdate, hotelConnection.getPartnerId());
        if (hotelConnectionProperties.isAllowCheckRuBankAccountCall() && !billingService.checkRuBankAccount(newLegalDetails.getBic(), newLegalDetails.getPaymentAccount())) {
            return UpdateResult.validationRequired(
                    hotelConnection,
                    originalLegalDetails,
                    existingLegalDetails,
                    newLegalDetails,
                    UpdateResult.ChangeRequisitesType.WRONG_BIK_AND_ACCOUNT_PAIR
            );
        }
        if (hotelConnection.getLegalDetails() == null) {
            //case when hotel is new
            boolean paperAgreementExists = agreementService.existsPaperAgreementForInn(connectionUpdate.getInn());
            if (existingLegalDetails == null) {
                if (paperAgreementExists) {
                    log.info("New hotel, existing LegalDetails not found, but paper agreement exists. Hotel {}; inn " +
                            "{}", hotelCode, inn);
                    return UpdateResult.validationRequired(
                            hotelConnection,
                            null,
                            null,
                            newLegalDetails,
                            UpdateResult.ChangeRequisitesType.PAPER_AGREEMENT_CHANGE
                    );
                } else {
                    //We can simply create new legal details for new hotel
                    log.info("New hotel, creating new LegalDetails. hotel_code: {}, inn: {}", hotelCode, inn);
                    return UpdateResult.successResult();
                }
            } else {
                if (paperAgreementExists) {
                    log.info("Existing legal details are paper legal details. Need to manually verify hotel " +
                            "connection. HotelCode: {}, inn: {}", hotelCode, inn);
                    return UpdateResult.validationRequired(
                            hotelConnection,
                            null,
                            existingLegalDetails,
                            newLegalDetails,
                            UpdateResult.ChangeRequisitesType.PAPER_AGREEMENT_CHANGE
                    );
                } else {
                    if (LegalDetailsService.hasNoMajorChanges(existingLegalDetails, connectionUpdate)) {
                        log.info("Legal Details have no major changes, scheduling change of non-major fields. " +
                                "hotel_code: {}, inn: {}", hotelCode, inn);
                        return UpdateResult.successResult();
                    } else {
                        //new hotel, which is bound to existing legal details, which have major changes
                        log.info("Legal Details have major changes, changes are scheduled, but suspended with st " +
                                "ticket. hotel_code {}, inn: {}", hotelCode, inn);
                        return UpdateResult.validationRequired(
                                hotelConnection,
                                null,
                                existingLegalDetails,
                                newLegalDetails,
                                UpdateResult.ChangeRequisitesType.OTHER_CHANGE);
                    }
                }
            }
        } else if (sameLegalDetails(hotelConnection.getLegalDetails(), connectionUpdate)) {
            //case of already published hotel, legal details are the same
            if (LegalDetailsService.hasChanges(hotelConnection.getLegalDetails(), connectionUpdate)
                    || !Objects.equals(hotelConnection.getAccountantEmail(), connectionUpdate.getAccountantEmail())) {
                boolean paperAgreementExists =
                        agreementService.existsPaperAgreementForInn(hotelConnection.getLegalDetails().getInn());
                if (paperAgreementExists) {
                    log.info("Same legal details: hotel is on paper agreement, creating st ticket. HotelCode {}, inn " +
                            "{}", hotelCode, inn);
                    return UpdateResult.validationRequired(
                            hotelConnection,
                            originalLegalDetails,
                            null,
                            newLegalDetails,
                            UpdateResult.ChangeRequisitesType.PAPER_AGREEMENT_CHANGE);
                } else {
                    if (LegalDetailsService.hasNoMajorChanges(hotelConnection.getLegalDetails(), connectionUpdate)) {
                        log.info("Same legal details: Legal Details have no major changes, scheduling change of " +
                                "non-major fields. hotel_code: {}, inn: {}", hotelCode, inn);
                        return UpdateResult.successResult();
                    } else {
                        log.info("Same legal details: Legal Details have major changes, changes are scheduled, but " +
                                "suspended with st ticket. hotel_code {}, inn: {}", hotelCode, inn);
                        return UpdateResult.validationRequired(
                                hotelConnection,
                                originalLegalDetails,
                                null,
                                newLegalDetails,
                                UpdateResult.ChangeRequisitesType.OTHER_CHANGE);
                    }
                }
            } else {
                log.info("Same legal details: no changes found. hotel_code: {}, inn: {}", hotelCode, inn);
                return UpdateResult.successResult();
            }
        } else {
            //case of already published hotel, legal details became different
            log.info("Hotel has completely changed its legal details");
            UpdateResult.ChangeRequisitesType changeType;
            if (agreementService.existsPaperAgreementForInn(connectionUpdate.getInn())) {
                changeType = UpdateResult.ChangeRequisitesType.PAPER_AGREEMENT_CHANGE;
            } else {
                if (existingLegalDetails == null) {
                    if (hotelConnection.getLegalDetails().getInn().equals(connectionUpdate.getInn())) {
                        changeType = UpdateResult.ChangeRequisitesType.BANK_ACCOUNT_CHANGE;
                    } else {
                        changeType = UpdateResult.ChangeRequisitesType.INN_CHANGE;
                    }
                } else {
                    changeType = UpdateResult.ChangeRequisitesType.OTHER_CHANGE;
                }
            }
            return UpdateResult.validationRequired(
                    hotelConnection,
                    originalLegalDetails,
                    existingLegalDetails,
                    newLegalDetails,
                    changeType
            );
        }
    }

    private ApplyUpdateResult actualizeLegalDetails(HotelConnection hotelConnection,
                                                    HotelConnectionUpdate connectionUpdate,
                                                    MessagingContext<?> context, boolean bankChangeInplaceMode) {
        String hotelCode = connectionUpdate.getHotelCode();
        String inn = connectionUpdate.getInn();
        LegalDetails newLegalDetails = convertToLegalDetails(connectionUpdate, hotelConnection.getPartnerId());
        if (hotelConnection.isPaperAgreement()) {
            throw new RuntimeException("Can't actualize legal details for connection with paper agreement");
        }
        if (hotelConnection.getLegalDetails() == null) {
            //case when hotel is new
            log.info("Creating or picking existing Legal Details for new hotel");
            LegalDetails existingLegalDetails = findExistingLegalDetails(connectionUpdate);
            if (existingLegalDetails == null) {
                log.info("Creating a new legal details for new hotel {}; inn {}", hotelCode, inn);
                hotelConnection.setLegalDetails(persistLegalDetails(newLegalDetails, context));
                return new ApplyUpdateResult(true);
            } else {
                Preconditions.checkState(bankChangeInplaceMode,
                        "Can't create new legal details because same legal details already exist (%s)", existingLegalDetails.getId());
                log.info("Binding new hotel {} to existing legal details; inn {}", hotelCode, inn);
                if (existingLegalDetails.isManagedByAdministrator()) {
                    log.info("Existing legal details: at least accountant emails should be updated. hotel_code: {}, inn: " +
                            "{}", hotelCode, inn);
                    hotelConnection.setLegalDetails(existingLegalDetails);
                    LegalDetailsUpdate legalDetailsUpdate = convertToLegalDetailsUpdate(hotelConnection, connectionUpdate);
                    scheduleLegalDetailsUpdate(hotelConnection.getLegalDetails(), legalDetailsUpdate, context);
                    return new ApplyUpdateResult(true);
                } else {
                    Preconditions.checkState(!LegalDetailsService.hasChanges(existingLegalDetails, connectionUpdate),
                            "Unable to apply update to hotel %s, because existing legal details found (%s) and they " +
                                    "are not managed by administrator", hotelCode, existingLegalDetails.getId());
                    log.info("Binding new hotel {} to existing legal details which are not managed by administrator; inn {}", hotelCode, inn);
                    hotelConnection.setLegalDetails(existingLegalDetails);
                    return new ApplyUpdateResult(false);
                }
            }
        } else if (sameLegalDetails(hotelConnection.getLegalDetails(), connectionUpdate)) {
            //case of already published hotel, legal details are the same
            log.info("Legal details are the same, looking for changes");
            if (LegalDetailsService.hasChanges(hotelConnection.getLegalDetails(), connectionUpdate)) {
                Preconditions.checkState(bankChangeInplaceMode,
                        "Can't create new legal details because same legal details already exist (%s)", hotelConnection.getLegalDetails().getId());
                Preconditions.checkState(hotelConnection.getLegalDetails().isManagedByAdministrator(),
                        "Can't update legal details which are not managed by administrator (%s)", hotelConnection.getLegalDetails().getId());

                log.info("Same legal details: some changes are found. hotel_code: {}, inn: {}", hotelCode, inn);
                LegalDetailsUpdate legalDetailsUpdate = convertToLegalDetailsUpdate(hotelConnection, connectionUpdate);
                scheduleLegalDetailsUpdate(hotelConnection.getLegalDetails(), legalDetailsUpdate, context);
                return new ApplyUpdateResult(true);
            } else {
                return new ApplyUpdateResult(false);
            }
        } else {
            //case of already published hotel, legal details became different
            log.info("Hotel has completely changed its legal details");
            LegalDetails existingLegalDetails = findExistingLegalDetails(connectionUpdate);
            if (existingLegalDetails == null) {
                if (hotelConnection.getLegalDetails().getInn().equals(connectionUpdate.getInn()) &&
                        bankChangeInplaceMode && hotelConnection.getLegalDetails().isManagedByAdministrator()) {
                    log.info("Different bank account: updating existing entities in Balance. hotel_code: {}, inn: {}"
                            , hotelCode, inn);
                    LegalDetailsUpdate legalDetailsUpdate = convertToLegalDetailsUpdate(hotelConnection,
                            connectionUpdate);
                    scheduleLegalDetailsUpdate(hotelConnection.getLegalDetails(), legalDetailsUpdate, context);
                    return new ApplyUpdateResult(true);
                } else {
                    log.info("Different Legal Details: Creating a new legal details for new hotel {}; inn {}",
                            hotelCode, inn);
                    hotelConnection.setLegalDetails(persistLegalDetails(newLegalDetails, context));
                    return new ApplyUpdateResult(true);
                }
            } else {
                log.info("Existing hotel has different legal details which are found");
                if (existingLegalDetails.isManagedByAdministrator()) {
                    log.info("At least accountant emails should be updated. hotel_code: {}, inn: {}", hotelCode, inn);
                    hotelConnection.setLegalDetails(existingLegalDetails);
                    LegalDetailsUpdate legalDetailsUpdate = convertToLegalDetailsUpdate(hotelConnection, connectionUpdate);
                    scheduleLegalDetailsUpdate(hotelConnection.getLegalDetails(), legalDetailsUpdate, context);
                    return new ApplyUpdateResult(true);
                } else {
                    Preconditions.checkState(!LegalDetailsService.hasChanges(existingLegalDetails, connectionUpdate),
                            "Unable to apply update to hotel %s, because existing legal details found (%s) and they " +
                                    "are not managed by administrator", hotelCode, existingLegalDetails.getId());
                    log.info("Different Legal Details: binding hotel {} to existing legal details which are not managed by administrator; inn {}", hotelCode, inn);
                    hotelConnection.setLegalDetails(existingLegalDetails);
                    return new ApplyUpdateResult(false);
                }
            }
        }
    }

    private EVat mapTaxTypeToEVat(HotelTaxType taxType) {
        switch (taxType) {
            case COMMON:
            case UNIFORM_AGRICULTURAL:
                return EVat.VAT_20_120;
            case COMMON_10_VAT:
                return EVat.VAT_10_110;
            case COMMON_0_VAT:
                return EVat.VAT_0;
            default:
                return EVat.VAT_NONE;
        }
    }

    private LegalDetails convertToLegalDetails(HotelConnectionUpdate connectionUpdate, EPartnerId partnerId) {
        return LegalDetails.builder()
                .id(UUID.randomUUID())
                .partnerId(partnerId)
                .state(ELegalDetailsState.DS_NEW)
                .inn(connectionUpdate.getInn())
                .kpp(StringUtils.trimToNull(connectionUpdate.getKpp()))
                .bic(connectionUpdate.getBic())
                .bankName(connectionUpdate.getBankName())
                .correspondingAccount(connectionUpdate.getCorrespondingAccount())
                .paymentAccount(connectionUpdate.getPaymentAccount())
                .legalName(connectionUpdate.getLegalName())
                .fullLegalName(connectionUpdate.getFullLegalName())
                .legalPostCode(connectionUpdate.getLegalPostCode())
                .legalAddress(connectionUpdate.getLegalAddress())
                .legalAddressUnified(connectionUpdate.isLegalAddressUnified())
                .postCode(connectionUpdate.getPostCode())
                .postAddress(connectionUpdate.getPostAddress())
                .phone(connectionUpdate.getLegalPhone())
                .hotelConnectionUpdateId(connectionUpdate.getId())
                .offerAccepted(connectionUpdate.isOfferAccepted())
                .managedByAdministrator(true)
                .build();
    }

    private void validateBankAccountDetails(BankAccountDetailsDTO bankAccountDetails) {
        Preconditions.checkNotNull(bankAccountDetails);
        Preconditions.checkNotNull(bankAccountDetails.getInn());
        Preconditions.checkNotNull(bankAccountDetails.getBic());
        Preconditions.checkNotNull(bankAccountDetails.getCurrentAccount());
    }

    private LegalDetails findExistingLegalDetails(HotelConnectionUpdate connectionUpdate) {
        return legalDetailsRepository.findByInnAndKppAndBicAndPaymentAccount(
                connectionUpdate.getInn(),
                StringUtils.trimToNull(connectionUpdate.getKpp()),
                connectionUpdate.getBic(),
                connectionUpdate.getPaymentAccount());
    }

    private LegalDetails persistLegalDetails(LegalDetails legalDetails, MessagingContext<?> context) {
        try {
            legalDetails = legalDetailsRepository.saveAndFlush(legalDetails);
        } catch (DataIntegrityViolationException | PessimisticLockingFailureException e) {
            // DataIntegrityViolationException: Unique index or primary key violation
            // PessimisticLockingFailureException: Concurrent update in table "LEGAL_DETAILS":
            //          another transaction has updated or deleted the same row [90131-199]
            throw new RetryableServiceException("Someone is trying to insert the same legal details", e);
        }

        Workflow legalDetailsWorkflow = Workflow.createWorkflowForEntity(legalDetails,
                KnownWorkflow.GENERIC_SUPERVISOR.getUuid());
        workflowRepository.saveAndFlush(legalDetailsWorkflow);

        context.scheduleExternalEvent(legalDetails.getWorkflow().getId(),
                TRegisterLegalDetails.newBuilder().build());

        return legalDetails;
    }

    private LegalDetailsUpdate convertToLegalDetailsUpdate(HotelConnection hotelConnection,
                                                           HotelConnectionUpdate connectionUpdate) {
        return LegalDetailsUpdate.builder()
                .state(LegalDetailsUpdateState.PENDING)
                .hotelConnection(hotelConnection)
                .legalDetails(hotelConnection.getLegalDetails())
                .hotelConnectionUpdateId(connectionUpdate.getId())
                .inn(connectionUpdate.getInn())
                .kpp(connectionUpdate.getKpp())
                .bic(connectionUpdate.getBic())
                .bankName(connectionUpdate.getBankName())
                .correspondingAccount(connectionUpdate.getCorrespondingAccount())
                .paymentAccount(connectionUpdate.getPaymentAccount())
                .legalName(connectionUpdate.getLegalName())
                .fullLegalName(connectionUpdate.getFullLegalName())
                .legalPostCode(connectionUpdate.getLegalPostCode())
                .legalAddress(connectionUpdate.getLegalAddress())
                .legalAddressUnified(connectionUpdate.isLegalAddressUnified())
                .originalLegalAddress(connectionUpdate.getOriginalLegalAddress())
                .postCode(connectionUpdate.getPostCode())
                .postAddress(connectionUpdate.getPostAddress())
                .phone(connectionUpdate.getLegalPhone())
                .build();
    }

    private void scheduleLegalDetailsUpdate(LegalDetails legalDetails, LegalDetailsUpdate update,
                                            MessagingContext<?> context) {
        Preconditions.checkState(legalDetails.isManagedByAdministrator(), "Can't update legal details which are not managed by hotel administrator (%s)", legalDetails.getId());
        legalDetailsUpdateRepository.saveAndFlush(update);
        context.scheduleExternalEvent(legalDetails.getWorkflow().getId(), TUpdateLegalDetails.newBuilder().build());
    }

    private boolean sameLegalDetails(LegalDetails legalDetails, HotelConnectionUpdate connectionUpdate) {
        if (legalDetails == null) {
            return false;
        }
        if (!legalDetails.getInn().equals(connectionUpdate.getInn())) {
            return false;
        }
        if (!Objects.equals(legalDetails.getKpp(), connectionUpdate.getKpp())) {
            return false;
        }
        if (!legalDetails.getBic().equals(connectionUpdate.getBic())) {
            return false;
        }
        if (!legalDetails.getPaymentAccount().equals(connectionUpdate.getPaymentAccount())) {
            return false;
        }
        return true;
    }

    private String getReservationPhone(List<HotelContactDTO> contacts) {
        HotelContactDTO reservationContact = getContactInfo(contacts, ContactType.RESERVATION);
        Preconditions.checkNotNull(reservationContact.getPhone(), "No reservation phone found: %s", reservationContact);
        return reservationContact.getPhone();
    }

    private String getAccountantEmail(List<HotelContactDTO> contacts) {
        HotelContactDTO accountingContact = getContactInfo(contacts, ContactType.ACCOUNTANT);
        Preconditions.checkNotNull(accountingContact.getEmail(), "No accountant email: %s", accountingContact);
        return accountingContact.getEmail();
    }

    private HotelContactDTO getContactInfo(List<HotelContactDTO> contacts, ContactType contactType) {
        Preconditions.checkNotNull(contacts, "Contacts should not be null");
        return contacts.stream()
                .filter(ci -> ci.getContactType() == contactType)
                .findFirst().orElseThrow(() -> new RuntimeException("No contact for " + contactType + " found"));
    }
}
