package ru.yandex.travel.hotels.administrator.service;

import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.administrator.repository.HotelConnectionUpdateRepository;
import ru.yandex.travel.hotels.administrator.workflow.proto.EHotelConnectionUpdateState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TApplyConnectionUpdate;
import ru.yandex.travel.hotels.administrator.workflow.proto.TRejectConnectionUpdate;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Slf4j
@Service
@AllArgsConstructor
public class HotelConnectionUpdateService {
    private final StarTrekService starTrekService;
    private final HotelConnectionUpdateRepository hotelConnectionUpdateRepository;
    private final WorkflowMessageSender workflowMessageSender;

    public void processStTicket(String ticketKey) {
        var hotelConnectionUpdates = hotelConnectionUpdateRepository.findAllByStTicket(ticketKey);
        Preconditions.checkState(hotelConnectionUpdates.size() == 1, String.format("Expected single connection update" +
                " with ticket %s", ticketKey));
        var hotelConnectionUpdate = hotelConnectionUpdates.get(0);

        try {
            if (hotelConnectionUpdate.getState() != EHotelConnectionUpdateState.HCU_NEW) {
                log.info("Hotel connection update {} is already processed, skipping it", hotelConnectionUpdate.getId());
                starTrekService.commentCantProcessHotelConnectionTwice(ticketKey);
            } else {
                log.info("Processing hotel connection update {}", hotelConnectionUpdate.getId());
                var action = starTrekService.getHotelConnectionUpdateAutoProcessingAction(ticketKey);

                if (starTrekService.isWrongBikAndAccountTicket(ticketKey) && action != StarTrekService.AutoProcessingActionEnum.REJECT) {
                    starTrekService.commentCantAcceptWrongBikAndAccount(ticketKey);
                    starTrekService.removeHotelConnectionUpdateActionTag(ticketKey, action);
                    return;
                }

                if (action == StarTrekService.AutoProcessingActionEnum.ACCEPT) {
                    workflowMessageSender.scheduleEvent(hotelConnectionUpdate.getWorkflow().getId(),
                            TApplyConnectionUpdate.newBuilder().setBankChangeInplaceMode(true).build());
                } else if (action == StarTrekService.AutoProcessingActionEnum.ACCEPT_NEW_CLIENT) {
                    workflowMessageSender.scheduleEvent(hotelConnectionUpdate.getWorkflow().getId(),
                            TApplyConnectionUpdate.newBuilder().setBankChangeInplaceMode(false).build());
                } else if (action == StarTrekService.AutoProcessingActionEnum.REJECT) {
                    workflowMessageSender.scheduleEvent(hotelConnectionUpdate.getWorkflow().getId(),
                            TRejectConnectionUpdate.newBuilder().build());
                } else {
                    throw new IllegalStateException(String.format("Unknown action: %s", action.toString()));
                }
            }
        } finally {
            // for safety: to avoid scheduling same event many times in case of some error
            starTrekService.removeHotelConnectionUpdateAutoProcessingTag(ticketKey);
        }
    }
}
