package ru.yandex.travel.hotels.administrator.service;

import java.util.List;
import java.util.Objects;
import java.util.Set;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.administrator.entity.HotelConnectionUpdate;
import ru.yandex.travel.hotels.administrator.entity.LegalDetails;
import ru.yandex.travel.hotels.administrator.entity.LegalDetailsUpdate;
import ru.yandex.travel.hotels.administrator.entity.LegalDetailsUpdateState;
import ru.yandex.travel.hotels.administrator.repository.HotelConnectionUpdateRepository;
import ru.yandex.travel.hotels.administrator.repository.LegalDetailsUpdateRepository;
import ru.yandex.travel.hotels.administrator.workflow.proto.ELegalDetailsState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TNotifyLegalDetailsPublished;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.StateContext;

@Service
@RequiredArgsConstructor
@Slf4j
public class LegalDetailsService {

    private final LegalDetailsUpdateRepository legalDetailsUpdateRepository;

    private final HotelConnectionUpdateRepository hotelConnectionUpdateRepository;

    @TransactionMandatory
    public boolean hasReadyUpdates(LegalDetails legalDetails) {
        return hasPendingUpdates(legalDetails) && !hasInProgressUpdates(legalDetails);
    }

    private boolean hasPendingUpdates(LegalDetails legalDetails) {
        return legalDetailsUpdateRepository.countByLegalDetailsAndStateIn(
                legalDetails, Set.of(LegalDetailsUpdateState.PENDING)) > 0;
    }

    private boolean hasInProgressUpdates(LegalDetails legalDetails) {
        return legalDetailsUpdateRepository.countByLegalDetailsAndStateIn(
                legalDetails, Set.of(LegalDetailsUpdateState.SYNCHRONIZING_WITH_BILLING)) > 0;
    }

    /**
     * @return true if the state of the legal details has changed and synchronization with billing is required
     */
    @TransactionMandatory
    public boolean applyReadyUpdates(LegalDetails legalDetails) {
        if (hasInProgressUpdates(legalDetails)) {
            log.info("Some legal details updates are still in progress, will wait for their completion; inn {}",
                    legalDetails.getInn());
            return false;
        }
        List<LegalDetailsUpdate> updates = legalDetailsUpdateRepository.findByLegalDetailsAndStateInOrderByCreatedAt(
                legalDetails, Set.of(LegalDetailsUpdateState.PENDING));
        if (updates.isEmpty()) {
            log.info("No pending updates for these legal details; inn {}", legalDetails.getInn());
            return false;
        }
        for (LegalDetailsUpdate update : updates) {
            log.info("Applying legal details update; inn {}, update id {}", legalDetails.getInn(), update.getId());
            applyUpdate(legalDetails, update);
            // the changes should be synchronized with Billing first, then they become APPLIED
            update.setState(LegalDetailsUpdateState.SYNCHRONIZING_WITH_BILLING);
        }
        return true;
    }

    @TransactionMandatory
    public void completeBillingSynchronization(LegalDetails legalDetails,
                                               StateContext<ELegalDetailsState, LegalDetails> context) {
        List<LegalDetailsUpdate> updates = legalDetailsUpdateRepository.findByLegalDetailsAndStateInOrderByCreatedAt(
                legalDetails, Set.of(LegalDetailsUpdateState.SYNCHRONIZING_WITH_BILLING));
        log.info("Marking {} updates as applied", updates.size());
        for (LegalDetailsUpdate update : updates) {
            log.info("Update {} has been synchronized with Billing", update.getId());
            update.setState(LegalDetailsUpdateState.APPLIED);
            if (update.getHotelConnectionUpdateId() != null) {
                hotelConnectionUpdateRepository.findById(update.getHotelConnectionUpdateId())
                        .ifPresent(connectionUpdate -> context.scheduleExternalEvent(connectionUpdate.getWorkflow().getId(), TNotifyLegalDetailsPublished.newBuilder().build()));
            }
        }
    }

    static boolean hasChanges(LegalDetails legalDetails, HotelConnectionUpdate connectionUpdate) {
        return !Objects.equals(legalDetails.getInn(), connectionUpdate.getInn()) ||
                !Objects.equals(legalDetails.getKpp(), connectionUpdate.getKpp()) ||
                !Objects.equals(legalDetails.getBic(), connectionUpdate.getBic()) ||
                !Objects.equals(legalDetails.getCorrespondingAccount(), connectionUpdate.getCorrespondingAccount()) ||
                !Objects.equals(legalDetails.getPaymentAccount(), connectionUpdate.getPaymentAccount()) ||
                !Objects.equals(legalDetails.getLegalName(), connectionUpdate.getLegalName()) ||
                !Objects.equals(legalDetails.getFullLegalName(), connectionUpdate.getFullLegalName()) ||
                !Objects.equals(legalDetails.getLegalPostCode(), connectionUpdate.getLegalPostCode()) ||
                !Objects.equals(legalDetails.getLegalAddress(), connectionUpdate.getLegalAddress()) ||
                !Objects.equals(legalDetails.isLegalAddressUnified(), connectionUpdate.isLegalAddressUnified()) ||
//                We do not update fact address: https://st.yandex-team.ru/TRAVELBACK-509
//                !Objects.equals(legalDetails.getPostCode(), connectionUpdate.getPostCode()) ||
//                !Objects.equals(legalDetails.getPostAddress(), connectionUpdate.getPostAddress()) ||
                !Objects.equals(legalDetails.getPhone(), connectionUpdate.getLegalPhone());
    }

    static boolean hasNoMajorChanges(LegalDetails legalDetails, HotelConnectionUpdate connectionUpdate) {
        return Objects.equals(legalDetails.getInn(), connectionUpdate.getInn()) &&
                Objects.equals(legalDetails.getKpp(), connectionUpdate.getKpp()) &&
                Objects.equals(legalDetails.getBic(), connectionUpdate.getBic()) &&
                Objects.equals(legalDetails.getPaymentAccount(), connectionUpdate.getPaymentAccount()) &&
                Objects.equals(legalDetails.getLegalName(), connectionUpdate.getLegalName()) &&
                Objects.equals(legalDetails.getFullLegalName(), connectionUpdate.getFullLegalName());
//                We do not update fact address: https://st.yandex-team.ru/TRAVELBACK-509
//                Objects.equals(legalDetails.getPostCode(), connectionUpdate.getPostCode()) &&
//                Objects.equals(legalDetails.getPostAddress(), connectionUpdate.getPostAddress());
    }

    private boolean hasSamePerson(LegalDetails legalDetails, LegalDetailsUpdate legalDetailsUpdate) {
        return Objects.equals(legalDetails.getInn(), legalDetailsUpdate.getInn()) ||
                Objects.equals(legalDetails.getKpp(), legalDetailsUpdate.getKpp()) ||
                Objects.equals(legalDetails.getBic(), legalDetailsUpdate.getBic()) ||
                Objects.equals(legalDetails.getPaymentAccount(), legalDetailsUpdate.getPaymentAccount());
    }

    private void applyUpdate(LegalDetails legalDetails, LegalDetailsUpdate legalDetailsUpdate) {
        Preconditions.checkArgument(hasSamePerson(legalDetails, legalDetailsUpdate),
                "Switching legal person is unsupported, use another legal detail's workflow; update id %s",
                legalDetailsUpdate.getId());
        // inn, kpp, bic & account haven't changed
        if (!legalDetails.getInn().equals(legalDetailsUpdate.getInn())) {
            throw new IllegalStateException("Illegal update of legal details. Attempt to change inn from "
                    + legalDetails.getInn() + " to " + legalDetailsUpdate.getInn());
        }
        legalDetails.setKpp(legalDetailsUpdate.getKpp());
        legalDetails.setBic(legalDetailsUpdate.getBic());
        legalDetails.setPaymentAccount(legalDetailsUpdate.getPaymentAccount());
        legalDetails.setLegalName(legalDetailsUpdate.getLegalName());
        legalDetails.setFullLegalName(legalDetailsUpdate.getFullLegalName());
        legalDetails.setLegalPostCode(legalDetailsUpdate.getLegalPostCode());
        legalDetails.setLegalAddress(legalDetailsUpdate.getLegalAddress());
        legalDetails.setLegalAddressUnified(legalDetailsUpdate.isLegalAddressUnified());
//        We do not update fact address: https://st.yandex-team.ru/TRAVELBACK-509
//        legalDetails.setPostCode(legalDetailsUpdate.getPostCode());
//        legalDetails.setPostAddress(legalDetailsUpdate.getPostAddress());
        legalDetails.setPhone(legalDetailsUpdate.getPhone());
        legalDetails.setBankName(legalDetailsUpdate.getBankName());
        legalDetails.setCorrespondingAccount(legalDetailsUpdate.getCorrespondingAccount());
        legalDetails.setFullLegalName(legalDetailsUpdate.getFullLegalName());
    }
}
