package ru.yandex.travel.hotels.administrator.service;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.bolts.collection.impl.ArrayListF;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.StartrekClientBuilder;
import ru.yandex.startrek.client.error.EntityNotFoundException;
import ru.yandex.startrek.client.error.StartrekClientException;
import ru.yandex.startrek.client.error.StartrekInternalClientError;
import ru.yandex.startrek.client.model.Comment;
import ru.yandex.startrek.client.model.CommentCreate;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.IssueCreate;
import ru.yandex.startrek.client.model.IssueRef;
import ru.yandex.startrek.client.model.IssueUpdate;
import ru.yandex.startrek.client.model.LocalLink;
import ru.yandex.startrek.client.model.Relationship;
import ru.yandex.startrek.client.model.Transition;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.administrator.cache.proto.TAltayPublishingRecord;
import ru.yandex.travel.hotels.administrator.cache.proto.TAltaySignalRecord;
import ru.yandex.travel.hotels.administrator.configuration.StarTrekConfigurationProperties;
import ru.yandex.travel.hotels.administrator.entity.HotelConnection;
import ru.yandex.travel.hotels.administrator.entity.LegalDetails;
import ru.yandex.travel.hotels.administrator.workflow.supervisor.SupervisorUtils;
import ru.yandex.travel.hotels.proto.EPartnerId;

@Service
@EnableConfigurationProperties(StarTrekConfigurationProperties.class)
@Slf4j
public class StarTrekService {

    private static final Set<String> CLOSED_TICKET_STATUSES = Set.of("resolved", "closed");

    private final Session startrekClient;

    private final TemplateService templateService;

    private final StarTrekConfigurationProperties properties;

    public StarTrekService(TemplateService templateService, StarTrekConfigurationProperties properties) {
        this.templateService = templateService;
        this.properties = properties;
        this.startrekClient = StartrekClientBuilder.newBuilder()
                .uri(properties.getUrl())
                .maxConnections(properties.getMaxConnections())
                .connectionTimeout(properties.getConnectionTimeout().toMillis(), TimeUnit.MILLISECONDS)
                .socketTimeout(properties.getSocketTimeout().toMillis(), TimeUnit.MILLISECONDS)
                .build(properties.getOauthToken());
    }

    private Issue getTicket(String ticketKey) {
        try {
            return startrekClient.issues().get(ticketKey);
        } catch (StartrekClientException e) {
            if (e.getErrors().getStatusCode() >= 500) {
                throw new StartrekRetryableException(e);
            } else {
                throw e;
            }
        } catch (StartrekInternalClientError e) {
            throw new StartrekRetryableException(e);
        }
    }

    public boolean isTicketClosed(String ticketKey) {
        Issue issue = getTicket(ticketKey);
        return issue.getStatus() != null && CLOSED_TICKET_STATUSES.contains(issue.getStatus().getKey());
    }

    public void commentRegisteredInBalance(String ticketKey, Long balanceClientId, Long balanceContractId) {
        addComment(ticketKey, String.format(properties.getCommentRegisteredInBalance(), balanceClientId,
                balanceContractId));
    }

    private void addComment(String ticketKey, String comment) {
        try {
            startrekClient.comments().create(ticketKey,
                    CommentCreate
                            .builder()
                            .comment(comment)
                            .build());
        } catch (StartrekClientException e) {
            if (e.getErrors().getStatusCode() >= 500) {
                throw new StartrekRetryableException(e);
            } else {
                throw e;
            }
        } catch (StartrekInternalClientError e) {
            throw new StartrekRetryableException(e);
        }
    }

    private List<String> getTags(String ticketKey) {
        try {
            return startrekClient.issues().get(ticketKey).getTags();
        } catch (StartrekClientException e) {
            if (e.getErrors().getStatusCode() >= 500) {
                throw new StartrekRetryableException(e);
            } else {
                throw e;
            }
        } catch (StartrekInternalClientError e) {
            throw new StartrekRetryableException(e);
        }
    }

    public String getLastComment(String ticketKey) {
        Issue ticket = getTicket(ticketKey);
        List<Comment> comments = ticket.getComments().collect(Collectors.toList());
        if (comments.isEmpty()) {
            return null;
        }
        return comments.get(comments.size() - 1).getText().get();
    }

    public String createHotelConnectionTicket(HotelConnection hotelConnection) {
        return createTicket(
                String.format(properties.getHotelConnectionTicketTitle(), hotelConnection.getHotelName(),
                        hotelConnection.getHotelCode(), hotelConnection.getPartnerId().toString(),
                        hotelConnection.getPermalink()),
                templateService.getHotelConnectionTicketDescription(hotelConnection),
                properties.getQueueName(),
                "epic",
                List.of(hotelConnection.getPartnerId() + "_hotels"),
                null);
    }

    public void updateHotelConnectionTicket(HotelConnection hotelConnection) {
        if (hotelConnection.getStTicket() != null) {
            updateTicket(
                    hotelConnection.getStTicket(),
                    String.format(properties.getHotelConnectionTicketTitle(), hotelConnection.getHotelName(),
                            hotelConnection.getHotelCode(), hotelConnection.getPartnerId().toString(),
                            hotelConnection.getPermalink()),
                    templateService.getHotelConnectionTicketDescription(hotelConnection));
        } else {
            log.warn("Startrek ticket is not created for hotel {}", hotelConnection.getHotelCode());
        }
    }

    private String updateTicket(String key, String title, String description) {
        try {
            IssueUpdate issueUpdateRequest = IssueUpdate.builder()
                    .summary(title)
                    .description(description)
                    .build();
            Issue issue = startrekClient.issues().update(key, issueUpdateRequest);
            return issue.getKey();
        } catch (StartrekClientException e) {
            if (e.getErrors().getStatusCode() >= 500) {
                throw new StartrekRetryableException(e);
            } else {
                throw e;
            }
        } catch (StartrekInternalClientError e) {
            throw new StartrekRetryableException(e);
        }
    }

    private String createTicket(String title, String description, String queue, String issueType, List<String> tags,
                                String parent) {
        try {
            IssueCreate.Builder issueCreateRequest = IssueCreate.builder()
                    .queue(queue)
                    .summary(title)
                    .type(issueType)
                    .tags(new ArrayListF<>(tags))
                    .description(description);
            if (parent != null) {
                issueCreateRequest.parent(parent);
            }
            Issue issue = startrekClient.issues().create(issueCreateRequest.build());
            return issue.getKey();
        } catch (StartrekClientException e) {
            if (e.getErrors().getStatusCode() >= 500) {
                throw new StartrekRetryableException(e);
            } else {
                throw e;
            }
        } catch (StartrekInternalClientError e) {
            throw new StartrekRetryableException(e);
        }
    }

    public String createOrUpdateLegalDetailsTicket(LegalDetails legalDetails) {
        if (legalDetails.getStTicket() == null) {
            return createTicket(
                    String.format(properties.getLegalDetailsTicketTitle(), legalDetails.getLegalName(),
                            legalDetails.getInn()),
                    templateService.getLegalDetailsTicketDescription(legalDetails),
                    properties.getQueueName(),
                    "task",
                    List.of(legalDetails.getPartnerId() + "_legal"),
                    null);
        } else {
            return updateTicket(
                    legalDetails.getStTicket(),
                    String.format(properties.getLegalDetailsTicketTitle(), legalDetails.getLegalName(),
                            legalDetails.getInn()),
                    templateService.getLegalDetailsTicketDescription(legalDetails));
        }
    }

    public String createVerifyClusteringTicket(String hotelName, String hotelCode, long permalink, String parent,
                                               EPartnerId partnerId) {
        return createTicket(
                String.format(properties.getVerifyClusteringTicketTitle(), hotelName, hotelCode, partnerId.toString(),
                        permalink),
                properties.getVerifyClusteringTicketDescription(),
                properties.getQueueName(),
                "task",
                List.of(partnerId + "_hotels_clustering"),
                parent);
    }

    public String createHotelIsMissingInGSTicket(String hotelCode, EPartnerId partnerId, String parent) {
        return createTicket(
                String.format(properties.getHotelIsMissingInGSTicketTitle(), hotelCode, partnerId.toString()),
                properties.getHotelIsMissingInGSTicketDescription(),
                properties.getQueueName(),
                "task",
                List.of("not_found_in_feed"),
                parent);
    }

    private String mapChangeTypeToTag(UpdateResult.ChangeRequisitesType changeType) {
        switch (changeType) {
            case INN_CHANGE:
                return "changed_inn";
            case BANK_ACCOUNT_CHANGE:
                return "changed_bank_account";
            case OTHER_CHANGE:
                return "changed_other";
            case PAPER_AGREEMENT_CHANGE:
                return "changed_paper_agreement";
            case WRONG_BIK_AND_ACCOUNT_PAIR:
                return "wrong_bik_and_account_pair";
            default:
                throw new RuntimeException("Unexpected changeType " + changeType);
        }
    }

    public void createManualVerificationTicket(UpdateResult updateResult) {
        HotelConnection hotelConnection = updateResult.getHotelConnection();
        String ticketKey = createTicket(
                String.format(
                        properties.getManualVerificationTicketTitle(),
                        hotelConnection.getHotelName(),
                        hotelConnection.getHotelCode(),
                        hotelConnection.getPartnerId().toString(),
                        hotelConnection.getPermalink()),
                generateManualVerificationTicketDescription(updateResult),
                properties.getQueueName(),
                "task",
                List.of("change_requisites", mapChangeTypeToTag(updateResult.getChangeType())),
                null);
        updateResult.getHotelConnectionUpdate().setStTicket(ticketKey);
         if (updateResult.getHotelConnection().getStTicket() != null) {
             linkTickets(ticketKey, updateResult.getHotelConnection().getStTicket());
         }
    }

    private String generateManualVerificationTicketDescription(UpdateResult updateResult) {
        return templateService.getManualVerificationTicketDescription(
                updateResult.getHotelConnection().getHotelName(),
                updateResult.getOldLegalDetails(),
                updateResult.getExistingLegalDetails(),
                updateResult.getNewLegalDetails(),
                updateResult.getHotelConnectionUpdate(),
                updateResult.getBoundHotels(),
                updateResult.getRelevantHotels(),
                updateResult.getChangeType());
    }

    public String createIssueForGenericWorkflowError(String title, String description) {
        return createTicket(
                title,
                description,
                properties.getQueueName(),
                "task",
                List.of("workflow_crashed", "hotel_duty_required"),
                null);
    }

    public void commentClusteringStarted(String ticketKey) {
        addComment(ticketKey, properties.getCommentClusteringStarted());
    }

    public void commentClusteringVerified(String ticketKey) {
        addComment(ticketKey, properties.getCommentClusteringVerified());
    }

    public void commentLegalDetailsReady(String ticketKey) {
        addComment(ticketKey, properties.getCommentLegalDetailsReady());
    }

    public void commentPermalinkChanged(String ticketKey, Long newPermalink) {
        String message = String.format("%s %d", properties.getCommentPermalinkChanged(), newPermalink);
        addComment(ticketKey, message);
    }

    public void closeLegalDetailsTicket(String ticketKey) {
        closeTicket(ticketKey, properties.getCommentLegalDetailsRegistered());
    }

    private Transition findCloseTransition(String ticketKey) {
        return startrekClient.transitions().getAll(ticketKey).stream()
                .filter(transition -> transition.getTo().getKey().equals("closed"))
                .findFirst().orElseThrow();
    }

    private void closeTicket(String ticketKey, String comment) {
        try {
            startrekClient.transitions().execute(
                    ticketKey,
                    findCloseTransition(ticketKey),
                    IssueUpdate
                            .comment(comment)
                            .resolution("fixed")
                            .build());
        } catch (StartrekClientException e) {
            if (e.getErrors().getStatusCode() >= 500) {
                throw new StartrekRetryableException(e);
            } else {
                throw e;
            }
        } catch (StartrekInternalClientError e) {
            throw new StartrekRetryableException(e);
        }
    }

    public void reopenTicket(String ticketKey) {
        try {
            startrekClient.transitions().execute(
                    ticketKey,
                    "reopen");
        } catch (StartrekClientException e) {
            if (e.getErrors().getStatusCode() >= 500) {
                throw new StartrekRetryableException(e);
            } else {
                throw e;
            }
        } catch (StartrekInternalClientError e) {
            throw new StartrekRetryableException(e);
        }
    }

    public void closeHotelConnectionTicket(String ticketKey) {
        closeTicket(ticketKey, properties.getCommentHotelGetsPublished());
    }

    public void linkTickets(String ticket1, String ticket2) {
        try {
            for (LocalLink existingLink: startrekClient.links().getLocal(ticket1)) {
                if (existingLink.getObject().getKey().equals(ticket2)) {
                    log.info("Tickets {} and {} are already linked", ticket1, ticket2);
                    return;
                }
            }
            startrekClient.links().create(ticket1, ticket2, Relationship.RELATES);
        } catch (StartrekClientException e) {
            if ((e instanceof EntityNotFoundException) || e.getErrors().getStatusCode() >= 500) {
                throw new StartrekRetryableException(e);
            } else {
                throw e;
            }
        } catch (StartrekInternalClientError e) {
            throw new StartrekRetryableException(e);
        }
    }

    public void commentHotelAppearedInGS(String ticketKey) {
        addComment(ticketKey, properties.getCommentHotelAppearedInGS());
    }

    public void closeAndCommentHotelAppearedInGS(String ticketKey) {
        closeTicket(ticketKey, properties.getCommentHotelAppearedInGS());
    }

    public void commentGSCallHotelNotFound(String ticketKey, String altaySignalLink, String altaySignalYqlLink,
                                           String feedYqlLink, String altayMappingsYqlLink, String geosearchLink,
                                           boolean foundInFeed, TAltaySignalRecord altaySignal,
                                           TAltayPublishingRecord bestAltayPublishingInfo,
                                           Collection<TAltayPublishingRecord> altayPublishingInfos,
                                           String altayCompanyYqlLink) {
        var comment = templateService.getHotelNotFoundComment(altaySignalLink, altaySignalYqlLink,
                feedYqlLink, altayMappingsYqlLink, geosearchLink, foundInFeed, altaySignal, bestAltayPublishingInfo,
                altayPublishingInfos, altayCompanyYqlLink);
        addComment(ticketKey, comment);
    }

    public void commentGSCallFailed(String ticketKey, Exception e) {
        addComment(ticketKey, String.format(properties.getCommentGSCallFailed(),
                SupervisorUtils.extractException(ProtoUtils.errorFromThrowable(e, true))));
    }

    public void commentHotelConnectionUpdateProcessing(String connectionUpdateTicket) {
        addComment(connectionUpdateTicket, properties.getCommentConnectionUpdateProcessing());
    }

    public void rejectConnectionUpdate(String connectionUpdateTicket) {
        closeTicket(connectionUpdateTicket, properties.getHotelConnectionUpdateRejected());
    }

    public void commentLegalDetailsHaveBeenPublished(String connectionUpdateTicket) {
        addComment(connectionUpdateTicket, properties.getCommentLegalDetailsHaveBeenPublished());
    }

    public void commentFinancialEventsUpdated(String connectionUpdateTicket) {
        addComment(connectionUpdateTicket, properties.getCommentFinancialEventsUpdated());
    }

    public void closeHotelConnectionUpdateTicket(String connectionUpdateTicket, long clientId) {
        closeTicket(connectionUpdateTicket, String.format(properties.getHotelConnectionUpdateFinished(), clientId));
    }

    public List<String> getHotelConnectionUpdateAutoProcessingTickets(Set<String> exclusions, int limit) {
        return getHotelConnectionUpdateAutoProcessingTicketsImpl(exclusions).limit(limit).collect(Collectors.toUnmodifiableList());
    }

    public long countHotelConnectionUpdateAutoProcessingTickets(Set<String> exclusions) {
        return getHotelConnectionUpdateAutoProcessingTicketsImpl(exclusions).count();
    }

    private Stream<String> getHotelConnectionUpdateAutoProcessingTicketsImpl(Set<String> exclusions) {
        return startrekClient.issues().find(String.format("QUEUE:%s AND Tags: %s", properties.getQueueName(), properties.getHotelConnectionUpdateAutoProcessingWaitTag()))
                .stream()
                .map(IssueRef::getKey)
                .filter(x -> !exclusions.contains(x))
                .filter(x -> startrekClient.issues().get(x).getTags().toList().containsTs(properties.getHotelConnectionUpdateAutoProcessingWaitTag()));
    }

    @Getter
    @AllArgsConstructor
    enum AutoProcessingActionEnum {
        ACCEPT("auto_processing_accept"),
        ACCEPT_NEW_CLIENT("auto_processing_accept_new_client"),
        REJECT("auto_processing_reject");

        private final String value;
    }

    public AutoProcessingActionEnum getHotelConnectionUpdateAutoProcessingAction(String ticketKey) {
        var mapping = Arrays.stream(AutoProcessingActionEnum.values())
                .collect(Collectors.toUnmodifiableMap(AutoProcessingActionEnum::getValue, x -> x));

        var actions = getTags(ticketKey)
                .stream()
                .filter(mapping::containsKey)
                .map(mapping::get)
                .collect(Collectors.toUnmodifiableList());

        Preconditions.checkState(actions.size() == 1, "Expected exactly one auto-action tag");
        return actions.get(0);
    }

    public boolean isWrongBikAndAccountTicket(String ticketKey) {
        var wrongBikAndAccountTag = mapChangeTypeToTag(UpdateResult.ChangeRequisitesType.WRONG_BIK_AND_ACCOUNT_PAIR);
        return getTags(ticketKey).stream().anyMatch(x -> x.equals(wrongBikAndAccountTag));
    }

    public void commentCantAcceptWrongBikAndAccount(String ticketKey) {
        addComment(ticketKey, properties.getCommentCantAcceptWrongBikAndAccount());
    }

    public void commentCantProcessHotelConnectionTwice(String ticketKey) {
        addComment(ticketKey, properties.getCommentCantProcessHotelConnectionTwice());
    }

    public void removeHotelConnectionUpdateAutoProcessingTag(String ticketKey) {
        startrekClient.issues()
                .get(ticketKey)
                .update(IssueUpdate.tags(
                        new ArrayListF<>(List.of()),
                        new ArrayListF<>(List.of(properties.getHotelConnectionUpdateAutoProcessingWaitTag()))
                ).build());
    }

    public void removeHotelConnectionUpdateActionTag(String ticketKey, AutoProcessingActionEnum action) {
        startrekClient.issues()
                .get(ticketKey)
                .update(IssueUpdate.tags(
                        new ArrayListF<>(List.of()),
                        new ArrayListF<>(List.of(action.getValue()))
                ).build());
    }
}
