package ru.yandex.travel.hotels.administrator.service;

import java.io.IOException;
import java.io.StringWriter;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableMap;
import com.mitchellbosecke.pebble.PebbleEngine;
import com.mitchellbosecke.pebble.error.PebbleException;
import com.mitchellbosecke.pebble.template.PebbleTemplate;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.administrator.cache.proto.TAltayPublishingRecord;
import ru.yandex.travel.hotels.administrator.cache.proto.TAltaySignalRecord;
import ru.yandex.travel.hotels.administrator.entity.HotelConnection;
import ru.yandex.travel.hotels.administrator.entity.HotelConnectionUpdate;
import ru.yandex.travel.hotels.administrator.entity.LegalDetails;
import ru.yandex.travel.workflow.entities.WorkflowEvent;

@Slf4j
@Service
public class TemplateService {

    private final static String TEMPLATES_FOLDER = "templates/";
    private final static String MANUAL_VERIFICATION_DESCRIPTION = "ManualVerificationDescription";
    private final static String HOTEL_CONNECTION_TICKET_DESCRIPTION = "HotelConnectionTicketDescription";
    private final static String LEGAL_DETAILS_TICKET_DESCRIPTION = "LegalDetailsTicketDescription";
    private final static String WORKFLOW_CRASHED_TICKET_DESCRIPTION = "WorkflowCrashedTicketDescription";
    private final static String HOTEL_NOT_FOUND_COMMENT = "HotelNotFoundComment";

    private final PebbleEngine pebbleEngine;

    private final ImmutableMap<String, PebbleTemplate> templateMap;


    public TemplateService() {
        this.pebbleEngine = new PebbleEngine.Builder()
                .build();
        templateMap = ImmutableMap.<String, PebbleTemplate>builder()
                .put(MANUAL_VERIFICATION_DESCRIPTION, readTemplate(TEMPLATES_FOLDER + "tickets/" + MANUAL_VERIFICATION_DESCRIPTION + ".peb"))
                .put(HOTEL_CONNECTION_TICKET_DESCRIPTION, readTemplate(TEMPLATES_FOLDER + "tickets/" + HOTEL_CONNECTION_TICKET_DESCRIPTION + ".peb"))
                .put(LEGAL_DETAILS_TICKET_DESCRIPTION, readTemplate(TEMPLATES_FOLDER + "tickets/" + LEGAL_DETAILS_TICKET_DESCRIPTION + ".peb"))
                .put(WORKFLOW_CRASHED_TICKET_DESCRIPTION, readTemplate(TEMPLATES_FOLDER + "tickets/" + WORKFLOW_CRASHED_TICKET_DESCRIPTION + ".peb"))
                .put(HOTEL_NOT_FOUND_COMMENT, readTemplate(TEMPLATES_FOLDER + "tickets/" + HOTEL_NOT_FOUND_COMMENT + ".peb"))
                .build();
    }

    public String getHotelNotFoundComment(String altaySignalLink, String altaySignalYqlLink,
                                          String feedYqlLink, String altayMappingsYqlLink, String geosearchLink,
                                          boolean foundInFeed, TAltaySignalRecord altaySignal,
                                          TAltayPublishingRecord bestAltayPublishingInfo,
                                          Collection<TAltayPublishingRecord> altayPublishingInfos,
                                          String altayCompanyYqlLink) {
        return evaluateTemplate(new HashMap<>() {{
            put("feedYqlLink", feedYqlLink);
            put("altaySignalYqlLink", altaySignalYqlLink);
            put("altaySignalLink", altaySignalLink);
            put("altayMappingsYqlLink", altayMappingsYqlLink);
            put("geosearchLink", geosearchLink);
            put("foundInFeed", foundInFeed);
            put("altaySignalStatus", altaySignal == null ? "UNKNOWN" : altaySignal.getStatus());
            put("altaySignalReason", altaySignal == null ? "UNKNOWN" : altaySignal.getReason());
            put("altaySignalComment", altaySignal == null ? "UNKNOWN" : altaySignal.getComment());
            put("bestAltayPublishing", bestAltayPublishingInfo);
            put("altayPublishingInfos", altayPublishingInfos);
            put("altayCompanyYqlLink", altayCompanyYqlLink);
        }}, HOTEL_NOT_FOUND_COMMENT);
    }

    public String getHotelConnectionTicketDescription(HotelConnection hotelConnection) {
        return evaluateTemplate(Map.of("connection", hotelConnection), HOTEL_CONNECTION_TICKET_DESCRIPTION);
    }

    public String getLegalDetailsTicketDescription(LegalDetails legalDetails) {
        return evaluateTemplate(Map.of("legalDetails", legalDetails), LEGAL_DETAILS_TICKET_DESCRIPTION);
    }

    public String getManualVerificationTicketDescription(String hotelName, LegalDetails oldDetails,
                                                         LegalDetails existingDetails, LegalDetails newDetails,
                                                         HotelConnectionUpdate connectionUpdate,
                                                         List<UpdateResult.HotelData> boundHotels,
                                                         List<UpdateResult.HotelData> relevantHotels,
                                                         UpdateResult.ChangeRequisitesType changeType) {
        Map<String, Object> context = new HashMap<>();
        context.put("updateId", connectionUpdate.getId().toString());
        context.put("hotelName", hotelName);
        context.put("oldDetails", oldDetails);
        context.put("existingDetails", existingDetails);
        context.put("newDetails", newDetails);
        context.put("boundHotels", boundHotels);
        context.put("relevantHotels", relevantHotels);
        context.put("changeType", changeType);
        return evaluateTemplate(context, MANUAL_VERIFICATION_DESCRIPTION);
    }

    public String getWorkflowCrashedTicketDescription(String entityType, String entityId, String workflowId,
                                                      String ticket, WorkflowEvent failedEvent, String exception) {
        Map<String, Object> context = new HashMap<>();
        context.put("entityType", entityType);
        context.put("entityId", entityId);
        context.put("workflowId", workflowId);
        context.put("ticket", ticket);
        context.put("failedEvent", failedEvent);
        context.put("exception", exception);
        return evaluateTemplate(context, WORKFLOW_CRASHED_TICKET_DESCRIPTION);
    }

    private String evaluateTemplate(@NonNull Map<String, Object> context,
                                    @NonNull String templateName) {
        try {
            StringWriter writer = new StringWriter();
            templateMap.get(templateName).evaluate(writer, context);
            return writer.toString();
        } catch (IOException | PebbleException e) {
            throw new RuntimeException("Error creating " + templateName + " from template", e);
        }
    }

    private PebbleTemplate readTemplate(String templatePath) {
        try {
            return pebbleEngine.getTemplate(templatePath);
        } catch (PebbleException e) {
            log.error("Error loading template from {}", templatePath);
            throw new RuntimeException(e);
        }
    }
}
