package ru.yandex.travel.hotels.administrator.service.partners;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.administrator.grpc.proto.EAgreementStatusType;
import ru.yandex.travel.hotels.administrator.service.AgreementService;
import ru.yandex.travel.hotels.administrator.service.partners.mappers.HotelDetailsMapper;
import ru.yandex.travel.hotels.administrator.service.partners.model.HotelConnectionStatus;
import ru.yandex.travel.hotels.administrator.service.partners.model.HotelDetailsDTO;
import ru.yandex.travel.hotels.common.partners.bnovo.BNovoClient;
import ru.yandex.travel.hotels.proto.EPartnerId;

@Service
@RequiredArgsConstructor
public class BnovoService implements PartnerService {

    private final BNovoClient bNovoClient;

    private final HotelDetailsMapper hotelDetailsMapper;

    private final AgreementService agreementService;

    @Override
    public HotelDetailsDTO getHotelDetails(String hotelCode) {
        return hotelDetailsMapper.convertBnovoHotelDetails(
                bNovoClient.getHotelDetailsSync(hotelCode).getHotelDetails());
    }

    @Override
    public void notifyHotelStatusChanged(String hotelCode) {
        bNovoClient.notifyHotelStatusChangedSync(hotelCode);
    }

    @Override
    public void verifyHotelDetails(HotelDetailsDTO hotelDetails) {
        Preconditions.checkState(hotelDetails.getConnectionStatus().equals(HotelConnectionStatus.CONNECTED),
                "The hotel status is not_connected; hotelCode %s",
                hotelDetails.getHotelCode());
        EAgreementStatusType agreementStatusType = agreementService.checkAgreement(
                hotelDetails.getBankAccountDetails().getInn());
        //TODO probably it is better to suspend connection process until agreement is accepted, currently the workflow will fail
        Preconditions.checkState(agreementStatusType.equals(EAgreementStatusType.AS_FOUND),
                "Active agreement has not been found for INN %s, agreement status: %s, partner: PI_BNOVO, hotelCode: %s",
                hotelDetails.getBankAccountDetails().getInn(),
                agreementStatusType,
                hotelDetails.getHotelCode());
    }

    @Override
    public EPartnerId getPartner() {
        return EPartnerId.PI_BNOVO;
    }
}
