package ru.yandex.travel.hotels.administrator.service.partners;

import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.administrator.service.partners.mappers.HotelDetailsMapper;
import ru.yandex.travel.hotels.administrator.service.partners.model.HotelDetailsDTO;
import ru.yandex.travel.hotels.administrator.service.partners.model.HotelOfferStatus;
import ru.yandex.travel.hotels.common.partners.travelline.TravellineClient;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelListItem;
import ru.yandex.travel.hotels.proto.EPartnerId;

@Service
@RequiredArgsConstructor
public class TravellineService implements PartnerService {

    private final TravellineClient travellineClient;

    private final HotelDetailsMapper hotelDetailsMapper;

    @Override
    public HotelDetailsDTO getHotelDetails(String hotelCode) {
        return hotelDetailsMapper.convertTravellineHotelDetails(
                travellineClient.getHotelDetailsSync(hotelCode).getHotelDetails());
    }

    @Override
    public void notifyHotelStatusChanged(String hotelCode) {
        travellineClient.notifyHotelStatusChangedSync(hotelCode);
    }

    @Override
    public void verifyHotelDetails(HotelDetailsDTO hotelDetails) {
        Preconditions.checkState(hotelDetails.getOfferStatus().equals(HotelOfferStatus.ACCEPTED),
                "The hotel hasn't accepted the offer; hotelCode %s, status %s",
                hotelDetails.getHotelCode(), hotelDetails.getOfferStatus());
    }

    public Set<String> getActiveHotelIds() {
        return travellineClient.listHotelsSync().getHotels().stream()
                .filter(hli -> hli.getInventoryVersion() > 0)
                .map(HotelListItem::getCode).collect(Collectors.toSet());
    }

    @Override
    public EPartnerId getPartner() {
        return EPartnerId.PI_TRAVELLINE;
    }
}
