package ru.yandex.travel.hotels.administrator.task;

import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;

import ru.yandex.travel.hotels.administrator.configuration.BillingContractFlagsSynchronizerProperties;
import ru.yandex.travel.hotels.administrator.entity.LegalDetails;
import ru.yandex.travel.hotels.administrator.repository.LegalDetailsRepository;
import ru.yandex.travel.hotels.administrator.service.Meters;
import ru.yandex.travel.hotels.administrator.workflow.proto.ELegalDetailsState;
import ru.yandex.travel.integration.balance.BillingApiClient;
import ru.yandex.travel.integration.balance.BillingClientContract;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.utils.ClockService;

@Slf4j
public class BillingContractFlagsSynchronizer {

    private static final String ERROR_COUNTER_TAG = "BillingFlagsSynchronizer";

    private final BillingContractFlagsSynchronizerProperties properties;
    private final LegalDetailsRepository legalDetailsRepository;
    private final BillingApiClient billingApiClient;
    private final ClockService clockService;
    private final Meters meters;

    public BillingContractFlagsSynchronizer(BillingContractFlagsSynchronizerProperties properties,
                                            LegalDetailsRepository legalDetailsRepository,
                                            BillingApiClient billingApiClient, ClockService clockService,
                                            Meters meters) {
        this.properties = properties;
        this.legalDetailsRepository = legalDetailsRepository;
        this.billingApiClient = billingApiClient;
        this.clockService = clockService;
        this.meters = meters;
        this.meters.initCounter(ERROR_COUNTER_TAG);
    }

    @TransactionMandatory
    public List<UUID> getReadyTasks(Set<UUID> excludeIds, int limit) {
        Pageable paging = PageRequest.of(0, limit);
        Set<UUID> safeExcludeIds = excludeIds != null && !excludeIds.isEmpty() ? excludeIds : LegalDetailsRepository.NO_EXCLUDE_IDS;
        return legalDetailsRepository.findIdsForFlagsSynchronization(getMaxSynchronizedAt(), safeExcludeIds,
                ELegalDetailsState.DS_REGISTERED, paging);
    }

    @TransactionMandatory
    public long countPendingTasks(Set<UUID> excludeIds) {
        Set<UUID> safeExcludeIds = excludeIds != null && !excludeIds.isEmpty() ? excludeIds : LegalDetailsRepository.NO_EXCLUDE_IDS;
        return legalDetailsRepository.countIdsForFlagsSynchronization(getMaxSynchronizedAt(), safeExcludeIds,
                ELegalDetailsState.DS_REGISTERED);
    }

    @TransactionMandatory
    public void processTask(UUID legalDetailsId) {
        try {
            log.debug("Synchronizing billing flags for LegalDetails with id {}", legalDetailsId);
            LegalDetails legalDetails = legalDetailsRepository.getOne(legalDetailsId);
            Optional<BillingClientContract> contractOptional =
                    billingApiClient.getClientContracts(legalDetails.getBalanceClientId()).stream()
                    .filter(contract -> contract.getId().equals(legalDetails.getBalanceContractId()))
                    .findFirst();
            if (!contractOptional.isPresent()) {
                log.error("No contract with ID "
                        + legalDetails.getBalanceContractId() + " was found in Balance for client "
                        + legalDetails.getBalanceClientId());
                meters.incrementCounter(ERROR_COUNTER_TAG);
                return;
            }
            BillingClientContract billingContract = contractOptional.get();
            legalDetails.setBillingActive(billingContract.isActive());
            legalDetails.setBillingSigned(billingContract.isSigned());
            legalDetails.setOfferAccepted(billingContract.isOfferAccepted());
            legalDetails.setBillingCancelled(billingContract.isCancelled());
            legalDetails.setBillingDeactivated(billingContract.isDeactivated());
            legalDetails.setBillingSuspended(billingContract.isSuspended());
            legalDetails.setFlagsSynchronizedAt(Instant.now(clockService.getUtc()));
        } catch (Exception e) {
            meters.incrementCounter(ERROR_COUNTER_TAG);
            throw e;
        }
    }

    public Duration getCurrentProcessingDelay() {
        Instant synchronizeSinceTs = getMaxSynchronizedAt();
        Instant earliestSyncTimestamp = legalDetailsRepository.findOldestTimestampForBillingFlagSynchronization(
                synchronizeSinceTs, ELegalDetailsState.DS_REGISTERED);
        if (earliestSyncTimestamp == null) {
            return Duration.ZERO;
        } else {
            return Duration.between(earliestSyncTimestamp, synchronizeSinceTs);
        }
    }

    private Instant getMaxSynchronizedAt() {
        return Instant.now(clockService.getUtc()).minus(properties.getSynchronizationInterval());
    }
}
