package ru.yandex.travel.hotels.administrator.task;

import java.time.Instant;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Timer;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.tables.types.NativeProtobufEntryType;
import ru.yandex.travel.commons.datetime.DateTimeUtils;
import ru.yandex.travel.commons.metrics.MetricsUtils;
import ru.yandex.travel.hotels.administrator.cache.HotelClusteringDictionary;
import ru.yandex.travel.hotels.administrator.configuration.BillingServiceProperties;
import ru.yandex.travel.hotels.administrator.configuration.YtConnectionProperties;
import ru.yandex.travel.hotels.administrator.configuration.YtPublishTaskProperties;
import ru.yandex.travel.hotels.administrator.entity.HotelConnection;
import ru.yandex.travel.hotels.administrator.entity.LegalDetails;
import ru.yandex.travel.hotels.administrator.export.proto.ContractInfo;
import ru.yandex.travel.hotels.administrator.export.proto.HotelAgreement;
import ru.yandex.travel.hotels.administrator.export.proto.HotelLegalInfo;
import ru.yandex.travel.hotels.administrator.export.proto.WhitelistedHotel;
import ru.yandex.travel.hotels.administrator.repository.HotelConnectionRepository;
import ru.yandex.travel.hotels.administrator.repository.LegalDetailsRepository;
import ru.yandex.travel.hotels.administrator.service.Meters;
import ru.yandex.travel.hotels.administrator.workflow.proto.EHotelConnectionState;
import ru.yandex.travel.hotels.administrator.workflow.proto.ELegalDetailsState;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.yt.util.YtHelper;
import ru.yandex.travel.yt.util.YtProtoUtils;

@Slf4j
public class HotelsYtPublishTask extends AbstractSingletonTask {
    private static final String SINGLETON_HOTEL_PUBLISH_TASK = "singletonHotelPublishTask";
    private static final boolean DO_NOT_PING_ANCESTOR_TRANSACTION = true;
    private static final String ERROR_COUNTER_TAG = "YtPublisher";
    private final YtConnectionProperties connectionProperties;
    private final YtPublishTaskProperties publisherProperties;
    private final BillingServiceProperties billingServiceProperties;
    private final HotelConnectionRepository hotelConnectionRepository;
    private final LegalDetailsRepository legalDetailsRepository;
    private final HotelClusteringDictionary hotelClusteringDictionary;
    private final Meters meters;
    private final Map<String, Yt> ytClusters;
    private final Map<String, Timer> clusterTimers = new HashMap<>();

    public HotelsYtPublishTask(YtConnectionProperties connectionProperties,
                               YtPublishTaskProperties publisherProperties,
                               BillingServiceProperties billingServiceProperties,
                               HotelConnectionRepository hotelConnectionRepository,
                               LegalDetailsRepository legalDetailsRepository,
                               HotelClusteringDictionary hotelClusteringDictionary, Meters meters) {
        this.publisherProperties = publisherProperties;
        this.connectionProperties = connectionProperties;
        this.billingServiceProperties = billingServiceProperties;
        this.hotelConnectionRepository = hotelConnectionRepository;
        this.legalDetailsRepository = legalDetailsRepository;
        this.hotelClusteringDictionary = hotelClusteringDictionary;
        ytClusters = publisherProperties.getClusters().stream().collect(Collectors.toMap(Function.identity(),
                cluster -> YtUtils.http(cluster, connectionProperties.getToken())));
        this.meters = meters;
        this.meters.initCounter(ERROR_COUNTER_TAG);
        ytClusters.keySet().forEach(cluster -> {
            clusterTimers.put(cluster,
                    Timer.builder("ytPublisher.publishTime")
                            .tag("ytCluster", cluster)
                            .serviceLevelObjectives(MetricsUtils.mediumDurationSla())
                            .publishPercentiles(MetricsUtils.higherPercentiles())
                            .publishPercentileHistogram()
                            .register(Metrics.globalRegistry));
        });
    }

    private String formatRegisteredAt(Instant registeredAt) {
        return DateTimeUtils.standardFormatLocalDate(LocalDate.ofInstant(registeredAt, billingServiceProperties.getBillingZoneOffset()));
    }

    @TransactionMandatory
    public void publishHotelsToYt(String taskId) {
        List<Exception> occurredExceptions = new ArrayList<>();
        for (Map.Entry<String, Yt> entry : ytClusters.entrySet()) {
            String cluster = entry.getKey();
            Yt yt = entry.getValue();
            long startTime = System.nanoTime();
            try {
                YtHelper.doInTx(yt, connectionProperties.getTransactionDuration(), txId -> {
                    log.info("Publishing hotels in yt transaction [{}]", txId);
                    Preconditions.checkArgument(SINGLETON_HOTEL_PUBLISH_TASK.equals(taskId));
                    List<HotelConnection> publishedHotels =
                            hotelConnectionRepository.findAllByState(EHotelConnectionState.CS_PUBLISHED).stream().filter(h -> h.getPermalink() != null).collect(Collectors.toList());
                    YPath whitelistTable = YtHelper.createTable(yt, txId, DO_NOT_PING_ANCESTOR_TRANSACTION,
                            publisherProperties.getWhitelistTablePath(),
                            YtProtoUtils.getTableSchemaForMessage(WhitelistedHotel.getDefaultInstance()));
                    NativeProtobufEntryType<WhitelistedHotel> whitelistedHotelEntryType =
                            new NativeProtobufEntryType<>(WhitelistedHotel.newBuilder());
                    var whitelistHotels = publishedHotels.stream().filter(HotelConnection::isReadyForOffercache);
                    if (publisherProperties.isUseMatchedForWhitelist()) {
                        whitelistHotels = whitelistHotels.filter(this::isClusteringVerified);
                    }
                    if (publisherProperties.isAutoDisableUnavailableHotels()) {
                        whitelistHotels = whitelistHotels.filter(h -> h.availableAtPartner);
                    }
                    Iterator<WhitelistedHotel> hotelRecords = mapToWhitelistRecords(whitelistHotels);
                    yt.tables().write(Optional.of(txId), DO_NOT_PING_ANCESTOR_TRANSACTION, whitelistTable,
                            whitelistedHotelEntryType, hotelRecords);
                    YPath agreementsTable = YtHelper.createTable(yt, txId, DO_NOT_PING_ANCESTOR_TRANSACTION,
                            publisherProperties.getAgreementTablePath(),
                            YtProtoUtils.getTableSchemaForMessage(HotelAgreement.getDefaultInstance()));
                    NativeProtobufEntryType<HotelAgreement> hotelAgreementEntryType =
                            new NativeProtobufEntryType<>(HotelAgreement.newBuilder());
                    Iterator<HotelAgreement> agreementRecords =
                            mapToHotelAgreementRecords(publishedHotels.stream().filter(HotelConnection::isReadyForOrchestrator));
                    yt.tables().write(Optional.of(txId), DO_NOT_PING_ANCESTOR_TRANSACTION, agreementsTable,
                            hotelAgreementEntryType, agreementRecords);
                    YPath legalInfoTable = YtHelper.createTable(yt, txId, DO_NOT_PING_ANCESTOR_TRANSACTION,
                            publisherProperties.getLegalInfoTablePath(),
                            YtProtoUtils.getTableSchemaForMessage(HotelLegalInfo.getDefaultInstance()));
                    NativeProtobufEntryType<HotelLegalInfo> hotelLegalInfoEntryType =
                            new NativeProtobufEntryType<>(HotelLegalInfo.newBuilder());
                    Iterator<HotelLegalInfo> hotelLegalInfoRecords =
                            mapToHotelLegalInfoRecords(publishedHotels.stream().filter(HotelConnection::isReadyForOffercache));
                    yt.tables().write(Optional.of(txId), DO_NOT_PING_ANCESTOR_TRANSACTION, legalInfoTable,
                            hotelLegalInfoEntryType, hotelLegalInfoRecords);
                    List<LegalDetails> registeredLegalDetails =
                            legalDetailsRepository.findAllByState(ELegalDetailsState.DS_REGISTERED);
                    YPath contractInfoTable = YtHelper.createTable(yt, txId, DO_NOT_PING_ANCESTOR_TRANSACTION,
                            publisherProperties.getContractInfoTablePath(),
                            YtProtoUtils.getTableSchemaForMessage(ContractInfo.getDefaultInstance()));
                    NativeProtobufEntryType<ContractInfo> contractInfoEntryType =
                            new NativeProtobufEntryType<>(ContractInfo.newBuilder());
                    Iterator<ContractInfo> contractInfoRecords = mapToContractInfoRecords(registeredLegalDetails);
                    yt.tables().write(Optional.of(txId), DO_NOT_PING_ANCESTOR_TRANSACTION, contractInfoTable,
                            contractInfoEntryType, contractInfoRecords);
                    List<HotelConnection> hotelConnections = hotelConnectionRepository.findAll();
                    YPath hotelConnectionTable = YtHelper.createTable(yt, txId, DO_NOT_PING_ANCESTOR_TRANSACTION,
                            publisherProperties.getHotelConnectionsTablePath(),
                            YtProtoUtils.getTableSchemaForMessage(ru.yandex.travel.hotels.administrator.export.proto.HotelConnection.getDefaultInstance()));
                    NativeProtobufEntryType<ru.yandex.travel.hotels.administrator.export.proto.HotelConnection> hotelConnectionEntryType =
                            new NativeProtobufEntryType<>(ru.yandex.travel.hotels.administrator.export.proto.HotelConnection.newBuilder());
                    Iterator<ru.yandex.travel.hotels.administrator.export.proto.HotelConnection> hotelConnectionRecords = mapToHotelConnectionRecords(hotelConnections);
                    yt.tables().write(Optional.of(txId), DO_NOT_PING_ANCESTOR_TRANSACTION, hotelConnectionTable,
                            hotelConnectionEntryType, hotelConnectionRecords);
                });
            } catch (Exception e) {
                occurredExceptions.add(e);
            }
            clusterTimers.get(cluster).record(System.nanoTime() - startTime, TimeUnit.NANOSECONDS);
        }
        if (!occurredExceptions.isEmpty()) {
            RuntimeException resultingException = new RuntimeException(String.format("%d exceptions occurred while " +
                    "publishing hotels ot yt clusters", occurredExceptions.size()));
            for (Exception exception : occurredExceptions) {
                resultingException.addSuppressed(exception);
            }
            meters.incrementCounter(ERROR_COUNTER_TAG);
            throw resultingException;
        }
    }

    private Iterator<ru.yandex.travel.hotels.administrator.export.proto.HotelConnection> mapToHotelConnectionRecords(List<HotelConnection> hotelConnections) {
        return hotelConnections.stream().map(hotelConnection -> {
            var hotelConnectionBuilder = ru.yandex.travel.hotels.administrator.export.proto.HotelConnection.newBuilder();
            if (hotelConnection.getHotelCode() != null) {
                hotelConnectionBuilder.setHotelCode(hotelConnection.getHotelCode());
            }
            if (hotelConnection.getExternalHotelId() != null) {
                hotelConnectionBuilder.setExternalHotelId(hotelConnection.getExternalHotelId());
            }
            if (hotelConnection.getPartnerId() != null) {
                hotelConnectionBuilder.setPartnerId(hotelConnection.getPartnerId());
            }
            if (hotelConnection.getHotelName() != null) {
                hotelConnectionBuilder.setHotelName(hotelConnection.getHotelName());
            }
            if (hotelConnection.getPermalink() != null) {
                hotelConnectionBuilder.setPermalink(hotelConnection.getPermalink());
            }
            if (hotelConnection.getStTicket() != null) {
                hotelConnectionBuilder.setStTicket(hotelConnection.getStTicket());
            }
            hotelConnectionBuilder.setReadyForOrchestrator(hotelConnection.isReadyForOrchestrator());
            hotelConnectionBuilder.setReadyForOffercache(hotelConnection.isReadyForOffercache());
            hotelConnectionBuilder.setClusteringVerified(hotelConnection.isClusteringVerified());
            hotelConnectionBuilder.setLegalDetailsRegistered(hotelConnection.isLegalDetailsRegistered());
            hotelConnectionBuilder.setGeoSearchCalled(hotelConnection.isGeoSearchCalled());
            hotelConnectionBuilder.setState(hotelConnection.getState());
            if (hotelConnection.getUnpublishedReason() != null) {
                hotelConnectionBuilder.setUnpublishedReason(hotelConnection.getUnpublishedReason().getValue());
            }
            if (hotelConnection.getAccountantEmail() != null) {
                hotelConnectionBuilder.setAccountantEmail(hotelConnection.getAccountantEmail());
            }
            if (hotelConnection.getReservationPhone() != null) {
                hotelConnectionBuilder.setReservationPhone(hotelConnection.getReservationPhone());
            }
            if (hotelConnection.getAddress() != null) {
                hotelConnectionBuilder.setAddress(hotelConnection.getAddress());
            }
            if (hotelConnection.getCityName() != null) {
                hotelConnectionBuilder.setCityName(hotelConnection.getCityName());
            }
            if (hotelConnection.getContractPersonName() != null) {
                hotelConnectionBuilder.setContractPersonName(hotelConnection.getContractPersonName());
            }
            if (hotelConnection.getContractPersonPosition() != null) {
                hotelConnectionBuilder.setContractPersonPosition(hotelConnection.getContractPersonPosition());
            }
            if (hotelConnection.getContractPersonPhone() != null) {
                hotelConnectionBuilder.setContractPersonPhone(hotelConnection.getContractPersonPhone());
            }
            if (hotelConnection.getContractPersonEmail() != null) {
                hotelConnectionBuilder.setContractPersonEmail(hotelConnection.getContractPersonEmail());
            }
            if (hotelConnection.getCreatedAt() != null) {
                hotelConnectionBuilder.setCreatedAt(hotelConnection.getCreatedAt().toEpochMilli());
            }
            if (hotelConnection.getFirstPublishAt() != null) {
                hotelConnectionBuilder.setFirstPublishAt(hotelConnection.getFirstPublishAt().toEpochMilli());
            }
            return hotelConnectionBuilder.build();
        }).iterator();
    }

    private Iterator<ContractInfo> mapToContractInfoRecords(List<LegalDetails> registeredLegalDetails) {
        return registeredLegalDetails.stream().map(legalDetails -> {
            ContractInfo.Builder contractInfoBuilder =
                    ContractInfo.newBuilder()
                            .setClientId(legalDetails.getBalanceClientId())
                            .setContractId(legalDetails.getBalanceContractId())
                            .setExternalContractId(legalDetails.getBalanceExternalContractId())
                            .setSendEmptyOrdersReport(legalDetails.isSendEmptyOrdersReport())
                            .setAccountantEmails(
                                    legalDetails.getHotelConnections().stream()
                                            .filter(hotelConnection -> hotelConnection.getState() != EHotelConnectionState.CS_NEW && hotelConnection.getState() != EHotelConnectionState.CS_PUBLISHING)
                                            .filter(hotelConnection -> hotelConnection.getAccountantEmail() != null)
                                            .map(HotelConnection::getAccountantEmail).collect(Collectors.joining(";")))
                            .setInn(legalDetails.getInn())
                            .setBillingActive(legalDetails.isBillingActive())
                            .setBillingSigned(legalDetails.isBillingSigned())
                            .setBillingOfferAccepted(legalDetails.isBillingOfferAccepted())
                            .setBillingCancelled(legalDetails.isBillingCancelled())
                            .setBillingDeactivated(legalDetails.isBillingDeactivated())
                            .setBillingSuspended(legalDetails.isBillingSuspended());
            if (!Strings.isNullOrEmpty(legalDetails.getFullLegalName())) {
                contractInfoBuilder.setFullLegalName(legalDetails.getFullLegalName());
            }
            if (legalDetails.getRegisteredAt() != null) {
                contractInfoBuilder.setExternalContractRegisterDate(formatRegisteredAt(legalDetails.getRegisteredAt()));
            }
            if (legalDetails.getFlagsSynchronizedAt() != null) {
                contractInfoBuilder.setBillingSynchronizedAt(legalDetails.getFlagsSynchronizedAt().toEpochMilli());
            }
            return contractInfoBuilder.build();
        }).iterator();
    }

    private Iterator<HotelLegalInfo> mapToHotelLegalInfoRecords(Stream<HotelConnection> hotelConnectionStream) {
        return hotelConnectionStream.map(hotelConnection -> {
            HotelLegalInfo.Builder legalInfoBuilder =
                    HotelLegalInfo.newBuilder().setHotelId(hotelConnection.getHotelCode()).setPartnerId(hotelConnection.getPartnerId()).setLegalName(hotelConnection.getLegalDetails().getLegalName())
                            .setInn(hotelConnection.getLegalDetails().getInn())
                            .setLegalAddress(extractLegalAddress(hotelConnection.getLegalDetails()))
                            .setWorkingHours("ежедневно, круглосуточно");
            if (StringUtils.isNotBlank(hotelConnection.getLegalDetails().getOgrn())) {
                legalInfoBuilder.setOgrn(hotelConnection.getLegalDetails().getOgrn());
            } else {
                legalInfoBuilder.setOgrn("");
            }
            return legalInfoBuilder.build();
        }).iterator();
    }

    private String extractLegalAddress(LegalDetails legalDetails) {
        if (!legalDetails.isLegalAddressUnified()) {
            return legalDetails.getLegalAddress();
        } else {
            return legalDetails.getLegalPostCode() + ", " + legalDetails.getLegalAddress();
        }
    }

    private Iterator<HotelAgreement> mapToHotelAgreementRecords(Stream<HotelConnection> hotelConnectionStream) {
        return hotelConnectionStream.flatMap(hotelConnection ->
                hotelConnection.getCommissions().stream().map(commission -> {
                    HotelAgreement.Builder agreementBuilder = HotelAgreement.newBuilder()
                            .setId(commission.getId())
                            .setHotelId(hotelConnection.getHotelCode())
                            .setPartnerId(hotelConnection.getPartnerId())
                            .setInn(hotelConnection.getLegalDetails().getInn())
                            .setFinancialClientId(hotelConnection.getLegalDetails().getBalanceClientId())
                            .setFinancialContractId(hotelConnection.getLegalDetails().getBalanceContractId())
                            .setOrderConfirmedRate(commission.getOrderConfirmedRate().toString())
                            .setOrderRefundedRate(commission.getOrderRefundedRate().toString())
                            .setAgreementStartDate(commission.getAgreementStartDate().toEpochMilli())
                            .setEnabled(commission.isEnabled())
                            .setPriority(commission.getPriority())
                            .setVatType(hotelConnection.getVatType())
                            .setSendEmptyOrdersReport(hotelConnection.getLegalDetails().isSendEmptyOrdersReport());
                    if (hotelConnection.getExternalHotelId() != null) {
                        agreementBuilder.setExternalHotelId(hotelConnection.getExternalHotelId());
                    }
                    if (hotelConnection.getAccountantEmail() != null) {
                        agreementBuilder.setAccountantEmail(hotelConnection.getAccountantEmail());
                    }
                    if (!Strings.isNullOrEmpty(hotelConnection.getLegalDetails().getFullLegalName())) {
                        agreementBuilder.setFullLegalName(hotelConnection.getLegalDetails().getFullLegalName());
                    }
                    if (!Strings.isNullOrEmpty(hotelConnection.getLegalDetails().getBalanceExternalContractId())) {
                        agreementBuilder.setExternalContractId(hotelConnection.getLegalDetails().getBalanceExternalContractId());
                    }
                    if (commission.getAgreementEndDate() != null) {
                        agreementBuilder.setAgreementEndDate(commission.getAgreementEndDate().toEpochMilli());
                    }
                    if (hotelConnection.getLegalDetails().getRegisteredAt() != null) {
                        agreementBuilder.setExternalContractRegisterDate(formatRegisteredAt(hotelConnection.getLegalDetails().getRegisteredAt()));
                    }
                    return agreementBuilder.build();
                })
        ).iterator();
    }

    private Iterator<WhitelistedHotel> mapToWhitelistRecords(Stream<HotelConnection> hotelConnectionStream) {
        return hotelConnectionStream.map(hotelConnection ->
                WhitelistedHotel.newBuilder()
                        .setOriginalId(hotelConnection.getHotelCode())
                        .setPartnerId(hotelConnection.getPartnerId().toString())
                        .setPermalink(hotelConnection.getPermalink())
                        .build()
        ).iterator();
    }

    private boolean isClusteringVerified(HotelConnection connection) {
        return hotelClusteringDictionary.isHotelClusteringVerified(
                connection.getPartnerId(),
                connection.getHotelCode()
        );
    }

    @Override
    String getTaskKey() {
        return SINGLETON_HOTEL_PUBLISH_TASK;
    }
}
