package ru.yandex.travel.hotels.administrator.task;

import java.util.ArrayList;
import java.util.List;

import com.google.common.base.Preconditions;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.Metrics;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.administrator.cache.HotelClusteringDictionary;
import ru.yandex.travel.hotels.administrator.cache.proto.THotelClustering;
import ru.yandex.travel.hotels.administrator.entity.HotelConnection;
import ru.yandex.travel.hotels.administrator.repository.HotelConnectionRepository;
import ru.yandex.travel.hotels.administrator.service.StarTrekService;
import ru.yandex.travel.hotels.administrator.workflow.proto.EHotelConnectionState;

@RequiredArgsConstructor
@Slf4j
public class PermalinkUpdateTask extends AbstractSingletonTask {

    private final static String TASK_ID = "permalinkUpdateTask";

    private final HotelConnectionRepository hotelConnectionRepository;
    private final HotelClusteringDictionary hotelClusteringDictionary;
    private final StarTrekService starTrekService;
    @Getter private int unmatchedCount = 0;

    {
        Gauge.builder("unmatchedHotels", this::getUnmatchedCount).register(Metrics.globalRegistry);
    }

    public void updatePermalinks(String taskId) {
        Preconditions.checkArgument(TASK_ID.equals(taskId));

        int unmatchedCount = 0;
        List<HotelConnection> hotels = hotelConnectionRepository.findAll();
        List<HotelConnection> hotelsToUpdate = new ArrayList<>();
        for (HotelConnection hotel : hotels) {
            var key = new HotelClusteringDictionary.ClusteringKey(hotel.getPartnerId(), hotel.getHotelCode());
            THotelClustering hotelClustering = hotelClusteringDictionary.getByKey(key);
            if (hotelClustering == null) {
                if (hotel.getState() == EHotelConnectionState.CS_PUBLISHED) {
                    unmatchedCount += 1;
                }
                continue;
            }

            Long savedPermalink = hotel.getPermalink();
            long actualPermalink = hotelClustering.getPermalink();

            boolean permalinkChanged;
            if (savedPermalink == null) {
                permalinkChanged = true;
            } else {
                permalinkChanged = !savedPermalink.equals(actualPermalink);
            }

            if (permalinkChanged) {
                hotel.setPermalink(actualPermalink);
                hotelsToUpdate.add(hotel);

                String stTicket = hotel.getStTicket();
                if (stTicket != null) {
                    starTrekService.commentPermalinkChanged(stTicket, actualPermalink);
                }
            }
        }
        if (!hotelsToUpdate.isEmpty()) {
            hotelConnectionRepository.saveAll(hotelsToUpdate);
            hotelConnectionRepository.flush();
        }
        this.unmatchedCount = unmatchedCount;
    }

    @Override
    String getTaskKey() {
        return TASK_ID;
    }
}
