package ru.yandex.travel.hotels.administrator.workflow.billingregistration;

import java.time.Instant;
import java.util.UUID;

import com.google.common.base.Preconditions;
import com.google.protobuf.Message;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.administrator.entity.BillingRegistration;
import ru.yandex.travel.hotels.administrator.entity.LegalDetails;
import ru.yandex.travel.hotels.administrator.service.BillingService;
import ru.yandex.travel.hotels.administrator.workflow.proto.EBillingRegistrationState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBillingRegistrationCreateClient;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBillingRegistrationCreateContract;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBillingRegistrationCreatePerson;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBillingRegistrationFinish;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBillingRegistrationStart;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBillingRegistrationStartUpdate;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBillingRegistrationUpdateClient;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBillingRegistrationUpdateFinished;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBillingRegistrationUpdatePerson;
import ru.yandex.travel.integration.balance.responses.BillingCreateContractResponse;
import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.base.AnnotatedWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

/**
 * As the workflow is a simple forward process without any external events,
 * all handlers are defined here in one place for simplicity.
 */
@Component
@RequiredArgsConstructor
@Slf4j
public class BillingRegistrationWorkflowHandler extends AnnotatedWorkflowEventHandler<BillingRegistration> {
    private final BillingService billingService;

    @HandleEvent
    public void startRegistration(TBillingRegistrationStart event, MessagingContext<BillingRegistration> context) {
        BillingRegistration registration = context.getWorkflowEntity();
        log.info("Starting registration of a new partner: {}", registration.getLegalDetails().getLegalName());
        checkState(registration, EBillingRegistrationState.BRS_NEW, event);
        Preconditions.checkState(registration.getClientId() == null, "Not empty clientId");
        Preconditions.checkState(registration.getPersonId() == null, "Not empty personId");
        Preconditions.checkState(registration.getContractId() == null, "Not empty contractId");
        Preconditions.checkState(registration.getExternalContractId() == null, "Not empty contractExternalId");
        registration.setState(EBillingRegistrationState.BRS_CREATING_CLIENT);
        context.scheduleEvent(TBillingRegistrationCreateClient.newBuilder().build());
    }

    @HandleEvent
    public void createClient(TBillingRegistrationCreateClient event, MessagingContext<BillingRegistration> context) {
        BillingRegistration registration = context.getWorkflowEntity();
        checkState(registration, EBillingRegistrationState.BRS_CREATING_CLIENT, event);

        LegalDetails legalDetails = registration.getLegalDetails();
        long newClientId = billingService.createClient(legalDetails);
        registration.setClientId(newClientId);

        registration.setState(EBillingRegistrationState.BRS_CREATING_PERSON);
        context.scheduleEvent(TBillingRegistrationCreatePerson.newBuilder().build());
    }

    @HandleEvent
    public void createPerson(TBillingRegistrationCreatePerson event, MessagingContext<BillingRegistration> context) {
        BillingRegistration registration = context.getWorkflowEntity();
        checkState(registration, EBillingRegistrationState.BRS_CREATING_PERSON, event);

        LegalDetails legalDetails = registration.getLegalDetails();
        long newPersonId = billingService.createPerson(registration.getClientId(), legalDetails);
        registration.setPersonId(newPersonId);

        registration.setState(EBillingRegistrationState.BRS_CREATING_CONTRACT);
        context.scheduleEvent(TBillingRegistrationCreateContract.newBuilder().build());
    }

    @HandleEvent
    public void createContract(TBillingRegistrationCreateContract event, MessagingContext<BillingRegistration> context) {
        BillingRegistration registration = context.getWorkflowEntity();
        checkState(registration, EBillingRegistrationState.BRS_CREATING_CONTRACT, event);

        BillingCreateContractResponse contractIds =
                billingService.createAcceptedOffer(registration.getClientId(), registration.getPersonId());
        registration.setContractId(contractIds.getContractId());
        registration.setExternalContractId(contractIds.getExternalId());
        registration.setRegisteredAt(Instant.now());
        log.info("Partner registration has completed: client id {}, person id {}, contract id {}",
                registration.getClientId(), registration.getPersonId(), registration.getContractId());

        registration.setState(EBillingRegistrationState.BRS_REGISTERED);
        context.scheduleExternalEvent(registration.getParentWorkflowId(), TBillingRegistrationFinish.newBuilder()
                .setClientId(registration.getClientId())
                .setPersonId(registration.getPersonId())
                .setContractId(registration.getContractId())
                .setExternalContractId(registration.getExternalContractId())
                .setRegisteredAt(ProtoUtils.fromInstant(registration.getRegisteredAt()))
                .build());
    }

    @HandleEvent
    public void startUpdate(TBillingRegistrationStartUpdate event, MessagingContext<BillingRegistration> context) {
        BillingRegistration registration = context.getWorkflowEntity();
        checkState(registration, EBillingRegistrationState.BRS_REGISTERED, event);

        log.info("Starting update of partner legal data; partner '{}'", registration.getLegalDetails().getLegalName());
        // attaching to the new caller (legal details step)
        registration.setParentWorkflowId(UUID.fromString(event.getNewParentWorkflowId()));

        registration.setState(EBillingRegistrationState.BRS_UPDATING_CLIENT);
        context.scheduleEvent(TBillingRegistrationUpdateClient.newBuilder().build());
    }

    @HandleEvent
    public void updateClient(TBillingRegistrationUpdateClient event, MessagingContext<BillingRegistration> context) {
        BillingRegistration registration = context.getWorkflowEntity();
        checkState(registration, EBillingRegistrationState.BRS_UPDATING_CLIENT, event);

        billingService.updateClient(registration.getLegalDetails());

        registration.setState(EBillingRegistrationState.BRS_UPDATING_PERSON);
        context.scheduleEvent(TBillingRegistrationUpdatePerson.newBuilder().build());
    }

    @HandleEvent
    public void updatePerson(TBillingRegistrationUpdatePerson event, MessagingContext<BillingRegistration> context) {
        BillingRegistration registration = context.getWorkflowEntity();
        checkState(registration, EBillingRegistrationState.BRS_UPDATING_PERSON, event);

        billingService.updatePerson(registration.getLegalDetails());

        registration.setState(EBillingRegistrationState.BRS_REGISTERED);
        context.scheduleExternalEvent(registration.getParentWorkflowId(),
                TBillingRegistrationUpdateFinished.newBuilder().build());
    }

    private void checkState(BillingRegistration step, EBillingRegistrationState expectedState, Message event) {
        if (step.getState() != expectedState) {
            throw new IllegalArgumentException(String.format(
                    "Got a %s message while in the %s state instead of %s",
                    event.getClass().getName(), step.getState(), expectedState
            ));
        }
    }
}
