package ru.yandex.travel.hotels.administrator.workflow.hotelconnection.handlers;

import java.util.UUID;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.administrator.entity.CallGeoSearchStep;
import ru.yandex.travel.hotels.administrator.entity.HotelConnection;
import ru.yandex.travel.hotels.administrator.entity.KnownWorkflow;
import ru.yandex.travel.hotels.administrator.repository.ConnectionStepRepository;
import ru.yandex.travel.hotels.administrator.service.HotelConnectionService;
import ru.yandex.travel.hotels.administrator.service.StarTrekService;
import ru.yandex.travel.hotels.administrator.service.UpdateResult;
import ru.yandex.travel.hotels.administrator.service.partners.PartnerServiceProvider;
import ru.yandex.travel.hotels.administrator.workflow.proto.EConnectionStepState;
import ru.yandex.travel.hotels.administrator.workflow.proto.EHotelConnectionState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TActualizeHotelConnection;
import ru.yandex.travel.hotels.administrator.workflow.proto.TCallGeoSearchFinish;
import ru.yandex.travel.hotels.administrator.workflow.proto.TCallGeoSearchStart;
import ru.yandex.travel.hotels.administrator.workflow.proto.TLegalDetailsUpdated;
import ru.yandex.travel.hotels.administrator.workflow.proto.TPublishingStart;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Slf4j
@IgnoreEvents(types = {TLegalDetailsUpdated.class})
public class NewStateHandler extends BaseHotelConnectionHandler {

    private final StarTrekService startrekService;

    private final HotelConnectionService hotelConnectionService;

    private final WorkflowRepository workflowRepository;

    private final ConnectionStepRepository connectionStepRepository;

    public NewStateHandler(PartnerServiceProvider partnerServiceProvider, StarTrekService startrekService,
                           HotelConnectionService hotelConnectionService,
                           WorkflowRepository workflowRepository, ConnectionStepRepository connectionStepRepository) {
        super(partnerServiceProvider);
        this.startrekService = startrekService;
        this.hotelConnectionService = hotelConnectionService;
        this.workflowRepository = workflowRepository;
        this.connectionStepRepository = connectionStepRepository;
    }

    @HandleEvent
    public void handleActualizeHotelConnection(TActualizeHotelConnection event,
                                               StateContext<EHotelConnectionState, HotelConnection> context) {
        HotelConnection hotelConnection = context.getWorkflowEntity();
        if (hotelConnection.getStTicket() == null) {
            hotelConnection.setStTicket(startrekService.createHotelConnectionTicket(hotelConnection));
        }
        if (!hotelConnection.isGeoSearchCalled()) {
            hotelConnection.setGeoSearchCalled(true);
            CallGeoSearchStep geoSearchStep = new CallGeoSearchStep();
            geoSearchStep.setId(UUID.randomUUID());
            geoSearchStep.setPartnerId(hotelConnection.getPartnerId());
            geoSearchStep.setHotelConnection(hotelConnection);
            geoSearchStep.setState(EConnectionStepState.CSS_NEW);
            geoSearchStep.setHotelConnectionStTicket(hotelConnection.getStTicket());
            geoSearchStep.setParentWorkflowId(hotelConnection.getWorkflow().getId());
            geoSearchStep.setHotelCode(hotelConnection.getHotelCode());

            Workflow geoSearchStepWorkflow = Workflow.createWorkflowForEntity(geoSearchStep,
                    KnownWorkflow.GENERIC_SUPERVISOR.getUuid());
            workflowRepository.saveAndFlush(geoSearchStepWorkflow);
            connectionStepRepository.saveAndFlush(geoSearchStep);
            context.scheduleExternalEvent(geoSearchStepWorkflow.getId(), TCallGeoSearchStart.newBuilder().build());
        } else if (hotelConnection.getHotelName() != null && hotelConnection.getPermalink() != null) {
            //if information from geosearch has been received
            UpdateResult updateResult = hotelConnectionService.actualizeHotelConnection(hotelConnection, context);
            if (updateResult.getType().equals(UpdateResult.ResultType.VALIDATION_REQUIRED)) {
                context.setState(EHotelConnectionState.CS_MANUAL_VERIFICATION);
                startrekService.createManualVerificationTicket(updateResult);
            } else {
                context.setState(EHotelConnectionState.CS_PUBLISHING);
                context.scheduleEvent(TPublishingStart.newBuilder().build());
            }
        }
    }

    @HandleEvent
    public void handleGeoSearchCallFinish(TCallGeoSearchFinish event,
                                          StateContext<EHotelConnectionState, HotelConnection> context) {
        HotelConnection hotelConnection = context.getWorkflowEntity();
        hotelConnection.setHotelName(event.getHotelName());
        hotelConnection.setPermalink(event.getPermalink());
        startrekService.updateHotelConnectionTicket(hotelConnection);
        context.scheduleEvent(TActualizeHotelConnection.newBuilder().build());
    }
}
