package ru.yandex.travel.hotels.administrator.workflow.hotelconnection.handlers;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.administrator.entity.HotelConnection;
import ru.yandex.travel.hotels.administrator.service.HotelConnectionService;
import ru.yandex.travel.hotels.administrator.service.StarTrekService;
import ru.yandex.travel.hotels.administrator.service.UpdateResult;
import ru.yandex.travel.hotels.administrator.service.partners.PartnerServiceProvider;
import ru.yandex.travel.hotels.administrator.workflow.proto.EHotelConnectionState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TActualizeHotelConnection;
import ru.yandex.travel.hotels.administrator.workflow.proto.TLegalDetailsRegistered;
import ru.yandex.travel.hotels.administrator.workflow.proto.TLegalDetailsUpdated;
import ru.yandex.travel.hotels.administrator.workflow.proto.TManualVerificationStart;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@IgnoreEvents(types = {TLegalDetailsRegistered.class})
public class PublishedStateHandler extends BaseHotelConnectionHandler {

    private final HotelConnectionService hotelConnectionService;

    private final StarTrekService starTrekService;

    public PublishedStateHandler(HotelConnectionService hotelConnectionService,
                                 PartnerServiceProvider partnerServiceProvider, StarTrekService starTrekService) {
        super(partnerServiceProvider);
        this.hotelConnectionService = hotelConnectionService;
        this.starTrekService = starTrekService;
    }

    @HandleEvent
    public void handleActualizeHotelConnection(TActualizeHotelConnection event,
                                               StateContext<EHotelConnectionState, HotelConnection> context) {
        HotelConnection hotelConnection = context.getWorkflowEntity();
        UpdateResult updateResult = hotelConnectionService.actualizeHotelConnection(hotelConnection, context);
        if (updateResult.getType().equals(UpdateResult.ResultType.VALIDATION_REQUIRED)) {
            context.setState(EHotelConnectionState.CS_MANUAL_VERIFICATION);
            starTrekService.createManualVerificationTicket(updateResult);
            context.scheduleEvent(TManualVerificationStart.newBuilder().build());
        } else {
            //Notify hotel is transferred to state "Published"
            partnerServiceProvider.getPartnerService(hotelConnection.getPartnerId()).notifyHotelStatusChanged(hotelConnection.getHotelCode());
        }
    }

    @HandleEvent
    public void handleLegalDetailsUpdated(TLegalDetailsUpdated event,
                                          StateContext<EHotelConnectionState, HotelConnection> context) {
        starTrekService.updateHotelConnectionTicket(context.getWorkflowEntity());
    }

}
