package ru.yandex.travel.hotels.administrator.workflow.hotelconnection.handlers;

import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.util.UUID;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.administrator.configuration.HotelConnectionProperties;
import ru.yandex.travel.hotels.administrator.entity.Commission;
import ru.yandex.travel.hotels.administrator.entity.HotelConnection;
import ru.yandex.travel.hotels.administrator.entity.KnownWorkflow;
import ru.yandex.travel.hotels.administrator.entity.VerifyClusteringStep;
import ru.yandex.travel.hotels.administrator.repository.CommissionRepository;
import ru.yandex.travel.hotels.administrator.repository.ConnectionStepRepository;
import ru.yandex.travel.hotels.administrator.service.StarTrekService;
import ru.yandex.travel.hotels.administrator.service.partners.PartnerServiceProvider;
import ru.yandex.travel.hotels.administrator.workflow.proto.EConnectionStepState;
import ru.yandex.travel.hotels.administrator.workflow.proto.EHotelConnectionState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TActualizeHotelConnection;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBindHotelToLegalDetails;
import ru.yandex.travel.hotels.administrator.workflow.proto.TLegalDetailsRegistered;
import ru.yandex.travel.hotels.administrator.workflow.proto.TLegalDetailsUpdated;
import ru.yandex.travel.hotels.administrator.workflow.proto.TPublishingStart;
import ru.yandex.travel.hotels.administrator.workflow.proto.TVerifyClusteringFinish;
import ru.yandex.travel.hotels.administrator.workflow.proto.TVerifyClusteringStart;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Slf4j
@IgnoreEvents(types = {TActualizeHotelConnection.class, TLegalDetailsUpdated.class})
public class PublishingStateHandler extends BaseHotelConnectionHandler {

    private final WorkflowRepository workflowRepository;

    private final ConnectionStepRepository connectionStepRepository;

    private final CommissionRepository commissionRepository;

    private final StarTrekService starTrekService;

    private final HotelConnectionProperties hotelConnectionProperties;

    public PublishingStateHandler(WorkflowRepository workflowRepository,
                                  ConnectionStepRepository connectionStepRepository, CommissionRepository commissionRepository, StarTrekService starTrekService, PartnerServiceProvider partnerServiceProvider, HotelConnectionProperties hotelConnectionProperties) {
        super(partnerServiceProvider);
        this.workflowRepository = workflowRepository;
        this.connectionStepRepository = connectionStepRepository;
        this.commissionRepository = commissionRepository;
        this.starTrekService = starTrekService;
        this.hotelConnectionProperties = hotelConnectionProperties;
    }

    @HandleEvent
    public void handlePublishingStart(TPublishingStart event,
                                      StateContext<EHotelConnectionState, HotelConnection> context) {
        HotelConnection hotelConnection = context.getWorkflowEntity();

        VerifyClusteringStep verifyClusteringStep = new VerifyClusteringStep();
        verifyClusteringStep.setId(UUID.randomUUID());
        verifyClusteringStep.setPartnerId(hotelConnection.getPartnerId());
        verifyClusteringStep.setHotelConnection(hotelConnection);
        verifyClusteringStep.setState(EConnectionStepState.CSS_NEW);
        verifyClusteringStep.setHotelConnectionStTicket(hotelConnection.getStTicket());
        verifyClusteringStep.setParentWorkflowId(hotelConnection.getWorkflow().getId());
        verifyClusteringStep.setHotelName(hotelConnection.getHotelName());
        verifyClusteringStep.setHotelCode(hotelConnection.getHotelCode());
        verifyClusteringStep.setPermalink(hotelConnection.getPermalink());

        Workflow clusteringWorkflow = Workflow.createWorkflowForEntity(verifyClusteringStep,
                KnownWorkflow.GENERIC_SUPERVISOR.getUuid());
        workflowRepository.saveAndFlush(clusteringWorkflow);
        connectionStepRepository.saveAndFlush(verifyClusteringStep);
        context.scheduleExternalEvent(clusteringWorkflow.getId(), TVerifyClusteringStart.newBuilder().build());

        context.scheduleExternalEvent(hotelConnection.getLegalDetails().getWorkflow().getId(),
                TBindHotelToLegalDetails
                        .newBuilder()
                        .setHotelWorkflowId(hotelConnection.getWorkflow().getId().toString())
                        .setHotelConnectionStTicket(hotelConnection.getStTicket())
                        .build());

        //Notify hotel is transferred to state "Publishing"
        partnerServiceProvider.getPartnerService(hotelConnection.getPartnerId()).notifyHotelStatusChanged(hotelConnection.getHotelCode());
    }

    @HandleEvent
    public void handleVerifyClusteringFinish(TVerifyClusteringFinish event,
                                          StateContext<EHotelConnectionState, HotelConnection> context) {
        HotelConnection hotelConnection = context.getWorkflowEntity();
        hotelConnection.setClusteringVerified(true);
        starTrekService.commentClusteringVerified(hotelConnection.getStTicket());
        hotelConnection.setReadyForOffercache(true);
        publishIfReady(hotelConnection, context);
    }

    @HandleEvent
    public void handleLegalDetailsUpdated(TLegalDetailsRegistered event,
                                          StateContext<EHotelConnectionState, HotelConnection> context) {
        HotelConnection hotelConnection = context.getWorkflowEntity();
        if (hotelConnection.getLegalDetails().getId().equals(UUID.fromString(event.getLegalDetailsID()))) {
            if (!hotelConnection.isLegalDetailsRegistered()) {
                hotelConnection.setLegalDetailsRegistered(true);
                starTrekService.commentLegalDetailsReady(hotelConnection.getStTicket());
            }
            if (hotelConnectionProperties.isCreateAgreements()) {
                for (HotelConnectionProperties.CommissionSchema commissionSchema : hotelConnectionProperties.getCommissions()) {
                    Commission commission = new Commission();
                    commission.setHotelConnection(hotelConnection);
                    commission.setOrderConfirmedRate(commissionSchema.getConfirmedRate());
                    commission.setOrderRefundedRate(commissionSchema.getRefundedRate());
                    if (commissionSchema.getStartDate() != null) {
                        commission.setAgreementStartDate(commissionSchema.getStartDate().minusHours(3).toInstant(ZoneOffset.UTC));
                    } else {
                        commission.setAgreementStartDate(LocalDate.now().atStartOfDay().minusHours(3).toInstant(ZoneOffset.UTC));
                    }
                    if (commissionSchema.getEndDate() != null) {
                        commission.setAgreementEndDate(commissionSchema.getEndDate().minusHours(3).toInstant(ZoneOffset.UTC));
                    }
                    commission.setEnabled(true);
                    commission.setPriority(commissionSchema.getPriority());
                    commissionRepository.saveAndFlush(commission);
                }
            } else {
                log.info("Commissions are not created for hotel with code {}, because of disabled property hotel-connection.create-agreements",
                        hotelConnection.getHotelCode());
            }
            hotelConnection.setReadyForOrchestrator(true);
            publishIfReady(hotelConnection, context);
        }
    }

    private void publishIfReady(HotelConnection hotelConnection, StateContext<EHotelConnectionState, HotelConnection> context) {
        if (readyToPublish(hotelConnection)) {
            starTrekService.closeHotelConnectionTicket(hotelConnection.getStTicket());
            hotelConnection.setState(EHotelConnectionState.CS_PUBLISHED);
            if (hotelConnection.getFirstPublishAt() == null) {
                hotelConnection.setFirstPublishAt(Instant.now());
            }
            context.scheduleEvent(TActualizeHotelConnection.newBuilder().build());
        }
    }

    private boolean readyToPublish(HotelConnection hotelConnection) {
        return hotelConnection.isLegalDetailsRegistered() && hotelConnection.isClusteringVerified();
    }
}
