package ru.yandex.travel.hotels.administrator.workflow.hotelconnectionupdate.handlers;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.administrator.entity.HotelConnection;
import ru.yandex.travel.hotels.administrator.entity.HotelConnectionUpdate;
import ru.yandex.travel.hotels.administrator.service.ApplyUpdateResult;
import ru.yandex.travel.hotels.administrator.service.HotelConnectionService;
import ru.yandex.travel.hotels.administrator.service.StarTrekService;
import ru.yandex.travel.hotels.administrator.workflow.proto.EHotelConnectionState;
import ru.yandex.travel.hotels.administrator.workflow.proto.EHotelConnectionUpdateState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TActualizeHotelConnection;
import ru.yandex.travel.hotels.administrator.workflow.proto.TApplyConnectionUpdate;
import ru.yandex.travel.hotels.administrator.workflow.proto.TNotifyLegalDetailsPublished;
import ru.yandex.travel.hotels.administrator.workflow.proto.TPublishingStart;
import ru.yandex.travel.hotels.administrator.workflow.proto.TRejectConnectionUpdate;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class NewStateHandler extends AnnotatedStatefulWorkflowEventHandler<EHotelConnectionUpdateState, HotelConnectionUpdate> {

    private final StarTrekService starTrekService;

    private final HotelConnectionService hotelConnectionService;

    @HandleEvent
    public void handleApplyUpdate(TApplyConnectionUpdate event,
                                               StateContext<EHotelConnectionUpdateState, HotelConnectionUpdate> context) {
        HotelConnectionUpdate hotelConnectionUpdate = context.getWorkflowEntity();
        HotelConnection hotelConnection = hotelConnectionUpdate.getHotelConnection();
        ApplyUpdateResult applyUpdateResult;
        if (hotelConnectionUpdate.getOriginalHotelConnectionState().equals(EHotelConnectionState.CS_NEW)) {
            applyUpdateResult = hotelConnectionService.applyHotelConnectionUpdate(hotelConnection,
                    hotelConnectionUpdate, context, event.getBankChangeInplaceMode());
            hotelConnection.setState(EHotelConnectionState.CS_PUBLISHING);
            context.scheduleExternalEvent(hotelConnection.getWorkflow().getId(), TPublishingStart.newBuilder().build());
        } else {
            applyUpdateResult = hotelConnectionService.applyHotelConnectionUpdate(hotelConnection,
                    hotelConnectionUpdate, context, event.getBankChangeInplaceMode());
        }

        starTrekService.commentHotelConnectionUpdateProcessing(hotelConnectionUpdate.getStTicket());
        hotelConnectionUpdate.setState(EHotelConnectionUpdateState.HCU_PROCESSING);

        //TODO: currently step of updating financial events is completely skipped
        hotelConnectionUpdate.setFinancialEventsUpdated(true);
        if (!applyUpdateResult.isLegalDetailsUpdated()) {
            //if legal details won't be updated
            context.scheduleEvent(TNotifyLegalDetailsPublished.newBuilder().build());
        }
    }

    @HandleEvent
    public void handleRejectUpdate(TRejectConnectionUpdate event,
                                   StateContext<EHotelConnectionUpdateState, HotelConnectionUpdate> context) {
        HotelConnectionUpdate hotelConnectionUpdate = context.getWorkflowEntity();
        HotelConnection hotelConnection = hotelConnectionUpdate.getHotelConnection();
        Preconditions.checkState(hotelConnection.getState().equals(EHotelConnectionState.CS_MANUAL_VERIFICATION),
                "HotelConnection had to be in manual_verification state but is " + hotelConnection.getState());
        switch (hotelConnectionUpdate.getOriginalHotelConnectionState()) {
            case CS_NEW:
            case CS_PUBLISHED:
                hotelConnection.setState(hotelConnectionUpdate.getOriginalHotelConnectionState());
                break;
            default:
                throw new IllegalStateException("Unexpected hotelConnection state when rejecting update: " + hotelConnection.getState());
        }
        context.scheduleExternalEvent(hotelConnection.getWorkflow().getId(),
                TActualizeHotelConnection.newBuilder().build());

        starTrekService.rejectConnectionUpdate(hotelConnectionUpdate.getStTicket());
        hotelConnectionUpdate.setState(EHotelConnectionUpdateState.HCU_REJECTED);
    }
}
