package ru.yandex.travel.hotels.administrator.workflow.hotelconnectionupdate.handlers;

import java.time.Instant;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.administrator.entity.HotelConnectionUpdate;
import ru.yandex.travel.hotels.administrator.service.StarTrekService;
import ru.yandex.travel.hotels.administrator.workflow.proto.EHotelConnectionState;
import ru.yandex.travel.hotels.administrator.workflow.proto.EHotelConnectionUpdateState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TActualizeHotelConnection;
import ru.yandex.travel.hotels.administrator.workflow.proto.TNotifyFinancialEventsUpdated;
import ru.yandex.travel.hotels.administrator.workflow.proto.TNotifyLegalDetailsPublished;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class ProcessingStateHandler extends AnnotatedStatefulWorkflowEventHandler<EHotelConnectionUpdateState, HotelConnectionUpdate> {

    private final StarTrekService starTrekService;

    @HandleEvent
    public void handleNotifyLegalDetailsPublished(TNotifyLegalDetailsPublished event,
                                                  StateContext<EHotelConnectionUpdateState, HotelConnectionUpdate> context) {
        HotelConnectionUpdate hotelConnectionUpdate = context.getWorkflowEntity();
        hotelConnectionUpdate.setLegalDetailsPublished(true);
        starTrekService.commentLegalDetailsHaveBeenPublished(hotelConnectionUpdate.getStTicket());
        if (hotelConnectionUpdate.getHotelConnection().getStTicket() != null &&
            hotelConnectionUpdate.getHotelConnection().getLegalDetails().getStTicket() != null) {
            starTrekService.linkTickets(hotelConnectionUpdate.getHotelConnection().getStTicket(),
                    hotelConnectionUpdate.getHotelConnection().getLegalDetails().getStTicket());
        }
        closeUpdateIfDone(hotelConnectionUpdate, context);
    }

    @HandleEvent
    public void handleNotifyFinancialEventsUpdated(TNotifyFinancialEventsUpdated event,
                                            StateContext<EHotelConnectionUpdateState, HotelConnectionUpdate> context) {
        HotelConnectionUpdate hotelConnectionUpdate = context.getWorkflowEntity();
        hotelConnectionUpdate.setFinancialEventsUpdated(true);
        starTrekService.commentFinancialEventsUpdated(hotelConnectionUpdate.getStTicket());
        closeUpdateIfDone(hotelConnectionUpdate, context);
    }

    private void closeUpdateIfDone(HotelConnectionUpdate connectionUpdate,
                                   StateContext<EHotelConnectionUpdateState, HotelConnectionUpdate> context) {
        if (connectionUpdate.isLegalDetailsPublished() && connectionUpdate.isFinancialEventsUpdated()) {
            if (!connectionUpdate.getOriginalHotelConnectionState().equals(EHotelConnectionState.CS_NEW)) {
                connectionUpdate.getHotelConnection().setState(EHotelConnectionState.CS_PUBLISHED);
                if (connectionUpdate.getHotelConnection().getFirstPublishAt() == null) {
                    connectionUpdate.getHotelConnection().setFirstPublishAt(Instant.now());
                }
            }
            starTrekService.closeHotelConnectionUpdateTicket(connectionUpdate.getStTicket(),
                    connectionUpdate.getHotelConnection().getLegalDetails().getBalanceClientId());
            connectionUpdate.setState(EHotelConnectionUpdateState.HCU_APPLIED);
            context.scheduleExternalEvent(connectionUpdate.getHotelConnection().getWorkflow().getId(),
                    TActualizeHotelConnection.newBuilder().build());
        }
    }
}
