package ru.yandex.travel.hotels.administrator.workflow.legaldetails.handlers;

import java.util.UUID;

import lombok.RequiredArgsConstructor;

import ru.yandex.travel.hotels.administrator.entity.HotelConnection;
import ru.yandex.travel.hotels.administrator.entity.KnownWorkflow;
import ru.yandex.travel.hotels.administrator.entity.LegalDetails;
import ru.yandex.travel.hotels.administrator.entity.UpdateBalanceDataStep;
import ru.yandex.travel.hotels.administrator.repository.ConnectionStepRepository;
import ru.yandex.travel.hotels.administrator.service.LegalDetailsService;
import ru.yandex.travel.hotels.administrator.service.StarTrekService;
import ru.yandex.travel.hotels.administrator.workflow.proto.ELegalDetailsState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBindHotelToLegalDetails;
import ru.yandex.travel.hotels.administrator.workflow.proto.TLegalDetailsRegistered;
import ru.yandex.travel.hotels.administrator.workflow.proto.TLegalDetailsUpdated;
import ru.yandex.travel.hotels.administrator.workflow.proto.TUpdateBalanceDataFinish;
import ru.yandex.travel.hotels.administrator.workflow.proto.TUpdateBalanceDataStart;
import ru.yandex.travel.hotels.administrator.workflow.proto.TUpdateLegalDetails;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@RequiredArgsConstructor
public class RegisteredStateHandler extends AnnotatedStatefulWorkflowEventHandler<ELegalDetailsState, LegalDetails> {

    private final StarTrekService starTrekService;

    private final LegalDetailsService legalDetailsService;

    private final ConnectionStepRepository connectionStepRepository;

    private final WorkflowRepository workflowRepository;

    @HandleEvent
    public void handleBindHotelToLegalDetails(TBindHotelToLegalDetails event,
                                              StateContext<ELegalDetailsState, LegalDetails> context) {
        LegalDetails legalDetails = context.getWorkflowEntity();
        starTrekService.linkTickets(legalDetails.getStTicket(), event.getHotelConnectionStTicket());
        context.scheduleExternalEvent(UUID.fromString(event.getHotelWorkflowId()),
                TLegalDetailsRegistered.newBuilder()
                        .setLegalDetailsID(legalDetails.getId().toString())
                        .build());
    }

    @HandleEvent
    public void updateLegalDetails(TUpdateLegalDetails event, StateContext<ELegalDetailsState, LegalDetails> context) {
        LegalDetails legalDetails = context.getWorkflowEntity();
        boolean changed = legalDetailsService.applyReadyUpdates(legalDetails);
        if (changed) {
            UpdateBalanceDataStep step = UpdateBalanceDataStep.createNewStepFor(legalDetails);
            Workflow workflow = Workflow.createWorkflowForEntity(step, KnownWorkflow.GENERIC_SUPERVISOR.getUuid());
            workflowRepository.saveAndFlush(workflow);
            connectionStepRepository.saveAndFlush(step);
            context.scheduleExternalEvent(workflow.getId(), TUpdateBalanceDataStart.newBuilder().build());
        }
    }

    @HandleEvent
    public void onBillingSynchronizationFinish(TUpdateBalanceDataFinish event,
                                               StateContext<ELegalDetailsState, LegalDetails> context) {
        LegalDetails legalDetails = context.getWorkflowEntity();
        legalDetailsService.completeBillingSynchronization(legalDetails, context);
        legalDetails.setStTicket(starTrekService.createOrUpdateLegalDetailsTicket(legalDetails));
        if (legalDetailsService.hasReadyUpdates(legalDetails)) {
            context.scheduleEvent(TUpdateLegalDetails.newBuilder().build());
        } else {
            for (HotelConnection hotelConnection: legalDetails.getHotelConnections()) {
                context.scheduleExternalEvent(hotelConnection.getWorkflow().getId(),
                        TLegalDetailsUpdated.newBuilder()
                                .build());
            }
        }
    }
}
