package ru.yandex.travel.hotels.administrator.workflow.legaldetails.handlers;

import java.util.UUID;

import lombok.RequiredArgsConstructor;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.administrator.entity.HotelConnection;
import ru.yandex.travel.hotels.administrator.entity.KnownWorkflow;
import ru.yandex.travel.hotels.administrator.entity.LegalDetails;
import ru.yandex.travel.hotels.administrator.entity.RegisterInBalanceStep;
import ru.yandex.travel.hotels.administrator.repository.ConnectionStepRepository;
import ru.yandex.travel.hotels.administrator.repository.HotelConnectionUpdateRepository;
import ru.yandex.travel.hotels.administrator.service.LegalDetailsService;
import ru.yandex.travel.hotels.administrator.service.StarTrekService;
import ru.yandex.travel.hotels.administrator.workflow.proto.EConnectionStepState;
import ru.yandex.travel.hotels.administrator.workflow.proto.ELegalDetailsState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBindHotelToLegalDetails;
import ru.yandex.travel.hotels.administrator.workflow.proto.TLegalDetailsRegistered;
import ru.yandex.travel.hotels.administrator.workflow.proto.TNotifyLegalDetailsPublished;
import ru.yandex.travel.hotels.administrator.workflow.proto.TRegisterInBalanceFinish;
import ru.yandex.travel.hotels.administrator.workflow.proto.TRegisterInBalanceStart;
import ru.yandex.travel.hotels.administrator.workflow.proto.TRegisteringStart;
import ru.yandex.travel.hotels.administrator.workflow.proto.TUpdateLegalDetails;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@IgnoreEvents(types = {
        TUpdateLegalDetails.class,
})
@RequiredArgsConstructor
public class RegisteringStateHandler extends AnnotatedStatefulWorkflowEventHandler<ELegalDetailsState, LegalDetails> {

    private final WorkflowRepository workflowRepository;

    private final ConnectionStepRepository connectionStepRepository;

    private final StarTrekService starTrekService;

    private final LegalDetailsService legalDetailsService;

    private final HotelConnectionUpdateRepository hotelConnectionUpdateRepository;

    @HandleEvent
    public void handleRegisteringStart(TRegisteringStart event,
                                       StateContext<ELegalDetailsState, LegalDetails> context) {
        LegalDetails legalDetails = context.getWorkflowEntity();

        RegisterInBalanceStep registerInBalanceStep = new RegisterInBalanceStep();
        registerInBalanceStep.setId(UUID.randomUUID());
        registerInBalanceStep.setLegalDetails(legalDetails);
        registerInBalanceStep.setState(EConnectionStepState.CSS_NEW);
        registerInBalanceStep.setLegalName(legalDetails.getLegalName());
        registerInBalanceStep.setParentWorkflowId(legalDetails.getWorkflow().getId());
        Workflow workflow = Workflow.createWorkflowForEntity(registerInBalanceStep,
                KnownWorkflow.GENERIC_SUPERVISOR.getUuid());
        workflowRepository.saveAndFlush(workflow);
        connectionStepRepository.saveAndFlush(registerInBalanceStep);
        context.scheduleExternalEvent(workflow.getId(), TRegisterInBalanceStart.newBuilder().build());
    }

    @HandleEvent
    public void handleRegisteredInBalanceFinish(TRegisterInBalanceFinish event,
                                                StateContext<ELegalDetailsState, LegalDetails> context) {
        LegalDetails legalDetails = context.getWorkflowEntity();
        legalDetails.setRegisteredInBalance(true);
        legalDetails.setBalanceClientId(event.getBalanceClientId());
        legalDetails.setBalancePersonId(event.getBalancePersonId());
        legalDetails.setBalanceContractId(event.getBalanceContractId());
        legalDetails.setBalanceExternalContractId(event.getBalanceExternalContractId());
        legalDetails.setRegisteredAt(ProtoUtils.toInstant(event.getRegisteredAt()));

        legalDetails.setState(ELegalDetailsState.DS_REGISTERED);
        for (HotelConnection hotelConnection : legalDetails.getHotelConnections()) {
            context.scheduleExternalEvent(hotelConnection.getWorkflow().getId(),
                    TLegalDetailsRegistered.newBuilder()
                            .setLegalDetailsID(legalDetails.getId().toString())
                            .build());
        }
        if (legalDetails.getHotelConnectionUpdateId() != null) {
            hotelConnectionUpdateRepository.findById(legalDetails.getHotelConnectionUpdateId()).ifPresent(connectionUpdate ->
                    context.scheduleExternalEvent(connectionUpdate.getWorkflow().getId(),
                            TNotifyLegalDetailsPublished.newBuilder().build())
            );
        }

        // making sure pending updates are applied if any have occurred during the initial registration
        if (legalDetailsService.hasReadyUpdates(legalDetails)) {
            context.scheduleEvent(TUpdateLegalDetails.newBuilder().build());
        }

        starTrekService.commentRegisteredInBalance(
                legalDetails.getStTicket(),
                event.getBalanceClientId(),
                event.getBalanceContractId());

        legalDetails.setStTicket(starTrekService.createOrUpdateLegalDetailsTicket(legalDetails));
        starTrekService.closeLegalDetailsTicket(legalDetails.getStTicket());
    }

    @HandleEvent
    public void handleBindHotelToLegalDetails(TBindHotelToLegalDetails event,
                                              StateContext<ELegalDetailsState, LegalDetails> context) {
        LegalDetails legalDetails = context.getWorkflowEntity();
        starTrekService.linkTickets(legalDetails.getStTicket(), event.getHotelConnectionStTicket());
    }
}
