package ru.yandex.travel.hotels.administrator.workflow.step.handlers;

import java.time.Instant;
import java.util.Map;
import java.util.concurrent.ExecutionException;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.hotels.administrator.cache.AltayPublishingDictionary;
import ru.yandex.travel.hotels.administrator.cache.AltaySignalsDictionary;
import ru.yandex.travel.hotels.administrator.cache.HotelFeedsDictionary;
import ru.yandex.travel.hotels.administrator.entity.CallGeoSearchStep;
import ru.yandex.travel.hotels.administrator.service.GeoHotelService;
import ru.yandex.travel.hotels.administrator.service.StarTrekService;
import ru.yandex.travel.hotels.administrator.workflow.proto.EConnectionStepState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TCallGeoSearchFinish;
import ru.yandex.travel.hotels.administrator.workflow.proto.TCallGeoSearchStart;
import ru.yandex.travel.hotels.administrator.workflow.proto.TTicketClosed;
import ru.yandex.travel.hotels.administrator.workflow.proto.TTicketOld;
import ru.yandex.travel.hotels.common.PartnerConfigService;
import ru.yandex.travel.hotels.geosearch.model.GeoHotel;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.base.AnnotatedWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Component
@RequiredArgsConstructor
@Slf4j
public class CallGeoSearchWorkflowHandler extends AnnotatedWorkflowEventHandler<CallGeoSearchStep> {

    private final GeoHotelService geoHotelService;

    private final StarTrekService starTrekService;

    private final PartnerConfigService partnerConfigService;

    private final HotelFeedsDictionary hotelFeedsDictionary;

    private final AltaySignalsDictionary altaySignalsDictionary;

    private final AltayPublishingDictionary altayPublishingDictionary;

    @HandleEvent
    public void handleCallGeoSearchStart(TCallGeoSearchStart event,
                                         MessagingContext<CallGeoSearchStep> context) {
        CallGeoSearchStep callGeoSearchStep = context.getWorkflowEntity();
        Preconditions.checkState(callGeoSearchStep.getState().equals(EConnectionStepState.CSS_NEW),
                "Unexpected step state: %s", callGeoSearchStep.getState());

        try {
            callGeoSearchStep.setLastGeoSearchCallAt(Instant.now());
            if (!tryCallGeoSearchAndFinishStep(callGeoSearchStep, context, callGeoSearchStep.getHotelCode(), callGeoSearchStep.getPartnerId())) {
                log.error("Hotel not found in geoSearch, creating manual ticket");
                handleGeoSearchProblemOnStart(callGeoSearchStep);
                commentHotelNotFound(callGeoSearchStep.getStTicket(), callGeoSearchStep.getPartnerId(), callGeoSearchStep.getHotelCode());
            }
        } catch (Exception e) {
            // call to geoSearch failed, need to create ticket and wait till it will be closed
            log.error("Call to geoSearch failed, creating manual ticket", e);
            handleGeoSearchProblemOnStart(callGeoSearchStep);
            starTrekService.commentGSCallFailed(callGeoSearchStep.getStTicket(), e);
        }
    }

    private void handleGeoSearchProblemOnStart(CallGeoSearchStep callGeoSearchStep) {
        callGeoSearchStep.setState(EConnectionStepState.CSS_WAIT_FOR_TICKET_RESOLUTION);
        if (callGeoSearchStep.getStTicket() == null) {
            callGeoSearchStep.setStTicket(starTrekService.createHotelIsMissingInGSTicket(callGeoSearchStep.getHotelCode(),
                    callGeoSearchStep.getPartnerId(), callGeoSearchStep.getHotelConnectionStTicket()));
        }
    }

    @HandleEvent
    public void handleTicketClosed(TTicketClosed event,
                                   MessagingContext<CallGeoSearchStep> context) {
        CallGeoSearchStep callGeoSearchStep = context.getWorkflowEntity();
        Preconditions.checkState(callGeoSearchStep.getState().equals(EConnectionStepState.CSS_WAIT_FOR_TICKET_RESOLUTION),
                "Unexpected step state: %s", callGeoSearchStep.getState());
        var needComment = false;
        try {
            callGeoSearchStep.setLastGeoSearchCallAt(Instant.now());
            if (tryCallGeoSearchAndFinishStep(callGeoSearchStep, context, callGeoSearchStep.getHotelCode(), callGeoSearchStep.getPartnerId())) {
                starTrekService.commentHotelAppearedInGS(callGeoSearchStep.getStTicket());
            } else {
                log.error("Hotel not found in geoSearch, reopening ticket");
                commentHotelNotFound(callGeoSearchStep.getStTicket(), callGeoSearchStep.getPartnerId(), callGeoSearchStep.getHotelCode());
                needComment = true;
            }
        } catch (Exception e) {
            // call to geoSearch failed, need to reopen ticket and wait till it will be closed once again
            log.error("Call to geoSearch failed, reopening ticket", e);
            starTrekService.commentGSCallFailed(callGeoSearchStep.getStTicket(), e);
            needComment = true;
        }
        if (needComment) {
            callGeoSearchStep.setTicketResultFetched(false);
            starTrekService.reopenTicket(callGeoSearchStep.getStTicket());
        }
    }

    @HandleEvent
    public void handleTicketOld(TTicketOld event,
                                MessagingContext<CallGeoSearchStep> context) {
        CallGeoSearchStep callGeoSearchStep = context.getWorkflowEntity();
        Preconditions.checkState(callGeoSearchStep.getState().equals(EConnectionStepState.CSS_WAIT_FOR_TICKET_RESOLUTION),
                "Unexpected step state: %s", callGeoSearchStep.getState());
        try {
            callGeoSearchStep.setLastGeoSearchCallAt(Instant.now());
            if (tryCallGeoSearchAndFinishStep(callGeoSearchStep, context, callGeoSearchStep.getHotelCode(), callGeoSearchStep.getPartnerId())) {
                starTrekService.closeAndCommentHotelAppearedInGS(callGeoSearchStep.getStTicket());
            } else {
                // hotel not found, ignoring it during old ticket auto-check
                callGeoSearchStep.setTicketResultFetched(false);
            }
        } catch (Exception e) {
            // call to geoSearch failed, ignoring it during old ticket auto-check
            callGeoSearchStep.setTicketResultFetched(false);
        }
    }

    private void commentHotelNotFound(String ticketKey, EPartnerId partnerId, String originalId) {
        var partnerConfig = partnerConfigService.getByKey(partnerId);
        var feedPath = partnerConfig.getFeedPath();
        var partnerCode = partnerConfig.getCode();
        var providerId = Map.of(
                EPartnerId.PI_TRAVELLINE, 3501497180L,
                EPartnerId.PI_BNOVO, 3501741817L
        ).get(partnerId);
        var foundInFeed = hotelFeedsDictionary.hasRecord(partnerId, originalId);
        var altaySignal = altaySignalsDictionary.getRecord(providerId, originalId);
        var altayPublishingInfos = altayPublishingDictionary.getRecords(providerId, originalId);
        var bestAltayPublishingInfo = altayPublishingInfos == null || altayPublishingInfos.isEmpty() ? null :
                altayPublishingInfos.stream().filter(x -> x.getPublishingStatus().equals("publish")).findAny()
                        .orElse(altayPublishingInfos.stream().findFirst().get());
        var altaySignalLink = String.format("https://altay.yandex-team.ru/signals/list?providers=%s%%7C%%7C*%%7C%%7C%s", providerId, originalId);
        var altaySignalYqlLink = String.format("https://yql.yandex-team.ru/?query=USE+hahn;%%0A%%0A" +
                "SELECT%%0A++raw_signal,%%0A" +
                "++source_data,%%0A" +
                "++warnings,%%0A" +
                "++unified_signal,%%0A" +
                "++reason,%%0A" +
                "++comment%%0A" +
                "FROM+`//home/sprav/import/production/full-loading-signal`%%0A" +
                "WHERE+source_data.original_id+==+\"%s\"+AND+source_data.provider_id+=+%s;", originalId, providerId);
        var feedYqlLink = String.format("https://yql.yandex-team.ru/?query=USE+hahn;%%0A%%0A" +
                "SELECT+*%%0A" +
                "FROM+RANGE(`%s/latest/parsed`)%%0A" +
                "WHERE+originalId+=+'%s';", feedPath, originalId);
        var altayMappingsYqlLink = String.format("https://yql.yandex-team.ru/?type=CLICKHOUSE&query=USE+chyt.hahn;%%0A" +
                "SELECT+*%%0A" +
                "FROM+`//home/travel/prod/general/altay_mappings/latest/partnerid_originalid_to_cluster_permalink`%%0A" +
                "WHERE+partner_name+=+'%s'+AND+`originalid`+=+'%s';", partnerCode, originalId);
        var geosearchLink = String.format("http://addrs-testing.search.yandex.net/search/stable/yandsearch?origin=travel-hotels-administrator&business_reference=%s:%s&business_show_closed=1&hr=1", partnerCode, originalId);
        var altayCompanyYqlLink = String.format("https://yql.yandex-team.ru/?query=USE+hahn;%%0A%%0A" +
                "SELECT+*%%0A" +
                "FROM+`//home/altay/db/export/current-state/snapshot/company`%%0A" +
                "WHERE+permalink+=+%s;", bestAltayPublishingInfo == null ? 0 : bestAltayPublishingInfo.getPermalink());

        starTrekService.commentGSCallHotelNotFound(ticketKey, altaySignalLink, altaySignalYqlLink, feedYqlLink,
                altayMappingsYqlLink, geosearchLink, foundInFeed, altaySignal, bestAltayPublishingInfo, altayPublishingInfos, altayCompanyYqlLink);
    }

    private boolean tryCallGeoSearchAndFinishStep(CallGeoSearchStep callGeoSearchStep,
                                                  MessagingContext<CallGeoSearchStep> context,
                                                  String hotelCode, EPartnerId partnerId) {
        try {
            GeoHotel geoHotel = geoHotelService.getHotel(partnerId, hotelCode);
            if (geoHotel == null) {
                log.warn("Hotel with id={} is not found in GeoSearch", hotelCode);
                return false;
            }
            TCallGeoSearchFinish finishMessage =  TCallGeoSearchFinish.newBuilder()
                    .setHotelName(geoHotel.getGeoObject().getName())
                    .setPermalink(geoHotel.getPermalink().asLong())
                    .build();
            callGeoSearchStep.setHotelName(finishMessage.getHotelName());
            callGeoSearchStep.setPermalink(finishMessage.getPermalink());
            callGeoSearchStep.setState(EConnectionStepState.CSS_DONE);
            context.scheduleExternalEvent(callGeoSearchStep.getParentWorkflowId(), finishMessage);
            return true;
        } catch (InterruptedException e) {
            log.error("GeoSearch call interrupted", e);
            Thread.currentThread().interrupt(); // preserved interruption status
            throw new RuntimeException(e);
        } catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }
}
