package ru.yandex.travel.hotels.administrator.workflow.step.handlers;

import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.administrator.entity.BillingRegistration;
import ru.yandex.travel.hotels.administrator.entity.KnownWorkflow;
import ru.yandex.travel.hotels.administrator.entity.LegalDetails;
import ru.yandex.travel.hotels.administrator.entity.RegisterInBalanceStep;
import ru.yandex.travel.hotels.administrator.repository.BillingRegistrationRepository;
import ru.yandex.travel.hotels.administrator.workflow.proto.EBillingRegistrationState;
import ru.yandex.travel.hotels.administrator.workflow.proto.EConnectionStepState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBillingRegistrationFinish;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBillingRegistrationStart;
import ru.yandex.travel.hotels.administrator.workflow.proto.TRegisterInBalanceFinish;
import ru.yandex.travel.hotels.administrator.workflow.proto.TRegisterInBalanceStart;
import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.base.AnnotatedWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Component
@RequiredArgsConstructor
@Slf4j
public class RegisterInBalanceWorkflowHandler extends AnnotatedWorkflowEventHandler<RegisterInBalanceStep> {

    private final WorkflowRepository workflowRepository;
    private final BillingRegistrationRepository billingRegistrationRepository;

    @HandleEvent
    public void handleStartRegister(TRegisterInBalanceStart event,
                                    MessagingContext<RegisterInBalanceStep> context) {
        RegisterInBalanceStep registerInBalanceStep = context.getWorkflowEntity();
        Preconditions.checkState(registerInBalanceStep.getState().equals(EConnectionStepState.CSS_NEW),
                "Unexpected step state: %s", registerInBalanceStep.getState());
        LegalDetails legalDetails = registerInBalanceStep.getLegalDetails();
        log.info("Starting a new workflow for Legal Details registration in Billing: {}", legalDetails.getLegalName());

        BillingRegistration registration = BillingRegistration.builder()
                .id(UUID.randomUUID())
                .legalDetails(legalDetails)
                .state(EBillingRegistrationState.BRS_NEW)
                .parentWorkflowId(context.getWorkflowId())
                .build();
        Workflow billingRegistrationWorkflow = Workflow.createWorkflowForEntity(registration,
                KnownWorkflow.GENERIC_SUPERVISOR.getUuid());

        context.scheduleExternalEvent(billingRegistrationWorkflow.getId(), TBillingRegistrationStart.newBuilder().build());
        registerInBalanceStep.setState(EConnectionStepState.CSS_STARTED);
        workflowRepository.saveAndFlush(billingRegistrationWorkflow);
        billingRegistrationRepository.saveAndFlush(registration);
    }

    @HandleEvent
    public void handleBillingRegistrationFinish(TBillingRegistrationFinish event,
                                                MessagingContext<RegisterInBalanceStep> context) {
        RegisterInBalanceStep registerInBalanceStep = context.getWorkflowEntity();
        Preconditions.checkState(registerInBalanceStep.getState().equals(EConnectionStepState.CSS_STARTED),
                "Unexpected step state: %s", registerInBalanceStep.getState());
        Preconditions.checkArgument(event.getClientId() != 0, "There should be a clientId");
        Preconditions.checkArgument(event.getContractId() != 0, "There should be a contractId");

        registerInBalanceStep.setState(EConnectionStepState.CSS_DONE);
        registerInBalanceStep.setBalanceClientId(event.getClientId());
        registerInBalanceStep.setBalanceContractId(event.getContractId());
        registerInBalanceStep.setRegisteredAt(ProtoUtils.toInstant(event.getRegisteredAt()));
        context.scheduleExternalEvent(registerInBalanceStep.getParentWorkflowId(),
                TRegisterInBalanceFinish.newBuilder()
                        .setBalanceClientId(event.getClientId())
                        .setBalancePersonId(event.getPersonId())
                        .setBalanceContractId(event.getContractId())
                        .setBalanceExternalContractId(event.getExternalContractId())
                        .setRegisteredAt(event.getRegisteredAt())
                        .build());
    }

}
