package ru.yandex.travel.hotels.administrator.workflow.step.handlers;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.hotels.administrator.entity.BillingRegistration;
import ru.yandex.travel.hotels.administrator.entity.LegalDetails;
import ru.yandex.travel.hotels.administrator.entity.UpdateBalanceDataStep;
import ru.yandex.travel.hotels.administrator.repository.BillingRegistrationRepository;
import ru.yandex.travel.hotels.administrator.workflow.proto.EConnectionStepState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBillingRegistrationStartUpdate;
import ru.yandex.travel.hotels.administrator.workflow.proto.TBillingRegistrationUpdateFinished;
import ru.yandex.travel.hotels.administrator.workflow.proto.TUpdateBalanceDataFinish;
import ru.yandex.travel.hotels.administrator.workflow.proto.TUpdateBalanceDataStart;
import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.base.AnnotatedWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Component
@RequiredArgsConstructor
@Slf4j
public class UpdateBalanceDataWorkflowHandler extends AnnotatedWorkflowEventHandler<UpdateBalanceDataStep> {
    private final BillingRegistrationRepository billingRegistrationRepository;

    @HandleEvent
    public void onStart(TUpdateBalanceDataStart event, MessagingContext<UpdateBalanceDataStep> context) {
        UpdateBalanceDataStep step = context.getWorkflowEntity();
        Preconditions.checkState(step.getState().equals(EConnectionStepState.CSS_NEW),
                "Unexpected step state: %s", step.getState());
        LegalDetails legalDetails = step.getLegalDetails();
        log.info("Starting an update of Legal Details data in Billing: {}", legalDetails.getLegalName());

        BillingRegistration registration = billingRegistrationRepository.findByLegalDetails(legalDetails);
        Preconditions.checkNotNull(registration, "No BillingRegistration for legal details %s", legalDetails.getId());

        context.scheduleExternalEvent(registration.getWorkflow().getId(),
                TBillingRegistrationStartUpdate.newBuilder()
                        .setNewParentWorkflowId(context.getWorkflowId().toString())
                        .build());
        step.setState(EConnectionStepState.CSS_STARTED);
    }

    @HandleEvent
    public void onFinish(TBillingRegistrationUpdateFinished event, MessagingContext<UpdateBalanceDataStep> context) {
        UpdateBalanceDataStep step = context.getWorkflowEntity();
        Preconditions.checkState(step.getState().equals(EConnectionStepState.CSS_STARTED),
                "Unexpected step state: %s", step.getState());

        step.setState(EConnectionStepState.CSS_DONE);
        context.scheduleExternalEvent(step.getParentWorkflowId(), TUpdateBalanceDataFinish.newBuilder().build());
    }
}
