package ru.yandex.travel.hotels.administrator.workflow.step.handlers;

import java.util.concurrent.ExecutionException;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.hotels.administrator.cache.HotelClusteringDictionary;
import ru.yandex.travel.hotels.administrator.entity.VerifyClusteringStep;
import ru.yandex.travel.hotels.administrator.service.ClusterizationService;
import ru.yandex.travel.hotels.administrator.service.GeoHotelService;
import ru.yandex.travel.hotels.administrator.service.StarTrekService;
import ru.yandex.travel.hotels.administrator.workflow.proto.EConnectionStepState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TClusterizationFinished;
import ru.yandex.travel.hotels.administrator.workflow.proto.TVerifyClusteringFinish;
import ru.yandex.travel.hotels.administrator.workflow.proto.TVerifyClusteringStart;
import ru.yandex.travel.hotels.geosearch.model.GeoHotel;
import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.base.AnnotatedWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@Component
@RequiredArgsConstructor
public class VerifyClusteringWorkflowHandler extends AnnotatedWorkflowEventHandler<VerifyClusteringStep> {

    private final GeoHotelService geoHotelService;
    private final HotelClusteringDictionary hotelClusteringDictionary;
    private final ClusterizationService clusterizationService;
    private final StarTrekService starTrekService;

    @HandleEvent
    public void handleVerifyClusteringStart(
            TVerifyClusteringStart event,
            MessagingContext<VerifyClusteringStep> context
    ) throws ExecutionException, InterruptedException {
        VerifyClusteringStep verifyClusteringStep = context.getWorkflowEntity();
        Preconditions.checkState(verifyClusteringStep.getState().equals(EConnectionStepState.CSS_NEW),
                "Unexpected step state: %s", verifyClusteringStep.getState());
        if (hotelClusteringDictionary.isHotelClusteringVerified(
                verifyClusteringStep.getPartnerId(),
                verifyClusteringStep.getHotelCode())
        ) {
            log.info("Clustering for hotel with code {} is already verified",
                    verifyClusteringStep.getHotelCode());
            verifyClusteringStep.setState(EConnectionStepState.CSS_DONE);
            context.scheduleExternalEvent(verifyClusteringStep.getParentWorkflowId(),
                    TVerifyClusteringFinish.newBuilder()
                            .build());
        } else {
            GeoHotel geoHotel = geoHotelService.getHotel(
                    verifyClusteringStep.getPartnerId(),
                    verifyClusteringStep.getHotelCode()
            );
            Preconditions.checkNotNull(
                    geoHotel,
                    String.format("Hotel with id=%s is not found in GeoSearch", verifyClusteringStep.getHotelCode())
            );
            clusterizationService.requestClusterization(
                    geoHotel.getPermalink().asLong(),
                    geoHotel.getGeoObject().getName(),
                    verifyClusteringStep.getHotelConnectionStTicket()
            );
            verifyClusteringStep.setState(EConnectionStepState.CSS_WAIT_FOR_CLUSTERIZATION);
            starTrekService.commentClusteringStarted(verifyClusteringStep.getHotelConnectionStTicket());
        }
    }

    @HandleEvent
    public void handleClusterizationFinished(TClusterizationFinished event,
                                   MessagingContext<VerifyClusteringStep> context) {
        VerifyClusteringStep verifyClusteringStep = context.getWorkflowEntity();
        // EConnectionStepState.CSS_WAIT_FOR_CLUSTERIZATION is left for migration period only,
        // new steps will get will get the message only in the CSS_COMMIT_CLUSTERIZATION state
        // todo(tlg-13): remove the old enum comparison after the code realease
        // and all pre-release TClusterizationFinished are processed
        Preconditions.checkState(verifyClusteringStep.getState().equals(EConnectionStepState.CSS_WAIT_FOR_CLUSTERIZATION)
                        || verifyClusteringStep.getState().equals(EConnectionStepState.CSS_COMMIT_CLUSTERIZATION),
                "Unexpected step state: %s", verifyClusteringStep.getState());
        verifyClusteringStep.setState(EConnectionStepState.CSS_DONE);
        context.scheduleExternalEvent(verifyClusteringStep.getParentWorkflowId(),
                TVerifyClusteringFinish.newBuilder()
                        .build());
    }

}
