package ru.yandex.travel.hotels.administrator.workflow.step.job;

import java.util.Collection;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.travel.hotels.administrator.service.ConnectionStepService;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;

@Configuration
@RequiredArgsConstructor
@ConditionalOnProperty(value = "ticket-task-processor.enabled", havingValue = "true")
@EnableConfigurationProperties({TicketTaskProcessorProperties.class, ClusterizationTaskProcessorProperties.class})
public class TaskProcessorsConfiguration {

    private final ConnectionStepService connectionStepService;

    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;

    private final TicketTaskProcessorProperties ticketTaskProcessorProperties;

    private final ClusterizationTaskProcessorProperties clusterizationTaskProcessorProperties;

    @Bean
    public TaskProcessor<UUID> ticketAwaitingResolutionTaskProcessor() {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return connectionStepService.getUnprocessedStartedStepIdsWithClosedTickets(getLockedTaskKeys(), maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return connectionStepService.countUnprocessedStartedStepsWithClosedTickets(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("TicketAwaitingResolution");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, connectionStepService::processStTicketClosedForStep,
                forcedRollbackTxManagerWrapper, txDefinition,
                ticketTaskProcessorProperties);
    }

    @Bean
    public TaskProcessor<UUID> clusterizationTaskProcessor() {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return connectionStepService.getUnprocessedClusterizationIds(getLockedTaskKeys(), maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return connectionStepService.countUnprocessedClusterizationIds(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("Clusterization");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, connectionStepService::processClusterizationFinished,
                forcedRollbackTxManagerWrapper, txDefinition,
                clusterizationTaskProcessorProperties);
    }
}
