package ru.yandex.travel.hotels.administrator.workflow.supervisor;

import java.util.UUID;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.administrator.configuration.StarTrekConfigurationProperties;
import ru.yandex.travel.hotels.administrator.entity.BillingRegistration;
import ru.yandex.travel.hotels.administrator.entity.HotelConnection;
import ru.yandex.travel.hotels.administrator.entity.HotelConnectionUpdate;
import ru.yandex.travel.hotels.administrator.entity.LegalDetails;
import ru.yandex.travel.hotels.administrator.entity.Ticket;
import ru.yandex.travel.hotels.administrator.entity.WorkflowEntityType;
import ru.yandex.travel.hotels.administrator.repository.BillingRegistrationRepository;
import ru.yandex.travel.hotels.administrator.repository.HotelConnectionRepository;
import ru.yandex.travel.hotels.administrator.repository.HotelConnectionUpdateRepository;
import ru.yandex.travel.hotels.administrator.repository.LegalDetailsRepository;
import ru.yandex.travel.hotels.administrator.repository.TicketRepository;
import ru.yandex.travel.hotels.administrator.service.Meters;
import ru.yandex.travel.hotels.administrator.service.TemplateService;
import ru.yandex.travel.hotels.administrator.workflow.proto.ETicketState;
import ru.yandex.travel.hotels.administrator.workflow.proto.TCreateIssue;
import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.TWorkflowCrashed;
import ru.yandex.travel.workflow.base.AnnotatedWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEvent;
import ru.yandex.travel.workflow.repository.WorkflowEventRepository;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Slf4j
public class GenericSupervisorWorkflowEventHandler extends AnnotatedWorkflowEventHandler<Void> {

    private static final String ERROR_COUNTER_TAG = "GenericSupervisor";

    private final TicketRepository ticketRepository;
    private final WorkflowRepository workflowRepository;
    private final WorkflowEventRepository eventRepository;
    private final Meters meters;
    private final StarTrekConfigurationProperties properties;
    private final TemplateService templateService;
    private final HotelConnectionRepository hotelConnectionRepository;
    private final HotelConnectionUpdateRepository hotelConnectionUpdateRepository;
    private final LegalDetailsRepository legalDetailsRepository;
    private final BillingRegistrationRepository billingRegistrationRepository;

    public GenericSupervisorWorkflowEventHandler(TicketRepository ticketRepository,
                                                 WorkflowRepository workflowRepository,
                                                 WorkflowEventRepository eventRepository, Meters meters,
                                                 StarTrekConfigurationProperties properties,
                                                 TemplateService templateService,
                                                 HotelConnectionRepository hotelConnectionRepository,
                                                 HotelConnectionUpdateRepository hotelConnectionUpdateRepository,
                                                 LegalDetailsRepository legalDetailsRepository,
                                                 BillingRegistrationRepository billingRegistrationRepository) {
        this.ticketRepository = ticketRepository;
        this.workflowRepository = workflowRepository;
        this.eventRepository = eventRepository;
        this.meters = meters;
        this.properties = properties;
        this.templateService = templateService;
        this.hotelConnectionRepository = hotelConnectionRepository;
        this.hotelConnectionUpdateRepository = hotelConnectionUpdateRepository;
        this.legalDetailsRepository = legalDetailsRepository;
        this.billingRegistrationRepository = billingRegistrationRepository;
        this.meters.initCounter(ERROR_COUNTER_TAG);
    }

    @HandleEvent
    public void handleWorkflowCrashedEvent(TWorkflowCrashed event, MessagingContext<Void> context) {
        String workflowId = event.getWorkflowId();
        try {
            String entityType = event.getEntityType();
            String entityId = event.getEntityId();
            long eventId = event.getEventId();
            Workflow workflow = workflowRepository.getOne(UUID.fromString(workflowId));
            WorkflowEvent failedEvent = eventRepository.getOne(eventId);
            String exception = SupervisorUtils.extractException(workflow);
            String linkedTicket = null;
            if (entityType.equals(WorkflowEntityType.HOTEL_CONNECTION.getDiscriminatorValue())) {
                HotelConnection hotelConnection = hotelConnectionRepository.getOne(UUID.fromString(entityId));
                linkedTicket = hotelConnection.getStTicket();
            } else if (entityType.equals(WorkflowEntityType.LEGAL_DETAILS.getDiscriminatorValue())) {
                LegalDetails legalDetails = legalDetailsRepository.getOne(UUID.fromString(entityId));
                linkedTicket = legalDetails.getStTicket();
            } else if (entityType.equals(WorkflowEntityType.BILLING_REGISTRATION.getDiscriminatorValue())) {
                BillingRegistration billingRegistration = billingRegistrationRepository.getOne(UUID.fromString(entityId));
                if (billingRegistration.getLegalDetails() != null) {
                    linkedTicket = billingRegistration.getLegalDetails().getStTicket();
                }
            } else if (entityType.equals(WorkflowEntityType.HOTEL_CONNECTION_UPDATE.getDiscriminatorValue())) {
                HotelConnectionUpdate hotelConnectionUpdate = hotelConnectionUpdateRepository.getOne(UUID.fromString(entityId));
                linkedTicket = hotelConnectionUpdate.getStTicket();
            }

            Ticket ticket = new Ticket();
            ticket.setState(ETicketState.TS_NEW);
            ticket.setId(UUID.randomUUID());
            ticket = ticketRepository.saveAndFlush(ticket);
            Workflow workflowForTicket = Workflow.createWorkflowForEntity(ticket);
            workflowForTicket = workflowRepository.saveAndFlush(workflowForTicket);
            TCreateIssue.Builder createIssueBuilder = TCreateIssue.newBuilder()
                    .setEntityType(entityType)
                    .setWorkflowId(workflowId)
                    .setTicketTitle(String.format(properties.getWorkflowCrashedTicketTitle(), entityType, workflowId))
                    .setTicketDescription(templateService.getWorkflowCrashedTicketDescription(entityType, entityId,
                            workflowId, linkedTicket, failedEvent, exception));
            context.scheduleExternalEvent(workflowForTicket.getId(), createIssueBuilder.build());
        } catch (Exception e) {
            // there is only one instance of this workflow and it should never crash
            log.error("Failed to create a ticket for the failed workflow {}", workflowId, e);
            meters.incrementCounter(ERROR_COUNTER_TAG);
        }
    }
}
