CREATE OR REPLACE FUNCTION discard_legal_details_updates_and_restart_publishing_process(hotelConnectionId uuid) RETURNS varchar AS
$$
    /*
     Данная функция предназачена для перезапуска публикации отеля, у которого был указан неверный юр. адрес.

     В таком случае нужно удостовериться, что адрес исправлен у партнера, что hotelConnection был в состоянии New, а
     LegalDetails - в состоянии Published

     Данная функция отменяет все апдейты для LegalDetails, хоть там и могут быть какие-то другие апдейты, они еще раз
     появятся, когда отель опубликуется.
     */
DECLARE
    hotelConnectionState         int;
    previousHotelConnectionState int;
    hotelConnectionWorkflowId    uuid;
    legalDetailsId               uuid;
    legalDetailsState            int;
    previousLegalDetailsState    int;

BEGIN
    SELECT state INTO STRICT hotelConnectionState FROM hotel_connections WHERE id = hotelConnectionId;
    SELECT workflow_id INTO STRICT hotelConnectionWorkflowId FROM hotel_connections WHERE id = hotelConnectionId;
    SELECT legal_details_id INTO STRICT legalDetailsId FROM hotel_connections WHERE id = hotelConnectionId;
    SELECT state INTO STRICT legalDetailsState FROM legal_details WHERE id = legalDetailsId;

    SELECT state
    INTO STRICT previousHotelConnectionState
    FROM hotel_connections_aud
    WHERE id = hotelConnectionId
    ORDER BY rev DESC
    OFFSET 1 LIMIT 1;

    SELECT state
    INTO STRICT previousLegalDetailsState
    FROM legal_details_aud
    WHERE id = legalDetailsId
    ORDER BY rev DESC
    OFFSET 1 LIMIT 1;

    RAISE NOTICE 'HotelConnectionState = %', hotelConnectionState;
    RAISE NOTICE 'legalDetailsState = %', legalDetailsState;
    RAISE NOTICE 'previousHotelConnectionState = %', previousHotelConnectionState;
    RAISE NOTICE 'previousLegalDetailsState = %', previousLegalDetailsState;

    assert hotelConnectionState = 6; -- manual verification
    assert legalDetailsState = 6; -- manual verification
    assert previousHotelConnectionState = 1; -- "new" state
    assert previousLegalDetailsState = 3; -- "Published" state

    -- skipping all updates - do not apply them to legal details
    UPDATE legal_details_updates
    SET state = 'skipped'
    WHERE legal_details_id = legalDetailsId;

    -- set state of Hotel Connection to "Publishing"
    UPDATE hotel_connections
    SET state = 2
    WHERE id = hotelConnectionId;

    -- set state of LegalDetails back to "Registered"
    UPDATE legal_details
    SET state = 3
    WHERE id = legalDetailsId;

    -- restart process of hotel publishing
    INSERT INTO workflow_events(id, workflow_id, created_at, class_name, data, state)
    VALUES (nextval('workflow_events_id_seq'),
            hotelConnectionWorkflowId,
            now(),
            'ru.yandex.travel.hotels.administrator.workflow.proto.TPublishingStart',
            decode('', 'base64'),
            1);

    RETURN 'Апдейт Юр. лица отменен, процесс подключения для отеля перезапущен';
END;
$$ LANGUAGE plpgsql;
