import argparse
import csv
import os
import shutil
import uuid

SUPERVISOR = 'b46b791e-f72e-4650-bb34-1df70360c102'


def map_vat_type(vat):
    if vat == 'vat_none':
        return 0
    elif vat == 'vat_20_120':
        return 7
    elif vat == 'vat_10_110':
        return 5
    else:
        raise AttributeError()


def map_partner(partner):
    if partner == 'travelline':
        return 13
    elif partner == 'bnovo':
        return 45
    else:
        raise AttributeError()


def end_date_to_string(end_date):
    if end_date:
        return "'{}'".format(end_date)
    else:
        return 'null'


def generate_insert_script(created_at):
    folder_name = created_at.replace('-', '_').replace(' ', '_').replace(':', '_')
    if not os.path.exists(folder_name):
        os.makedirs(folder_name)

    with open("hotel_data.csv", mode='r') as csv_file:
        csv_reader = csv.DictReader(csv_file)
        hotels = {}
        commissions = []
        for row in csv_reader:
            if not hotels.get(row['hotel_id']):
                hotel = {
                    'id': str(uuid.uuid4()),
                    'wf_id': str(uuid.uuid4()),
                    'hotel_id': row['hotel_id'],
                    'legal_details_id': row['legal_details_id'],
                    'partner': row['partner'],
                    'permalink': row['permalink'],
                    'name': row['name'],
                    'vat_type': row['vat_type'],
                    'accountant_email': row['accountant_email']
                }
                hotels[row['hotel_id']] = hotel

            commission = {
                'hotel_id': row['hotel_id'],
                'confirmed_rate': row['order_confirmed_rate'],
                'refunded_rate': row['order_refunded_rate'],
                'start_date': row['agreement_start_date'],
                'enabled': row['enabled'],
                'priority': row['priority']
            }
            if row['agreement_end_date']:
                commission['end_date'] = row['agreement_end_date']

            commissions.append(commission)

    with open("{}/insert_hotels.sql".format(folder_name), 'w') as script_file:
        script_file.write("begin transaction;\n\n")
        script_file.write("-- Workflows for hotel_connections\n")
        for hotel_id in hotels:
            script_file.write("""INSERT INTO workflows (id, entity_id, entity_type, created_at, updated_at, next_reservation_refresh_at, version, sleep_till, supervisor_id, state, workflow_version)
VALUES ('{}', '{}', 'hotel_connection', '{}', '{}', null, 0, null, '{}', 1, 0);

""".format(hotels[hotel_id]['wf_id'], hotels[hotel_id]['id'], created_at, created_at, SUPERVISOR))

        script_file.writelines("-- Hotel_connections:\n")
        for hotel_id in hotels:
            hotel = hotels[hotel_id]
            script_file.write("""INSERT INTO hotel_connections (id, hotel_code, partner_id, hotel_name, permalink, st_ticket, ready_to_publish,
             legal_details_id, ready_for_orchestrator, ready_for_offercache, clustering_verified,
             legal_details_registered, workflow_id, state, created_at, updated_at, last_transition_at,
             accountant_email, geo_search_called, vat_type)
VALUES ('{}', '{}', {}, '{}', {}, null, true, '{}', true, true, true, true, '{}', 3, '{}', '{}', '{}', '{}', true, {});

""".format(hotel['id'], hotel['hotel_id'], map_partner(hotel['partner']), hotel['name'], hotel['permalink'],
           hotel['legal_details_id'], hotel['wf_id'], created_at, created_at, created_at, hotel['accountant_email'],
           map_vat_type(hotel['vat_type'])))

        script_file.write("-- Commissions:\n")
        for commission in commissions:
            hotel = hotels[commission['hotel_id']]
            script_file.write(
                """INSERT INTO commissions (id, hotel_connection_id, order_confirmed_rate, order_refunded_rate, agreement_start_date, agreement_end_date, enabled, priority)
VALUES (nextval('hotel_agreements_id_seq'), '{}', {}, {}, '{}', {}, {}, {});

""".format(hotel['id'], commission['confirmed_rate'], commission['refunded_rate'],
           commission['start_date'], end_date_to_string(commission.get('end_date')), commission['enabled'],
           commission['priority']))

        script_file.write("commit;\n")

    shutil.copyfile("hotel_data.csv", "{}/hotel_data.csv".format(folder_name))


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--created-at",
                        required=True,
                        help="Creation timestamp of entities, for example: '2020-04-06 15:00:00'")
    args = parser.parse_args()
    generate_insert_script(args.created_at)


if __name__ == '__main__':
    main()
